﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/arp/arp_Resource.h>
#include <nn/arp/arp_Configs.h>

#include <new>

#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_HipcSimpleAllInOneServerManager.h>

namespace
{

typedef std::aligned_storage<sizeof(nn::arp::DefaultServiceResource), NN_ALIGNOF(nn::arp::DefaultServiceResource)>::type ResourceStorageType;
ResourceStorageType g_ResourceStorage;

nn::Result StartServer(nn::arp::DefaultServiceResource& resource) NN_NOEXCEPT;

// --------------------------------------------------------------------------------------------

const size_t ServerThreadStackSize = 4 * 4096;
NN_ALIGNAS(4096) char g_ServerThreadStack[ServerThreadStackSize];
nn::os::ThreadType g_ServerThread;

const int WriterSessionCountMax = 8;
const int ReaderSessionCountMax = 16;
const int SessionCountMax = WriterSessionCountMax + ReaderSessionCountMax;
const int PortCountMax = 2;
struct ServerOption
{
    static const bool CanDeferInvokeRequest = false;
    static const int ObjectInSubDomainCountMax = 24 * 8;
    static const size_t PointerTransferBufferSize = 4096;
    static const int SubDomainCountMax = 24;
};

class Server
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, ServerOption>
{
} g_Server;

void ServerThreadFunction(void*) NN_NOEXCEPT
{
    g_Server.LoopAuto();
}

nn::Result StartServer(nn::arp::DefaultServiceResource& resource) NN_NOEXCEPT
{
    g_Server.RegisterObjectForPort(resource.GetWriterPtr(), WriterSessionCountMax, nn::arp::PortNameForArpWriter);
    g_Server.RegisterObjectForPort(resource.GetReaderPtr(), ReaderSessionCountMax, nn::arp::PortNameForArpReader);

    NN_RESULT_DO(nn::os::CreateThread(
        &g_ServerThread,
        ServerThreadFunction, nullptr,
        g_ServerThreadStack, sizeof(g_ServerThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(arp, IpcServer)));
    nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(arp, IpcServer));

    g_Server.Start();
    nn::os::StartThread(&g_ServerThread);
    NN_RESULT_SUCCESS;
}

} // ~namespace <anonymous>

void EntryArp()
{
    // サービスの初期化 -------------------------------------------------------------------------------
    auto pSrc = new(&g_ResourceStorage) nn::arp::DefaultServiceResource;

    // もともとシャットダウンは考慮されていなかったので今は単にコメントアウト
    // NN_UTIL_SCOPE_EXIT
    // {
    //     pSrc->~ServiceResource();
    // };

    // サーバーの開始
    NN_ABORT_UNLESS_RESULT_SUCCESS(StartServer(*pSrc));

    NN_LOG("[arp process] running\n");
}
