﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>

#include "creport_ModuleInfo.h"

namespace nn { namespace creport {

    bool ModuleManager::FindCodeRegionStart(uintptr_t* startAddress, uintptr_t hint) NN_NOEXCEPT
    {
        svc::MemoryInfo block;
        svc::PageInfo   page;

        // 開始アドレスが実行可能領域でなければ失敗
        auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, hint);
        if (result.IsFailure() || block.permission != svc::MemoryPermission_ReadExecute)
        {
            return false;
        }

        // Free 領域が見つかるまで前向きに辿る
        uintptr_t ptr = hint;
        do {
            auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, ptr);
            if (result.IsFailure())
            {
                return false;
            }

            if (block.state == svc::MemoryState_Free)
            {
                // Free 領域に当たったら直後の領域（=コード領域の先頭）を返す
                *startAddress = block.baseAddress + block.size;
                return true;
            }

            ptr = block.baseAddress - 4;

        } while (block.baseAddress > 0);

        // メモリの一番上まで辿ったが見つからなかった場合
        return false;
    }

    void ModuleManager::GetModuleName(char* pOutModuleName, size_t moduleNameSize, uintptr_t roStart) NN_NOEXCEPT
    {
        const size_t ModuleNamePathSize = 512;
        const size_t ModulePathOffset = 8;
        char         modulePathBuffer[ModuleNamePathSize];

        auto result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(modulePathBuffer), m_DebugHandle, roStart + ModulePathOffset, sizeof(modulePathBuffer));
        if (result.IsSuccess())
        {
            // 最後の '/' より後だけ残す
            int pos = strnlen(modulePathBuffer, sizeof(modulePathBuffer));
            for (; pos >= 0; --pos)
            {
                if (modulePathBuffer[pos] == '/' || modulePathBuffer[pos] == '\\')
                {
                    break;
                }
            }

            size_t size = std::min(moduleNameSize, ModuleNamePathSize - pos - 1);
            std::memset(pOutModuleName, 0, moduleNameSize);
            std::strncpy(pOutModuleName, modulePathBuffer + pos + 1, size);
            pOutModuleName[moduleNameSize - 1] = '\0';
        }
    }

    void ModuleManager::GetModuleId(Bit8* pOutModuleId, size_t moduleIdSize, uintptr_t roStart) NN_NOEXCEPT
    {
        svc::MemoryInfo block;
        svc::PageInfo   page;

        // 出力を初期化
        // 何らかの原因で ModuleId を取得できなかった場合は、この初期値が返る
        memset(pOutModuleId, 0, moduleIdSize);

        auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, roStart);
        if (result.IsFailure())
        {
            return;
        }

        if (block.permission != svc::MemoryPermission_Read)
        {
            return;
        }

        //
        // コード領域直後の RO セクションの末尾 8192 バイトを探索して ModuleId を取得する
        //

        static const Bit8 ModuleIdSignature[4] = {0x47, 0x4E, 0x55, 0x00}; // ro 末尾の GNU. の後ろに ModuleId が来る

        static Bit8 s_Buffer[os::MemoryPageSize * 2];
        static_assert(os::MemoryPageSize == 4096, "");

        uintptr_t bufferAddress = block.baseAddress;
        size_t    bufferSize    = os::MemoryPageSize;

        if (block.size >= os::MemoryPageSize * 2)
        {
            bufferSize    = os::MemoryPageSize * 2;
            bufferAddress = block.baseAddress + block.size - bufferSize;
        }

        NN_SDK_LOG("[creport] module id region: 0x%llx-0x%llx\n", bufferAddress, bufferAddress + bufferSize);

        result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(s_Buffer), m_DebugHandle, bufferAddress, bufferSize);
        if (result.IsFailure())
        {
            return;
        }

        // 前方探索
        for (int offset = bufferSize - sizeof(ModuleIdSignature) - MaxModuleIdLength; offset >= 0; offset--)
        {
            if (std::memcmp(s_Buffer + offset, ModuleIdSignature, sizeof(ModuleIdSignature)) == 0)
            {
                // 見つかった
                int copySize = (moduleIdSize < MaxModuleIdLength) ? moduleIdSize : MaxModuleIdLength;
                memcpy(pOutModuleId, s_Buffer + offset + sizeof(ModuleIdSignature), copySize);
                return;
            }
        }
    }

    void ModuleManager::CollectModuleInformation(svc::Handle handle, uintptr_t regPc, uintptr_t regLr) NN_NOEXCEPT
    {
        svc::MemoryInfo block;
        svc::PageInfo   page;
        uintptr_t       ptr;

        m_DebugHandle = handle;

        // PC を基準にコード領域を探索する
        if (FindCodeRegionStart(&ptr, regPc) == false)
        {
            // 変なアドレスに飛んで止まった場合 PC が狂っているかもしれない
            // その場合は LR を基準にコード領域を探索する
            if (FindCodeRegionStart(&ptr, regLr) == false)
            {
                // LR もダメなら諦める
                return;
            }
        }

        do {
            auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, ptr);
            if (result.IsFailure())
            {
                break;
            }

            // コード領域を保存
            if (block.permission == svc::MemoryPermission_ReadExecute)
            {
                ModuleInfo& mi = m_Module[m_ModuleCount];

                NN_SDK_LOG("[creport] code region: 0x%llx-0x%llx\n", block.baseAddress, block.baseAddress + block.size);

                mi.codeBase = block.baseAddress;
                mi.codeEnd  = block.baseAddress + block.size;
                std::memset(mi.moduleName, 0, sizeof(mi.moduleName));
                GetModuleName(mi.moduleName, sizeof(mi.moduleName), block.baseAddress + block.size);
                GetModuleId(mi.moduleId, sizeof(mi.moduleId), block.baseAddress + block.size);

                ++m_ModuleCount;
                if (m_ModuleCount >= MaxModuleCount)
                {
                    break;
                }
            }

            // Free もしくは Inaccessible 状態のメモリに到達するか、一周してオーバーフローしたら終わり
            if (block.state == svc::MemoryState_Free || block.state == svc::MemoryState_Inaccessible || ptr + block.size <= ptr || block.size == 0)
            {
                break;
            }

            ptr += block.size;

        } while (ptr > 0);
    }

    Bit64 ModuleManager::GetOffset(uintptr_t addr) const NN_NOEXCEPT
    {
        for (auto&& mi : m_Module)
        {
            if (addr >= mi.codeBase && addr < mi.codeEnd)
            {
                return addr - mi.codeBase;
            }
        }
        return addr & 0xFFF;
    }

    void ModuleManager::AppendToBuffer(InfoBuffer* pOutBuffer) NN_NOEXCEPT
    {
        pOutBuffer->Append(&m_ModuleCount, sizeof(m_ModuleCount));
        for (int i = 0; i < m_ModuleCount; ++i)
        {
            pOutBuffer->Append(&m_Module[i], sizeof(m_Module[i]));
        }
    }

}}
