﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/btm/btm_Result.h>
#include <nn/btm/btm_Types.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include "btm_Utility.h"
#include "btm_Devices.h"
#include "btm_BleScanFilters.h"

namespace nn { namespace btm {

//[Todo]事前条件チェック（特にリストのCount）を共通処理として定義しておく


//--------------------------------------------------
// DevicePropertyContainer
//--------------------------------------------------
DevicePropertyContainer::DevicePropertyContainer()
{
    this->Init();
}

DevicePropertyContainer::~DevicePropertyContainer()
{

}

void DevicePropertyContainer::Init()
{
    m_DevicePropertyList.deviceCount = 0;
}

DevicePropertyList DevicePropertyContainer::Get() const
{
    return m_DevicePropertyList;
}

void DevicePropertyContainer::Add(const DeviceProperty& deviceProperty)
{
    if(m_DevicePropertyList.deviceCount < COUNT_OF_DP_LIST)
    {
        m_DevicePropertyList.device[m_DevicePropertyList.deviceCount] = deviceProperty;
        m_DevicePropertyList.deviceCount+=1;
    }
    else
    {
        BTM_LOG("Num of devices overs COUNT_OF_DP_LIST.\n");
    }
}

bool DevicePropertyContainer::SearchBdAddress(BdAddress* pBdAddress, const char bdName[], uint8_t bdNameSize) const
{
    if(bdNameSize > SIZE_OF_BDNAME)
    {
        return false;
    }

    for(int i=0;i<m_DevicePropertyList.deviceCount;i++)
    {
        if(0 == memcmp(&bdName[0], &m_DevicePropertyList.device[i].bdName.name[0] ,bdNameSize))
        {
            *pBdAddress = m_DevicePropertyList.device[i].bdAddress;
            return true;
        }
    }
    return false;
}





//--------------------------------------------------
// DeviceConditionContainer
//--------------------------------------------------
DeviceConditionContainer::DeviceConditionContainer()
{
    this->Init();
}

DeviceConditionContainer::~DeviceConditionContainer()
{

}

void DeviceConditionContainer::Init()
{
    m_DeviceConditionList.deviceCount = 0;
    m_DeviceConditionList.bluetoothMode = BluetoothMode_Auto;
    m_DeviceConditionList.wlanMode = WlanMode_None;
    m_DeviceConditionList.isSlotSavingForPairing = false;
    m_DeviceConditionList.isSlotSaving = false;
    m_DeviceConditionList.isLargeSlotRequiredForBle = false;
    m_DeviceConditionList.deviceCountCapacity = 8;
}

DeviceConditionList DeviceConditionContainer::Get() const
{
    return m_DeviceConditionList;
}

const DeviceConditionList* DeviceConditionContainer::GetPtr() const
{
    return &m_DeviceConditionList;
}

bool DeviceConditionContainer::Add(const DeviceInfo& deviceInfo)
{
    uint8_t index;
    bool isThereDeviceCondition;
    isThereDeviceCondition = SearchDeviceConditionIndex(&index, deviceInfo.bdAddress);

    DeviceCondition* pDeviceCondition;
    if(isThereDeviceCondition)
    {
        pDeviceCondition = &m_DeviceConditionList.device[index];
        //下層のバグ洗い出しのため、あえてABORTさせる
        BTM_LOG("Multiple connection were reported from a lower layer.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorMultipleConnection());
    }
    else
    {
        if(m_DeviceConditionList.deviceCount == COUNT_OF_DC_LIST)
        {
            return false;
        }
        pDeviceCondition = &m_DeviceConditionList.device[m_DeviceConditionList.deviceCount];
        m_DeviceConditionList.deviceCount+=1;
    }

    //DeviceConditionを作成
    pDeviceCondition->bdAddress = deviceInfo.bdAddress;
    pDeviceCondition->hidDeviceCondition.isBurstMode = false;
    pDeviceCondition->hidDeviceCondition.zeroRetransmissionList.enabledReportIdCount = 0;
    pDeviceCondition->profile = Profile_Hid;
    pDeviceCondition->hidDeviceCondition.sniffMode = SniffMode_Active;
    pDeviceCondition->hidDeviceCondition.slotMode = SlotMode_Active;
    pDeviceCondition->hidDeviceCondition.isLargeSlotRequired = false;
    //BdName, VID, PIDの設定
    memcpy(&pDeviceCondition->bdName.name[0], &deviceInfo.bdName.name[0], SIZE_OF_BDNAME);
    pDeviceCondition->hidDeviceCondition.vid = deviceInfo.hidDeviceInfo.vid;
    pDeviceCondition->hidDeviceCondition.pid = deviceInfo.hidDeviceInfo.pid;

    return true;
}

bool DeviceConditionContainer::Remove(const BdAddress& bdAddress)
{
    uint8_t index;
    bool isThereDeviceCondition;
    isThereDeviceCondition = SearchDeviceConditionIndex(&index, bdAddress);

    if(isThereDeviceCondition)
    {
        m_DeviceConditionList.deviceCount-=1;
        for(int i=index;i<COUNT_OF_DC_LIST - 1;i++)
        {
            m_DeviceConditionList.device[i] = m_DeviceConditionList.device[i + 1];
        }
    }
    if(isThereDeviceCondition)
    {
        return true;
    }
    else
    {
        return false;
    }
}

void DeviceConditionContainer::Update(const BdAddress& bdAddress, const ZeroRetransmissionList& zeroRetransmissionList)
{
    for(int i=0;i<m_DeviceConditionList.deviceCount;i++)
    {
        bool isSame = IsSameBdAddress(&m_DeviceConditionList.device[i].bdAddress, &bdAddress);
        if(isSame)
        {
            //[Todo]下層のイベントに最大値を入れてもらい、きちんと反映させる
            m_DeviceConditionList.device[i].hidDeviceCondition.zeroRetransmissionList.enabledReportIdCount = 0;
        }
    }
}

void DeviceConditionContainer::Update(const DeviceConditionList& deviceConditionList)
{
    m_DeviceConditionList.bluetoothMode = deviceConditionList.bluetoothMode;
    m_DeviceConditionList.wlanMode = deviceConditionList.wlanMode;
    m_DeviceConditionList.isSlotSavingForPairing = deviceConditionList.isSlotSavingForPairing;
    m_DeviceConditionList.isSlotSaving = deviceConditionList.isSlotSaving;
    m_DeviceConditionList.isLargeSlotRequiredForBle = deviceConditionList.isLargeSlotRequiredForBle;
    // イベントドリブンに変更される値なので、ここでは上書きしない
//    m_DeviceConditionList.isLargeSlotRequiredForBle = deviceConditionList.isLargeSlotRequiredForBle;
    m_DeviceConditionList.deviceCountCapacity = deviceConditionList.deviceCountCapacity;
    for(int i = 0; i < m_DeviceConditionList.deviceCount; i++)
    {
        for(int j = 0; j < deviceConditionList.deviceCount; j++)
        {
            if(IsSameBdAddress(&m_DeviceConditionList.device[i].bdAddress,
                    &deviceConditionList.device[j].bdAddress))
            {
                m_DeviceConditionList.device[i] = deviceConditionList.device[j];
            }
        }
    }
}


DeviceConditionContainer::DiffIndex DeviceConditionContainer::SearchDifferentSlotModeIndex(const DeviceConditionList& deviceConditionList) const
{
    DiffIndex diffIndex;
    diffIndex.count = 0;

    for(int i=0;i<deviceConditionList.deviceCount;i++)
    {
        for(int j=0;j<m_DeviceConditionList.deviceCount;j++)
        {
            if(IsSameBdAddress(&deviceConditionList.device[i].bdAddress, &m_DeviceConditionList.device[j].bdAddress))
            {
                if(deviceConditionList.device[i].hidDeviceCondition.slotMode == m_DeviceConditionList.device[j].hidDeviceCondition.slotMode)
                {
                }
                else
                {
                    //スロットモードに差があるので計上
                    diffIndex.index[diffIndex.count] = i;
                    diffIndex.count++;
                }
            }
        }
    }
    return diffIndex;
}

bool DeviceConditionContainer::SearchDeviceCondition(DeviceCondition* pDeviceCondition, const BdAddress& bdAddress) const
{
    uint8_t index;
    bool isThereDeviceCondition;
    isThereDeviceCondition = SearchDeviceConditionIndex(&index, bdAddress);

    if(isThereDeviceCondition)
    {
        *pDeviceCondition = m_DeviceConditionList.device[index];
        return true;
    }
    else
    {
        return false;
    }
}

void DeviceConditionContainer::SetLargeSlotModeRequiredForBle(bool required)
{
    m_DeviceConditionList.isLargeSlotRequiredForBle = required;
}

bool DeviceConditionContainer::SearchDeviceConditionIndex(uint8_t* pIndex, const BdAddress& bdAddress) const
{
    uint8_t deviceCount = m_DeviceConditionList.deviceCount;
    for(int i=0;i<deviceCount;i++)
    {
        if(IsSameBdAddress(&m_DeviceConditionList.device[i].bdAddress, &bdAddress))
        {
            *pIndex = i;
            return true;
        }
    }
    return false;
}

void DeviceConditionContainer::Update(uint8_t deviceCountCapacity)
{
    if(deviceCountCapacity > COUNT_OF_DC_LIST)
    {
        BTM_LOG("Invalid deviceCountCapacity\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }
    m_DeviceConditionList.deviceCountCapacity = deviceCountCapacity;
}



//--------------------------------------------------
// DeviceInfoContainer
//--------------------------------------------------
DeviceInfoContainer::DeviceInfoContainer()
{

}

DeviceInfoContainer::~DeviceInfoContainer()
{

}

void DeviceInfoContainer::InitByNvm()
{
    const uint8_t count = nn::settings::system::GetBluetoothDevicesSettings(&m_DeviceSettingList[0], nn::settings::system::BluetoothDevicesSettingsCountMax);
    DiDsUtility::ConvertList(&m_DeviceInfoList, &m_DeviceSettingList[0], count);
}

void DeviceInfoContainer::WriteToNvm()
{
    InitDeviceSettingList();
    nn::settings::system::SetBluetoothDevicesSettings(&m_DeviceSettingList[0], m_DeviceInfoList.deviceCount);
}

DeviceInfoList DeviceInfoContainer::Get() const
{
    return m_DeviceInfoList;
}

DeviceInfo DeviceInfoContainer::Get(uint8_t index) const
{
    return m_DeviceInfoList.device[index];
}

const nn::settings::system::BluetoothDevicesSettings* DeviceInfoContainer::GetPtr()
{
    InitDeviceSettingList();
    return &m_DeviceSettingList[0];
}

void DeviceInfoContainer::Add(const DeviceInfo& deviceInfo)
{
    bool isSameBdAddress;
    uint8_t index;
    isSameBdAddress = DiDsUtility::SearchIndex(&index, m_DeviceInfoList, deviceInfo.bdAddress);
    if(isSameBdAddress)
    {
        //同様のBdAddressがあったら、上書き
        m_DeviceInfoList.device[index] = deviceInfo;
        m_DeviceInfoList.device[index].bdName.name[SIZE_OF_BDNAME - 1] = '\0';//NULL終端
    }
    else if(m_DeviceInfoList.deviceCount == COUNT_OF_DI_LIST)
    {
        //容量いっぱいの時は、最前列を消し、前詰めのうえ、最後尾に追記
        for(int i=0;i<COUNT_OF_DI_LIST - 1;i++)
        {
            m_DeviceInfoList.device[i] = m_DeviceInfoList.device[i + 1];
        }
        m_DeviceInfoList.device[COUNT_OF_DI_LIST - 1] = deviceInfo;
        m_DeviceInfoList.device[COUNT_OF_DI_LIST - 1].bdName.name[SIZE_OF_BDNAME - 1] = '\0';//NULL終端
    }
    else
    {
        //容量いっぱいでないときは、新しいindexに追記
        m_DeviceInfoList.device[m_DeviceInfoList.deviceCount] = deviceInfo;
        m_DeviceInfoList.device[m_DeviceInfoList.deviceCount].bdName.name[SIZE_OF_BDNAME - 1] = '\0';//NULL終端
        m_DeviceInfoList.deviceCount++;
    }
}

void DeviceInfoContainer::Add(const nn::settings::system::BluetoothDevicesSettings& setting)
{
    DeviceInfo deviceInfo;
    DiDsUtility::Convert(&deviceInfo, setting);
    Add(deviceInfo);
}

void DeviceInfoContainer::Remove(const BdAddress& bdAddress)
{
    bool isSameBdAddress;
    uint8_t index;

    isSameBdAddress = DiDsUtility::SearchIndex(&index, m_DeviceInfoList, bdAddress);
    if(isSameBdAddress)
    {
        //同様のBdAddressがあったら、削除して前詰め
        for(int i=index;i<m_DeviceInfoList.deviceCount - 1;i++)
        {
            m_DeviceInfoList.device[i] = m_DeviceInfoList.device[i + 1];
        }
        //最後尾が削除対象だった場合は上記ループに入らないが、indexが1減るので削除扱いになる
        m_DeviceInfoList.deviceCount--;
    }
}

void DeviceInfoContainer::Remove()
{
    m_DeviceInfoList.deviceCount = 0;
}

bool DeviceInfoContainer::SearchDeviceInfo(DeviceInfo* pDeviceInfo, const BdAddress& bdAddress) const
{
    uint8_t index;
    if(DiDsUtility::SearchIndex(&index, m_DeviceInfoList, bdAddress))
    {
        if (pDeviceInfo != nullptr) {
            *pDeviceInfo = m_DeviceInfoList.device[index];
        }
        return true;
    }
    else
    {
        return false;
    }
}

uint8_t DeviceInfoContainer::GetCount() const
{
    return m_DeviceInfoList.deviceCount;
}

void DeviceInfoContainer::InitDeviceSettingList()
{
    DiDsUtility::ConvertList(&m_DeviceSettingList[0], m_DeviceInfoList);
}

void DeviceInfoContainer::MoveToTail(const BdAddress& bdAddress)
{
    bool isSameBdAddress;
    uint8_t index;
    DeviceInfo targetDeviceInfo;

    isSameBdAddress = DiDsUtility::SearchIndex(&index, m_DeviceInfoList, bdAddress);
    if(isSameBdAddress)
    {
        targetDeviceInfo = m_DeviceInfoList.device[index];
        Remove(bdAddress);
        Add(targetDeviceInfo);
    }
}



//--------------------------------------------------
// DiDsUtility
//--------------------------------------------------
void DiDsUtility::Reset(nn::settings::system::BluetoothDevicesSettings* pSetting)
{
    //[Todo]固定値の定義
    memset(pSetting, 0, sizeof(nn::settings::system::BluetoothDevicesSettings));
    pSetting->link_key_present = 1;
    pSetting->trusted_services = 1048576;
    pSetting->key_type = 0x04;
    //io_cap
    //features
    pSetting->attribute_mask = 0xFF;
    pSetting->sub_class = 0x08;
    //peerinfo
    //feature_mask
    pSetting->brr_size = 0;
    pSetting->brr[8] = 0;
    //descripter系
    //enable mip
    //handle
    //no peer info
}

bool DiDsUtility::SearchIndex(uint8_t* pIndex, const DeviceInfoList& deviceInfoList, const BdAddress& bdAddress)
{
    for(int i=0;i<deviceInfoList.deviceCount;i++)
    {
        if(IsSameBdAddress(&bdAddress, &deviceInfoList.device[i].bdAddress))
        {
            *pIndex = i;
            return true;
        }
    }
    return false;
}

void DiDsUtility::Convert(nn::settings::system::BluetoothDevicesSettings* pSetting, const DeviceInfo& deviceInfo)
{
    Reset(pSetting);
    memcpy(&pSetting->bd_addr[0], &deviceInfo.bdAddress.address[0], SIZE_OF_BDADDRESS);
    memcpy(&pSetting->device_name[0], &deviceInfo.bdName.name[0], SIZE_OF_BDNAME);
    pSetting->device_name[SIZE_OF_BDNAME - 1] = '\0';//NULL終端。DeviceInfoListのBdNameがNULL終端保障されているため、本来ここでは不要
    memcpy(&pSetting->class_of_device[0], &deviceInfo.classOfDevice.cod[0], SIZE_OF_COD);
    memcpy(&pSetting->link_key[0], &deviceInfo.linkKey.key[0], SIZE_OF_LINKKEY);
    pSetting->vid = deviceInfo.hidDeviceInfo.vid;
    pSetting->pid = deviceInfo.hidDeviceInfo.pid;
}

void DiDsUtility::Convert(DeviceInfo* pDeviceInfo, const nn::settings::system::BluetoothDevicesSettings& setting)
{
    memcpy(&pDeviceInfo->bdAddress.address[0], &setting.bd_addr[0], SIZE_OF_BDADDRESS);
    memcpy(&pDeviceInfo->bdName.name[0],       &setting.device_name[0], SIZE_OF_BDNAME);
    pDeviceInfo->bdName.name[SIZE_OF_BDNAME - 1] = '\0';//NULL終端
    memcpy(&pDeviceInfo->classOfDevice.cod[0], &setting.class_of_device[0], SIZE_OF_COD);
    memcpy(&pDeviceInfo->linkKey.key[0],       &setting.link_key[0], SIZE_OF_LINKKEY);
    //[Todo]プロファイルの判別方法を決める
    pDeviceInfo->profile = Profile_Hid;
    pDeviceInfo->hidDeviceInfo.vid = setting.vid;
    pDeviceInfo->hidDeviceInfo.pid = setting.pid;
}

void DiDsUtility::ConvertList(nn::settings::system::BluetoothDevicesSettings buffer[], const DeviceInfoList& deviceInfoList)
{
    //カウント情報はBluetoothDevicesSettingsに残らないので注意
    for(int i=0;i<deviceInfoList.deviceCount;i++)
    {
        Convert(&buffer[i], deviceInfoList.device[i]);
    }
}

void DiDsUtility::ConvertList(DeviceInfoList* pDeviceInfoList, const nn::settings::system::BluetoothDevicesSettings buffer[], uint8_t bufferCount)
{
    pDeviceInfoList->deviceCount = bufferCount;
    for(int i=0;i<bufferCount;i++)
    {
        Convert(&pDeviceInfoList->device[i], buffer[i]);
    }
}




//--------------------------------------------------
// SystemEventContainer
//--------------------------------------------------
//[Todo]Device情報ではないので、同種のクラスが増えてきたらファイル分割する
SystemEventContainer::SystemEventContainer()
{
    for(int i=0;i<COUNT_OF_EVENT_TYPE;i++)
    {
        for(int j=0;j<COUNT_OF_EVENT_UNIT;j++)
        {
            m_EventList[i].eventUnit[j].isInUse = false;
        }
    }
}

nn::Result SystemEventContainer::Acquire(nn::os::NativeHandle* pHandle, EventType eventType)
{
    uint8_t tempId;
    return Acquire(pHandle, eventType, &tempId);
}

nn::Result SystemEventContainer::Acquire(nn::os::NativeHandle* pHandle, EventType eventType, uint8_t* pId)
{
    Validate(eventType);

    for(int i=0;i<COUNT_OF_EVENT_UNIT;i++)
    {
        EventUnit& eventUnit = m_EventList[eventType].eventUnit[i];
        if(!eventUnit.isInUse)
        {
            auto result = nn::os::CreateSystemEvent(&eventUnit.event, nn::os::EventClearMode_AutoClear, true);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            auto handle = nn::os::GetReadableHandleOfSystemEvent(&eventUnit.event);
            *pHandle = handle;
            eventUnit.isInUse = true;
            *pId = i;
            return ResultSuccess();
        }
    }

    NN_DETAIL_BTM_FATAL("There are no system event for register.");
    return nn::btm::ResultInternalErrorNoResource();
}

void SystemEventContainer::Discard(EventType eventType, uint8_t id)
{
    BTM_LOG("DiscardEvent. type = enum(%d), id = %d\n", eventType, id);
    if(!m_EventList[eventType].eventUnit[id].isInUse)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }

    DestroySystemEvent(&m_EventList[eventType].eventUnit[id].event);
    m_EventList[eventType].eventUnit[id].isInUse = false;
}

void SystemEventContainer::Signal(SystemEventContainer::EventType eventType)
{
    Validate(eventType);
    uint8_t count = 0;

    for(int i=0;i<COUNT_OF_EVENT_UNIT;i++)
    {
        if(m_EventList[eventType].eventUnit[i].isInUse)
        {
            SignalSystemEvent(&m_EventList[eventType].eventUnit[i].event);
            count++;
        }
    }

    BTM_LOG("SystemEventSignaled EventType=enum(%d), Count=%d\n", eventType, count);
}

void SystemEventContainer::Validate(EventType eventType) const
{
    if(eventType < 0 || eventType >= COUNT_OF_EVENT_TYPE)
    {
        BTM_LOG("Invalid event type.");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }
}




//--------------------------------------------------
// HostConditionContainer
//--------------------------------------------------
HostConditionContainer::HostConditionContainer()
{
    m_IsDiscoverying = false;
    m_IsAutoPairing = false;
    m_IsRadio = true;
    m_IsConnectionRefusal = false;
    m_HostDeviceProperty.bdName.name[0] = '\0';
    m_IsLlr = false;
    for(int i=0;i<HostConditionContainer::COUNT_OF_SMH;i++)
    {
        m_Smh[i].time =0;
        m_Smh[i].sniffMode = nn::btm::SniffMode_5ms;
    }
    m_PseudoConnectionList.count = 0;
}

void HostConditionContainer::SetDiscoverying(bool isDiscoverying)
{
    m_IsDiscoverying = isDiscoverying;
}

void HostConditionContainer::SetAutoPairing(bool isEnabled)
{
    m_IsAutoPairing = isEnabled;
}

void HostConditionContainer::SetRadio(bool isEnabled)
{
    m_IsRadio = isEnabled;
    nn::settings::system::SetBluetoothEnabled(isEnabled);
}

void HostConditionContainer::SetConnectionRefusal(bool isEnabled)
{
    m_IsConnectionRefusal = isEnabled;
}

void HostConditionContainer::SetLlrEnabled(const BdAddress& bdAddress)
{
    m_IsLlr = true;
    m_LlrAddress = bdAddress;
}

void HostConditionContainer::SetLlrDisabled()
{
    m_IsLlr = false;
}

bool HostConditionContainer::IsDiscoverying() const
{
    return m_IsDiscoverying;
}

bool HostConditionContainer::IsAutoPairing() const
{
    return m_IsAutoPairing;
}

void HostConditionContainer::ValidatePseudoConnectionList() const
{
    if(m_PseudoConnectionList.count > COUNT_OF_PSEUDO_CONNECTION)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }
}

uint8_t HostConditionContainer::SearchPseudoConnectionIndex(const BdAddress& bdAddress) const
{
    for(int i=0;i<m_PseudoConnectionList.count;i++)
    {
        bool isSame = IsSameBdAddress(&bdAddress, &m_PseudoConnectionList.bdAddress[i]);
        if(isSame)
        {
            return i;
        }
    }
    return 0xFF;
}

void HostConditionContainer::AddPseudoConnection(const BdAddress& bdAddress)
{
    uint8_t index = SearchPseudoConnectionIndex(bdAddress);
    if(index != 0xFF)
    {
        //同じ情報がある場合は上書き（実質無視）
        m_PseudoConnectionList.bdAddress[index] = bdAddress;
    }
    else
    {
        if(m_PseudoConnectionList.count < COUNT_OF_PSEUDO_CONNECTION)
        {
            //空きがある場合は最後尾に追加
            m_PseudoConnectionList.bdAddress[m_PseudoConnectionList.count] = bdAddress;
            m_PseudoConnectionList.count++;
        }
        else if(m_PseudoConnectionList.count == COUNT_OF_PSEUDO_CONNECTION)
        {
            //空きがない場合は、先頭を消して前詰めの上、最後尾に追加
            for(int i=0;i<COUNT_OF_PSEUDO_CONNECTION - 1;i++)
            {
                m_PseudoConnectionList.bdAddress[i] = m_PseudoConnectionList.bdAddress[i + 1];
            }
            m_PseudoConnectionList.bdAddress[COUNT_OF_PSEUDO_CONNECTION - 1] = bdAddress;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
        }
    }

    ValidatePseudoConnectionList();
}

void HostConditionContainer::RemovePseudoConnection(const BdAddress& bdAddress)
{
    uint8_t index = SearchPseudoConnectionIndex(bdAddress);
    if(index != 0xFF)
    {
        //1件減らして前詰め
        for(int i=index;i<m_PseudoConnectionList.count - 1;i++)
        {
            m_PseudoConnectionList.bdAddress[i] = m_PseudoConnectionList.bdAddress[i + 1];
        }
        m_PseudoConnectionList.count--;
    }
    else
    {
        //対象無しのため何もしない
    }

    ValidatePseudoConnectionList();
}

bool HostConditionContainer::IsPseudoConnection(const BdAddress& bdAddress) const
{
    uint8_t index = SearchPseudoConnectionIndex(bdAddress);
    if(index == 0xFF)
    {
        return false;
    }
    else
    {
        return true;
    }
}

HostConditionContainer::PseudoConnectionList HostConditionContainer::GetPseudoConnectionList() const
{
    return m_PseudoConnectionList;
}

bool HostConditionContainer::IsRadio() const
{
    return m_IsRadio;
}

bool HostConditionContainer::IsConnectionRefusal() const
{
    return m_IsConnectionRefusal;
}

bool HostConditionContainer::IsLlr() const
{
    return m_IsLlr;
}

bool HostConditionContainer::IsLlr(const BdAddress& bdAddress) const
{
    if(!m_IsLlr)
    {
        return false;
    }
    for(int i=0;i<SIZE_OF_BDADDRESS;i++)
    {
        if(m_LlrAddress.address[i] != bdAddress.address[i])
        {
            return false;
        }
    }
    return true;
}

const char* HostConditionContainer::GetApdn(uint8_t index) const
{
    if(index >= APDN_COUNT)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }

    return &m_Apdn[index][0];
}

uint8_t HostConditionContainer::GetApdnSize(uint8_t index) const
{
    if(index >= APDN_COUNT)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }

    return m_ApdnSize[index];
}

void HostConditionContainer::SetProperty(HostDeviceProperty& hdp)
{
    m_HostDeviceProperty = hdp;
}

HostDeviceProperty HostConditionContainer::GetProperty() const
{
    return m_HostDeviceProperty;
}


bool HostConditionContainer::IsRapidChange(const uint64_t& previousTimeMs, const uint64_t& newestTimeMs, const uint64_t& thresholdTimeMs)
{
    if(newestTimeMs - previousTimeMs < thresholdTimeMs)
    {
        return true;
    }
    else
    {
        return false;
    }
}

bool HostConditionContainer::IsSlowChange(const uint64_t& previousTimeMs, const uint64_t& newestTimeMs, const uint64_t& thresholdTimeMs)
{
    if(newestTimeMs - previousTimeMs > thresholdTimeMs)
    {
        return true;
    }
    else
    {
        return false;
    }
}

void HostConditionContainer::AddSniffModeHistory(const HostConditionContainer::SniffModeHistory& smh)
{
    //(COUNT_OF_SMH + 1)回の変更が15秒以内に発生しているか確認
    if(m_Smh[HostConditionContainer::COUNT_OF_SMH - 1].time == 0)
    {
        //評価に十分な情報が蓄積されていない
    }
    else
    {
        if(IsRapidChange(m_Smh[HostConditionContainer::COUNT_OF_SMH - 1].time, smh.time, 15000))
        {
            NN_DETAIL_BTM_ERROR("Rapid sniff change was detected. %d times in 15 seconds.\n", COUNT_OF_SMH + 1);
        }
    }

    //15ms　Sniff状態が1分以上継続しているか確認
    if(m_Smh[0].time == 0)
    {
        //過去情報が無い
    }
    else
    {
        if(m_Smh[0].sniffMode == nn::btm::SniffMode_15ms)
        {
            //[Todo]Dynamicでの15msを検出してしまって良いか決める
            if(IsSlowChange(m_Smh[0].time, smh.time, 60000))
            {
                NN_DETAIL_BTM_ERROR("Slow sniff was kept in a long time. 15ms sniff was kept over 60 seconds.\n");
            }
        }
        else
        {
            //SniffModeが15msでは無い
        }
    }

    //目的外の整合性チェックはしない。例：最新の時刻が過去の時刻より古いことはチェックしない
    for(int i=0;i<HostConditionContainer::COUNT_OF_SMH - 1;i++)
    {
        m_Smh[i + 1] = m_Smh[i];
    }
    m_Smh[0] = smh;
}

//--------------------------------------------------
// BleGattClientConditionContainer
//--------------------------------------------------
BleGattClientConditionContainer::BleGattClientConditionContainer()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        Clear(i);
    }

    m_GattClientConditionListInternal.deviceCount               = 0;
    m_GattClientConditionListInternal.pendingConnectionCount    = 0;
    m_GattClientConditionListInternal.deviceCountCapacity       = NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients);
    m_GattClientConditionListInternal.isLargeCeLengthRequired   = false;

    SetConnectionRefusal(false);

    for (auto& history : m_GattClientConnectionHistroyList)
    {
        memset(&history.address, 0x00, sizeof(BdAddress));
        history.connectionHandle    = nn::bluetooth::BleInvalidConnectionHandle;
        history.disconnectionReason = nn::btm::debug::BleDisconnectionReason_Unknown;
    }
}

void BleGattClientConditionContainer::Clear(int index)
{
    NN_SDK_REQUIRES_RANGE(index, 0, NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients));

    auto& gattClient = m_GattClientConditionListInternal.gattClients[index];

    gattClient.handle                                   = nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE;
    gattClient.connectedServer.connectionHandle         = nn::bluetooth::BleInvalidConnectionHandle;
    memset(gattClient.connectedServer.address.address, 0x00, SIZE_OF_BDADDRESS);
    gattClient.connectedServer.connectionOwnerApplet    = nn::applet::AppletResourceUserId::GetInvalidId();
    gattClient.connectedServer.isPendingConnection      = false;

    for (int i = 0; i < nn::bluetooth::GattAttributeCountMaxClient; ++i)
    {
        memset(&gattClient.connectedServer.serviceList[i], 0x00, sizeof(user::GattService));
        memset(&gattClient.connectedServer.charList[i], 0x00, sizeof(user::GattCharacteristic));
        memset(&gattClient.connectedServer.descList[i], 0x00, sizeof(user::GattDescriptor));
    }

    gattClient.connectedServer.serviceNum         = 0;
    gattClient.connectedServer.charNum            = 0;
    gattClient.connectedServer.descNum            = 0;
    gattClient.connectedServer.supportPairing     = false;
    gattClient.connectionInterval                 = DefaultConnectionIntervalMin;
    gattClient.slaveLatency                       = DefaultSlaveLatency;
    gattClient.supervisionTimeout                 = DefaultSupervisionTimeout;
    gattClient.maxCeLength                        = CeLength_Invalid;
    gattClient.mtu                                = nn::bluetooth::BleMtuDefault;
    gattClient.isConfiguringMtu                   = false;

    gattClient.connectionIntervalReqMin           = 0;
    gattClient.connectionIntervalReqMax           = 0;
    gattClient.slaveLatencyReq                    = 0;
    gattClient.supervisionTimeoutReq              = 0;
}

const GattClientConditionList* BleGattClientConditionContainer::GetPtr()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionList.gattClients); ++i)
    {
        auto& gattClient                = m_GattClientConditionList.gattClients[i];
        const auto& gattClientInternal  = m_GattClientConditionListInternal.gattClients[i];

        gattClient.connectedServer.address          = gattClientInternal.connectedServer.address;
        gattClient.connectedServer.connectionHandle = gattClientInternal.connectedServer.connectionHandle;

        gattClient.handle             = gattClientInternal.handle;

        gattClient.connectionInterval = gattClientInternal.connectionInterval;
        gattClient.slaveLatency       = gattClientInternal.slaveLatency;
        gattClient.supervisionTimeout = gattClientInternal.supervisionTimeout;
        gattClient.maxCeLength        = gattClientInternal.maxCeLength;
        gattClient.mtu                = gattClientInternal.mtu;
    }

    m_GattClientConditionList.isLargeCeLengthRequired = m_GattClientConditionListInternal.isLargeCeLengthRequired;

    m_GattClientConditionList.deviceCount           = m_GattClientConditionListInternal.deviceCount;
    m_GattClientConditionList.deviceCountCapacity   = m_GattClientConditionListInternal.deviceCountCapacity;

    return &m_GattClientConditionList;
}

void BleGattClientConditionContainer::Update(const GattClientConditionList* gattClientConditionList)
{
    m_GattClientConditionListInternal.isLargeCeLengthRequired   = gattClientConditionList->isLargeCeLengthRequired;
    m_GattClientConditionListInternal.deviceCountCapacity       = gattClientConditionList->deviceCountCapacity;

    for (auto& masterClient : m_GattClientConditionListInternal.gattClients)
    {
        for (const auto& client : gattClientConditionList->gattClients)
        {
            // 最新のリストにある接続のみを更新する
            if (masterClient.connectedServer.address == client.connectedServer.address &&
                masterClient.connectedServer.connectionHandle == client.connectedServer.connectionHandle)
            {
                masterClient.connectionInterval = client.connectionInterval;
                masterClient.slaveLatency       = client.slaveLatency;
                masterClient.supervisionTimeout = client.supervisionTimeout;
                masterClient.maxCeLength        = client.maxCeLength;

                break;
            }
        }
    }
}

void BleGattClientConditionContainer::Update(uint8_t deviceCountCapacity)
{
    if (deviceCountCapacity > NN_ARRAY_SIZE(m_GattClientConditionList.gattClients))
    {
        BTM_LOG("Invalid deviceCountCapacity\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::ResultInternalErrorFaultyDesign());
    }

    m_GattClientConditionListInternal.deviceCountCapacity = deviceCountCapacity;
}

int BleGattClientConditionContainer::GetClientIndex(uint32_t connHandle) const
{
    if (connHandle == nn::bluetooth::BleInvalidConnectionHandle)
    {
        return -1;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[i].connectedServer.connectionHandle == connHandle)
        {
            return i;
        }
    }

    return -1;
}

int BleGattClientConditionContainer::GetServiceIndex(uint32_t connHandle, uint16_t serviceHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return -1;
    }

    for (int i = 0; i < m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.serviceNum; ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.serviceList[i].handle == serviceHandle)
        {
            return i;
        }
    }

    return -1;
}

int BleGattClientConditionContainer::GetCharacteristicIndex(uint32_t connHandle, uint16_t charHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return -1;
    }

    for (int i = 0; i < m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.charNum; ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.charList[i].handle == charHandle)
        {
            return i;
        }
    }

    return -1;
}

int BleGattClientConditionContainer::GetDescriptorIndex(uint32_t connHandle, uint16_t descHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return -1;
    }

    for (int i = 0; i < m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.descNum; ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.descList[i].handle == descHandle)
        {
            return i;
        }
    }

    return -1;
}

bool BleGattClientConditionContainer::AddClient(uint8_t handle)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[i].handle == nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE)
        {
            m_GattClientConditionListInternal.gattClients[i].handle = handle;
            return true;
        }
    }

    return false;
}

bool BleGattClientConditionContainer::RemoveClient(uint8_t handle)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[i].handle == handle)
        {
            Clear(i);
            return true;
        }
    }

    return false;
}

bool BleGattClientConditionContainer::ConnectClient(uint8_t handle, uint32_t connHandle, BdAddress address)
{
    for (auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.handle == handle &&
            gattClient.connectedServer.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle &&
            gattClient.connectedServer.address == address &&
            gattClient.connectedServer.isPendingConnection)
        {
            m_GattClientConditionListInternal.deviceCount++;
            m_GattClientConditionListInternal.pendingConnectionCount--;

            gattClient.connectedServer.connectionHandle       = connHandle;
            gattClient.connectedServer.address                = address;
            gattClient.connectedServer.isPendingConnection    = false;

            return true;
        }
    }

    return false;
}

bool BleGattClientConditionContainer::DisconnectClient(uint8_t handle, uint32_t connHandle, uint16_t reason)
{
    bool ret = false;

    for(int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        auto& gattClient = m_GattClientConditionListInternal.gattClients[i];

        if (gattClient.handle == handle)
        {
            //!< 接続が完了していた
            if (gattClient.connectedServer.connectionHandle == connHandle &&
                gattClient.connectedServer.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
            {
                AddGattClientConnectionHistory(gattClient.connectedServer.connectionHandle,
                    gattClient.connectedServer.address,
                    static_cast<debug::BleDisconnectionReason>(reason));

                m_GattClientConditionListInternal.deviceCount--;

                // <! 接続が完了していた場合は、UpdateUsecase の必要がある
                ret = true;
            }
            //!< 接続が未完了だった
            else if (gattClient.connectedServer.isPendingConnection)
            {
                m_GattClientConditionListInternal.pendingConnectionCount--;
            }

            // 何があろうと Clear する
            Clear(i);
            gattClient.handle = handle;

            break;
        }
    }

    return ret;
}

bool BleGattClientConditionContainer::OverrideConnection(uint32_t connectionHandle, const nn::applet::AppletResourceUserId& aruid)
{
    int clientIndex = GetClientIndex(connectionHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.connectionOwnerApplet = aruid;

    return true;
}

void BleGattClientConditionContainer::AddPendingConnection(uint8_t handle, const BdAddress& address, const nn::applet::AppletResourceUserId& aruid)
{
    for (auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.isPendingConnection &&
            gattClient.connectedServer.address == address)
        {
            gattClient.connectedServer.connectionOwnerApplet = aruid;
            return;
        }
    }

    for (auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.handle == handle &&
            !gattClient.connectedServer.isPendingConnection &&
            gattClient.connectedServer.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
        {
            gattClient.connectedServer.address                  = address;
            gattClient.connectedServer.connectionOwnerApplet    = aruid;
            gattClient.connectedServer.isPendingConnection      = true;

            m_GattClientConditionListInternal.pendingConnectionCount++;

            return;
        }
    }
}

int BleGattClientConditionContainer::GetPendingConnection(BdAddress* pAddress, int addressCount) const
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);

    int outNum = 0;

    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (outNum < addressCount &&
            gattClient.connectedServer.isPendingConnection)
        {
            pAddress[outNum] = gattClient.connectedServer.address;
            outNum++;
        }
    }

    return outNum;
}

bool BleGattClientConditionContainer::IsPendingConnection(nn::applet::AppletResourceUserId* pAruid, const BdAddress& address)
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.isPendingConnection &&
            gattClient.connectedServer.address == address)
        {
            *pAruid = gattClient.connectedServer.connectionOwnerApplet;
            return true;
        }
    }

    return false;
}

void BleGattClientConditionContainer::SetConnectionRefusal(bool refusal)
{
    m_ConnectionRefusal = refusal;
}

bool BleGattClientConditionContainer::IsConnectionRefusal() const
{
    return m_ConnectionRefusal;
}

uint8_t BleGattClientConditionContainer::GetConnectableClientHandle() const
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.connectionHandle == nn::bluetooth::BleInvalidConnectionHandle &&
            !gattClient.connectedServer.isPendingConnection)
        {
            return gattClient.handle;
        }
    }

    return nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE;
}

uint8_t BleGattClientConditionContainer::GetClientHandle(uint32_t connHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE;
    }

    return m_GattClientConditionListInternal.gattClients[clientIndex].handle;
}

uint8_t BleGattClientConditionContainer::GetClientHandle(const BdAddress& address) const
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.address == address)
        {
            return gattClient.handle;
        }
    }

    return nn::bluetooth::BLE_INVALID_GATT_CLIENT_HANDLE;
}


uint32_t BleGattClientConditionContainer::GetConnectionHandle(BdAddress address) const
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.address == address)
        {
            return gattClient.connectedServer.connectionHandle;
        }
    }

    return nn::bluetooth::BleInvalidConnectionHandle;
}

const BdAddress* BleGattClientConditionContainer::GetGattServerAddress(uint32_t connHandle) const
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.connectedServer.connectionHandle == connHandle)
        {
            return &gattClient.connectedServer.address;
        }
    }

    return nullptr;
}

const nn::applet::AppletResourceUserId* BleGattClientConditionContainer::GetConnectionOwner(uint32_t connHandle)
{
    for (const auto& client : m_GattClientConditionListInternal.gattClients)
    {
        if (client.connectedServer.connectionHandle == connHandle &&
            client.connectedServer.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
        {
            return &client.connectedServer.connectionOwnerApplet;
        }
    }

    return nullptr;
}

bool BleGattClientConditionContainer::UpdateConnectionParameters(uint32_t connHandle, uint16_t connectionInterval, uint16_t slaveLatency, uint16_t supervisionTimeout)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].connectionInterval = connectionInterval;
    m_GattClientConditionListInternal.gattClients[clientIndex].slaveLatency       = slaveLatency;
    m_GattClientConditionListInternal.gattClients[clientIndex].supervisionTimeout = supervisionTimeout;

    return true;
}

bool BleGattClientConditionContainer::GetConnectionParameters(uint32_t connHandle,
                                                              uint16_t* pConnectionInterval, uint16_t* pSlaveLatency, uint16_t* pSupervisionTimeout, CeLength* pCeLengthMax) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    *pConnectionInterval    = m_GattClientConditionListInternal.gattClients[clientIndex].connectionInterval;
    *pSlaveLatency          = m_GattClientConditionListInternal.gattClients[clientIndex].slaveLatency;
    *pSupervisionTimeout    = m_GattClientConditionListInternal.gattClients[clientIndex].supervisionTimeout;
    *pCeLengthMax           = m_GattClientConditionListInternal.gattClients[clientIndex].maxCeLength;

    return true;
}

bool BleGattClientConditionContainer::UpdateConnectionParameterRequest(uint32_t connHandle,
                                                                       uint16_t connectionIntervalMin, uint16_t connectionIntervalMax,
                                                                       uint16_t slaveLatency, uint16_t supervisionTimeout)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].connectionIntervalReqMin = connectionIntervalMin;
    m_GattClientConditionListInternal.gattClients[clientIndex].connectionIntervalReqMax = connectionIntervalMax;
    m_GattClientConditionListInternal.gattClients[clientIndex].slaveLatencyReq          = slaveLatency;
    m_GattClientConditionListInternal.gattClients[clientIndex].supervisionTimeoutReq    = supervisionTimeout;

    return true;
}

bool BleGattClientConditionContainer::IsLargeCeLengthRequired()
{
    return m_GattClientConditionListInternal.isLargeCeLengthRequired;
}

bool BleGattClientConditionContainer::StartConfigureMtu(uint32_t connHandle)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].isConfiguringMtu = true;

    return true;
}

bool BleGattClientConditionContainer::UpdateMtu(uint32_t connHandle, uint16_t mtu)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].mtu = mtu;
    m_GattClientConditionListInternal.gattClients[clientIndex].isConfiguringMtu = false;

    return true;
}

uint16_t BleGattClientConditionContainer::GetMtu(uint32_t connHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return 0;
    }

    return m_GattClientConditionListInternal.gattClients[clientIndex].mtu;
}

bool BleGattClientConditionContainer::IsConfiguringMtu() const
{
    for (const auto& gattClient : m_GattClientConditionListInternal.gattClients)
    {
        if (gattClient.isConfiguringMtu)
        {
            return true;
        }
    }

    return false;
}

bool BleGattClientConditionContainer::SupportPairing(uint32_t connHandle)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.supportPairing = true;
    return true;
}

bool BleGattClientConditionContainer::CheckIfSupportPairing(uint32_t connHandle) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    return m_GattClientConditionListInternal.gattClients[clientIndex].connectedServer.supportPairing;
}

int BleGattClientConditionContainer::GetConnectionState(user::BleClientConnState(&connState)[nn::bluetooth::BleConnectionCountMaxClient]) const
{
    int validConnectionCount = 0;

    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConditionListInternal.gattClients); ++i)
    {
        if (m_GattClientConditionListInternal.gattClients[i].connectedServer.connectionHandle !=
            nn::bluetooth::BleInvalidConnectionHandle)
        {
            connState[i].address            = m_GattClientConditionListInternal.gattClients[i].connectedServer.address;
            connState[i].connectionHandle   = m_GattClientConditionListInternal.gattClients[i].connectedServer.connectionHandle;

            validConnectionCount++;
        }
        else
        {
            memset(connState[i].address.address, 0x00, SIZE_OF_BDADDRESS);
            connState[i].connectionHandle = nn::bluetooth::BleInvalidConnectionHandle;
        }
    }

    return validConnectionCount;
}

bool BleGattClientConditionContainer::AddGattServerService(uint32_t connHandle, user::GattService service)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int i = 0; i < nn::bluetooth::GattAttributeCountMaxClient; ++i)
    {
        if (pClient->connectedServer.serviceList[i].handle == nn::bluetooth::GattAttributeInvalidHandle &&
            pClient->connectedServer.serviceNum < nn::bluetooth::GattAttributeCountMaxClient)
        {
            pClient->connectedServer.serviceList[i] = service;
            pClient->connectedServer.serviceNum++;

            return true;
        }
    }

    return false;
}

uint8_t BleGattClientConditionContainer::GetGattServerServices(user::GattService* pServices, uint8_t inNum, uint32_t connHandle) const
{
    int clientIndex = GetClientIndex(connHandle);
    uint8_t outNum = 0;

    if (clientIndex < 0)
    {
        return outNum;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    outNum = (inNum < pClient->connectedServer.serviceNum) ? inNum : pClient->connectedServer.serviceNum;

    for (int i = 0; i < outNum; ++i)
    {
        memcpy(&pServices[i], &pClient->connectedServer.serviceList[i], sizeof(user::GattService));
    }

    return outNum;
}

const user::GattService* BleGattClientConditionContainer::GetGattServerService(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return nullptr;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int i = 0; i < pClient->connectedServer.serviceNum; ++i)
    {
        if (pClient->connectedServer.serviceList[i].uuid == uuid)
        {
            return &pClient->connectedServer.serviceList[i];
        }
    }

    return nullptr;
}

bool BleGattClientConditionContainer::AddGattServerCharacteristic(uint32_t connHandle, user::GattCharacteristic characteristic)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int i = 0; i < nn::bluetooth::GattAttributeCountMaxClient; ++i)
    {
        if (pClient->connectedServer.charList[i].handle == nn::bluetooth::GattAttributeInvalidHandle &&
            pClient->connectedServer.charNum < nn::bluetooth::GattAttributeCountMaxClient)
        {
            pClient->connectedServer.charList[i] = characteristic;
            pClient->connectedServer.charNum++;

            return true;
        }
    }

    return false;
}

uint8_t BleGattClientConditionContainer::GetGattServerCharacteristics(user::GattCharacteristic* pCharacteristic, uint8_t inNum, uint32_t connHandle, uint16_t serviceHandle) const
{
    int clientIndex = GetClientIndex(connHandle);
    int serviceIndex = GetServiceIndex(connHandle, serviceHandle);
    uint8_t outNum = 0;

    if (clientIndex < 0 || serviceIndex < 0)
    {
        return outNum;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];
    const user::GattService* pService = &pClient->connectedServer.serviceList[serviceIndex];

    for (int i = 0; i < pClient->connectedServer.charNum; ++i)
    {
        if (outNum == inNum)
        {
            return outNum;
        }

        if (pService->handle < pClient->connectedServer.charList[i].handle &&
            pClient->connectedServer.charList[i].handle <= pService->endGroupHandle)
        {
            memcpy(&pCharacteristic[outNum], &pClient->connectedServer.charList[i], sizeof(user::GattCharacteristic));
            outNum++;
        }
    }

    return outNum;
}

const user::GattCharacteristic* BleGattClientConditionContainer::GetGattServerCharacteristic(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return nullptr;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int i = 0; i < pClient->connectedServer.charNum; ++i)
    {
        if (pClient->connectedServer.charList[i].uuid == uuid)
        {
            return &pClient->connectedServer.charList[i];
        }
    }

    return nullptr;
}

bool BleGattClientConditionContainer::AddGattServerDescriptor(uint32_t connHandle, user::GattDescriptor descriptor)
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return false;
    }

    GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int j = 0; j < nn::bluetooth::GattAttributeCountMaxClient; ++j)
    {
        if (pClient->connectedServer.descList[j].handle == nn::bluetooth::GattAttributeInvalidHandle &&
            pClient->connectedServer.descNum < nn::bluetooth::GattAttributeCountMaxClient)
        {
            pClient->connectedServer.descList[j] = descriptor;
            pClient->connectedServer.descNum++;

            return true;
        }
    }

    return false;
}

uint8_t BleGattClientConditionContainer::GetGattServerDescritpors(user::GattDescriptor* pDescriptors, uint8_t inNum, uint32_t connHandle, uint16_t charHandle) const
{
    int clientIndex = GetClientIndex(connHandle);
    int charIndex = GetCharacteristicIndex(connHandle, charHandle);
    uint8_t outNum = 0;

    if (clientIndex < 0 || charIndex < 0)
    {
        return outNum;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];
    const user::GattCharacteristic* pChar = &pClient->connectedServer.charList[charIndex];

    if (charIndex == pClient->connectedServer.charNum - 1)
    {
        for (int i = 0; i < pClient->connectedServer.descNum; ++i)
        {
            if (pChar->handle <= pClient->connectedServer.descList[i].handle)
            {
                memcpy(&pDescriptors[outNum], &pClient->connectedServer.descList[i], sizeof(user::GattDescriptor));
                outNum++;
            }
        }
    }
    else
    {
        for (int i = 0; i < pClient->connectedServer.descNum; ++i)
        {
            if (pChar->handle <= pClient->connectedServer.descList[i].handle && pClient->connectedServer.descList[i].handle < pClient->connectedServer.charList[charIndex + 1].handle)
            {
                memcpy(&pDescriptors[outNum], &pClient->connectedServer.descList[i], sizeof(user::GattDescriptor));
                outNum++;
            }
        }
    }

    return outNum;
}

const user::GattDescriptor* BleGattClientConditionContainer::GetGattServerDescritpor(uint32_t connHandle, const nn::bluetooth::GattAttributeUuid& uuid) const
{
    int clientIndex = GetClientIndex(connHandle);

    if (clientIndex < 0)
    {
        return nullptr;
    }

    const GattClientInternal* pClient = &m_GattClientConditionListInternal.gattClients[clientIndex];

    for (int i = 0; i < pClient->connectedServer.descNum; ++i)
    {
        if (pClient->connectedServer.descList[i].uuid == uuid)
        {
            return &pClient->connectedServer.descList[i];
        }
    }

    return nullptr;
}

void BleGattClientConditionContainer::AddGattClientConnectionHistory(uint32_t connectionHandle, const BdAddress& address, debug::BleDisconnectionReason reason)
{
    int index = 0;

    for (index = 0; index < NN_ARRAY_SIZE(m_GattClientConnectionHistroyList); ++index)
    {
        if (m_GattClientConnectionHistroyList[index].connectionHandle == nn::bluetooth::BleInvalidConnectionHandle)
        {
            m_GattClientConnectionHistroyList[index].address             = address;
            m_GattClientConnectionHistroyList[index].connectionHandle    = connectionHandle;
            m_GattClientConnectionHistroyList[index].disconnectionReason = reason;

            return;
        }
    }

    // 空きが無かった
    // 上詰め
    for (int i = 0; i < NN_ARRAY_SIZE(m_GattClientConnectionHistroyList) - 1; ++i)
    {
        m_GattClientConnectionHistroyList[i] = m_GattClientConnectionHistroyList[i + 1];
    }

    // 上書き
    m_GattClientConnectionHistroyList[index - 1].address                = address;
    m_GattClientConnectionHistroyList[index - 1].connectionHandle       = connectionHandle;
    m_GattClientConnectionHistroyList[index - 1].disconnectionReason    = reason;

    return;
}

bool BleGattClientConditionContainer::GetGattClientDisconnectionReason(debug::BleDisconnectionReason* pOutReason,
                                                                       uint32_t connectionHandle, const BdAddress& address)
{
    // 新しい側（配列の後ろ）から走査
    for (int i = NN_ARRAY_SIZE(m_GattClientConnectionHistroyList) - 1; i >= 0; --i)
    {
        if (m_GattClientConnectionHistroyList[i].address == address &&
            m_GattClientConnectionHistroyList[i].connectionHandle == connectionHandle)
        {
            *pOutReason = m_GattClientConnectionHistroyList[i].disconnectionReason;
            return true;
        }
    }

    *pOutReason = debug::BleDisconnectionReason_Unknown;

    return false;
}

bool BleGattClientConditionContainer::GetConnectionParameter(uint16_t* pInterval, uint16_t *pLatency, uint16_t* pTimeout,
                                                             uint32_t connHandle) const
{
    for (auto client : m_GattClientConditionListInternal.gattClients)
    {
        if (client.connectedServer.connectionHandle == connHandle)
        {
            *pInterval  = client.connectionInterval;
            *pLatency   = client.slaveLatency;
            *pTimeout   = client.supervisionTimeout;

            return true;
        }
    }

    *pInterval  = 0;
    *pLatency   = 0;
    *pTimeout   = 0;

    return false;
}

bool BleGattClientConditionContainer::GetConnectionParameterReq(uint16_t* pIntervaMin, uint16_t* pIntervalMax, uint16_t *pLatency, uint16_t* pTimeout,
                                                                uint32_t connHandle) const
{
    for (auto client : m_GattClientConditionListInternal.gattClients)
    {
        if (client.connectedServer.connectionHandle == connHandle)
        {
            *pIntervaMin    = client.connectionIntervalReqMin;
            *pIntervalMax   = client.connectionIntervalReqMax;
            *pLatency       = client.slaveLatency;
            *pTimeout       = client.supervisionTimeout;

            return true;
        }
    }

    *pIntervaMin    = 0;
    *pIntervalMax   = 0;
    *pLatency       = 0;
    *pTimeout       = 0;

    return false;
}

//--------------------------------------------------
// BleScannerCondition
//--------------------------------------------------
const BleScannerCondition::BleScanParameter BleScannerCondition::BleScanParameter_LowDuty   = { 512, 10 };
const BleScannerCondition::BleScanParameter BleScannerCondition::BleScanParameter_HighDuty  = { 512, 10 };

BleScannerCondition::BleScannerCondition()
{
    m_IsScanning = false;

    m_ScanFilterForGeneral.filterIndex          = BLE_SCAN_FILTER_INDEX_FOR_GENERAL;
    m_ScanFilterForGeneral.conditionNumMax      = BLE_SCAN_CONDITION_MAX_FOR_GENERAL;

    m_ScanFilterForPaired.filterIndex           = BLE_SCAN_FILTER_INDEX_FOR_PAIRED;
    m_ScanFilterForPaired.conditionNumMax       = BLE_SCAN_CONDITION_MAX_FOR_PAIRED;

    m_ScanFilterForWakeOn.filterIndex           = BLE_SCAN_FILTER_INDEX_FOR_WAKE_ON;
    m_ScanFilterForWakeOn.conditionNumMax       = BLE_SCAN_CONDITION_MAX_FOR_WAKE_ON;

    m_ScanFilterForUnconnectableScan.filterIndex       = BLE_SCAN_FILTER_INDEX_FOR_UNCONNECTABLE_SCAN;
    m_ScanFilterForUnconnectableScan.conditionNumMax   = BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN;

    m_ScanFilterForSdGeneral.filterIndex        = BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL;
    m_ScanFilterForSdGeneral.conditionNumMax    = BLE_SCAN_CONDITION_MAX_FOR_SD_GENERAL;

    m_ScanFilterForSdUnconnectableScan.filterIndex     = BLE_SCAN_FILTER_INDEX_FOR_SD_UNCONNECTABLE_SCAN;
    m_ScanFilterForSdUnconnectableScan.conditionNumMax = BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN;

    m_ScanResultListGeneral.count = 0;
    m_ScanResultListSd.count = 0;

    ClearScanResultsGeneral();
    ClearScanResultsSd();

    for (int i = 0; i < BLE_SCAN_CONDITION_MAX_FOR_UNCONNECTABLE_SCAN; ++i)
    {
        m_UnconnectableScanResultListGeneral[i].inUse = false;
    }

    for (int i = 0; i < BLE_SCAN_CONDITION_MAX_FOR_SD_UNCONNECTABLE_SCAN; ++i)
    {
        m_UnconnectableScanResultListSd[i].inUse = false;
    }

    ClearFilters();
}

BleScannerCondition::ScanFilter* BleScannerCondition::GetScanFilter(uint8_t filterIndex)
{
    switch (filterIndex)
    {
    case BLE_SCAN_FILTER_INDEX_FOR_GENERAL:
        return &m_ScanFilterForGeneral;
    case BLE_SCAN_FILTER_INDEX_FOR_PAIRED:
        return &m_ScanFilterForPaired;
    case BLE_SCAN_FILTER_INDEX_FOR_WAKE_ON:
        return &m_ScanFilterForWakeOn;
    case BLE_SCAN_FILTER_INDEX_FOR_UNCONNECTABLE_SCAN:
        return &m_ScanFilterForUnconnectableScan;
    case BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL:
        return &m_ScanFilterForSdGeneral;
    case BLE_SCAN_FILTER_INDEX_FOR_SD_UNCONNECTABLE_SCAN:
        return &m_ScanFilterForSdUnconnectableScan;
    default:
        break;
    }

    return nullptr;
}

void BleScannerCondition::ClearFilters()
{
    DeleteFilter(m_ScanFilterForGeneral.filterIndex);
    DeleteFilter(m_ScanFilterForPaired.filterIndex);
    DeleteFilter(m_ScanFilterForWakeOn.filterIndex);
    DeleteFilter(m_ScanFilterForUnconnectableScan.filterIndex);
    DeleteFilter(m_ScanFilterForSdGeneral.filterIndex);
    DeleteFilter(m_ScanFilterForSdUnconnectableScan.filterIndex);
}

void BleScannerCondition::DeleteFilter(uint8_t filterIndex)
{
    ScanFilter *pFilter = GetScanFilter(filterIndex);
    NN_ABORT_UNLESS_NOT_NULL(pFilter);

    for (int i = 0; i < pFilter->conditionNumMax; ++i)
    {
        pFilter->conditions[i].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
        memset(&(pFilter->conditions[i].condition), 0x00, sizeof(nn::bluetooth::BleAdvertiseFilter));
        pFilter->conditionNum = 0;
        pFilter->conditions[i].inUse = false;
    }
}

bool BleScannerCondition::GetIsScanConditionRegistered(const nn::bluetooth::BleAdvertiseFilter& condition)
{
    ScanFilter *pFilter = GetScanFilter(condition.filterIndex);
    if (!pFilter)
    {
        return false;
    }

    for (int i = 0; i < pFilter->conditionNumMax; ++i)
    {
        if (pFilter->conditions[i].inUse && pFilter->conditions[i].condition == condition)
        {
            // ARUID はチェックしない
            return true;
        }
    }

    return false;
}

bool BleScannerCondition::AddFilterCondition(const ScanFilterCondition& condition)
{
    ScanFilter *pFilter = GetScanFilter(condition.condition.filterIndex);
    if (!pFilter)
    {
        return false;
    }

    // すでにCondition がないかチェック
    if (GetIsScanConditionRegistered(condition.condition))
    {
        return true;
    }

    for (int i = 0; i < pFilter->conditionNumMax; ++i)
    {
        if (pFilter->conditions[i].inUse == false)
        {
            pFilter->conditions[i].aruid        = condition.aruid;
            pFilter->conditions[i].condition    = condition.condition;
            pFilter->conditions[i].inUse        = true;

            pFilter->conditionNum++;
            NN_ABORT_UNLESS_LESS_EQUAL(pFilter->conditionNum, pFilter->conditionNumMax);

            return true;
        }
    }

    return false;
}

bool BleScannerCondition::DeleteFilterCondition(const ScanFilterCondition& condition)
{
    ScanFilter *pFilter = GetScanFilter(condition.condition.filterIndex);
    if (!pFilter)
    {
        return false;
    }

    for (int i = 0; i < pFilter->conditionNumMax; ++i)
    {
        if (pFilter->conditions[i].inUse &&
            pFilter->conditions[i].condition == condition.condition &&
            pFilter->conditions[i].aruid == condition.aruid)
        {
            pFilter->conditions[i].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
            memset(&(pFilter->conditions[i].condition), 0x00, sizeof(nn::bluetooth::BleAdvertiseFilter));
            pFilter->conditions[i].inUse = false;

            pFilter->conditionNum--;
            NN_ABORT_UNLESS_GREATER_EQUAL(pFilter->conditionNum, 0);

            break;
        }
    }

    return true;
}

const nn::applet::AppletResourceUserId* BleScannerCondition::GetFilterConditionOwner(const nn::bluetooth::BleAdvertiseFilter& condition)
{
    ScanFilter* pFilter = GetScanFilter(condition.filterIndex);

    if (!pFilter)
    {
        return nullptr;
    }

    for (int i = 0; i < pFilter->conditionNumMax; ++i)
    {
        if (pFilter->conditions[i].inUse && pFilter->conditions[i].condition == condition)
        {
            return &pFilter->conditions[i].aruid;
        }
    }

    return nullptr;
}

uint8_t BleScannerCondition::GetFilterConditionCount(uint8_t filterIndex)
{
    ScanFilter *pFilter = GetScanFilter(filterIndex);
    if (!pFilter)
    {
        return BLE_SCAN_CONDITION_MAX;
    }

    return pFilter->conditionNum;
}

uint8_t BleScannerCondition::GetFilterConditionCountMax(uint8_t filterIndex)
{
    ScanFilter *pFilter = GetScanFilter(filterIndex);
    if (!pFilter)
    {
        return 0;
    }

    return pFilter->conditionNumMax;
}

bool BleScannerCondition::IsFilterConditionExisting()
{
    return (m_ScanFilterForGeneral.conditionNum != 0)               ||
           (m_ScanFilterForSdGeneral.conditionNum != 0)             ||
           (m_ScanFilterForPaired.conditionNum != 0)                ||
           (m_ScanFilterForUnconnectableScan.conditionNum != 0)     ||
           (m_ScanFilterForSdUnconnectableScan.conditionNum != 0)   ||
           (m_ScanFilterForSdUnconnectableScan.conditionNum != 0);
}

bool BleScannerCondition::GetIfMatchScanFilter(nn::bluetooth::BleAdvertiseFilter* pCondition, nn::applet::AppletResourceUserId* pOwner,
                                               uint8_t filterIndex, user::ScanResult result)
{
    bool filterConditionExist = false;
    ScanFilter *pFilter = GetScanFilter(filterIndex);

    // フィルタが存在しなければfalse
    if (!pFilter)
    {
        return filterConditionExist;
    }

    for (int i = 0; i < pFilter->conditionNum; ++i)
    {
        if (pFilter->conditions[i].inUse)
        {
            int targetResultAdTypeIndex = -1;

            // ScanResult がFilterCondition が指定するAdType を含むかチェック
            for (int j = 0; j < result.adStructureNum; ++j)
            {
                if (pFilter->conditions[i].condition.structure.adType == result.adStructures[j].adType)
                {
                    targetResultAdTypeIndex = j;
                    break;
                }
            }

            // 含まなければ次のFilterCondition へ
            if (targetResultAdTypeIndex == -1)
            {
                continue;
            }

            bool IsFilterMatched = true;

            // ScanResult のデータとFilterConditionMask のAnd をとり、FilterCondition と一致すれば、フィルタが存在
            for (int j = 0; j < pFilter->conditions[i].condition.maskeLength; ++j)
            {
                if ((result.adStructures[targetResultAdTypeIndex].data[j] & pFilter->conditions[i].condition.mask[j]) != pFilter->conditions[i].condition.structure.data[j])
                {
                    IsFilterMatched = false;
                    break;
                }
            }

            if (IsFilterMatched)
            {
                *pCondition = pFilter->conditions[i].condition;
                *pOwner     = pFilter->conditions[i].aruid;

                filterConditionExist = true;

                break;
            }
            else
            {
                continue;
            }
        }
    }

    return filterConditionExist;
}

int BleScannerCondition::GetScanFilterCondition(ScanFilterCondition* pOutConditions, int inNum, uint8_t filterIndex)
{
    int outNum = 0;
    ScanFilter *pFilter = GetScanFilter(filterIndex);

    // フィルタが存在しなければ 0
    if (!pFilter)
    {
        return outNum;
    }

    for (int i = 0; i < pFilter->conditionNumMax && outNum < inNum; ++i)
    {
        if (pFilter->conditions[i].inUse)
        {
            pOutConditions[outNum] = pFilter->conditions[i];
            outNum++;
        }
    }

    return outNum;
}

void BleScannerCondition::ClearScanResultsGeneral()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResultListGeneral.results); ++i)
    {
        m_ScanResultListGeneral.results[i].result.rssi = 0;
        m_ScanResultListGeneral.results[i].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
        memset(m_ScanResultListGeneral.results[i].result.address.address, 0x00, SIZE_OF_BDADDRESS);
        m_ScanResultListGeneral.results[i].scannedTick = nn::os::Tick(0);
    }

    m_ScanResultListGeneral.count = 0;
}

bool BleScannerCondition::AddScanResultsGeneral(const user::ScanResult result)
{
    size_t listSize = NN_ARRAY_SIZE(m_ScanResultListGeneral.results);

    // 追加時にもタイムアウトチェックする
    RefreshScanResultGeneral();

    nn::bluetooth::BleAdvertiseFilter   condition;
    nn::applet::AppletResourceUserId    aruid;

    // スキャンフィルタに一致するかチェック
    if (GetIfMatchScanFilter(&condition, &aruid, BLE_SCAN_FILTER_INDEX_FOR_GENERAL, result) == false)
    {
        return false;
    }

    NN_UNUSED(condition);

    bool isAlreadyFoundDevice = false;

    // 同じアドレスのデータがあれば上書き
    for (int i = 0; i < listSize; ++i)
    {
        if (m_ScanResultListGeneral.results[i].result.address == result.address)
        {
            m_ScanResultListGeneral.results[i].aruid  = aruid;
            m_ScanResultListGeneral.results[i].result = result;
            m_ScanResultListGeneral.results[i].scannedTick = nn::os::GetSystemTick();

            isAlreadyFoundDevice = true;
            break;
        }
    }

    // 見つかっていない場合、
    if (!isAlreadyFoundDevice)
    {
        // 末尾の結果のRSSI = 0 (埋まっていない）なら、とりあえず末尾に追加
        if (m_ScanResultListGeneral.results[listSize - 1].result.rssi == 0)
        {
            m_ScanResultListGeneral.results[listSize - 1].aruid  = aruid;
            m_ScanResultListGeneral.results[listSize - 1].result = result;
            m_ScanResultListGeneral.results[listSize - 1].scannedTick = nn::os::GetSystemTick();
            m_ScanResultListGeneral.count++;
        }
        // 末尾の結果のRSSI != 0（埋まっている）なら、RSSI を比較して、大きければ末尾を上書き
        else if (m_ScanResultListSd.results[listSize - 1].result.rssi < result.rssi)
        {
            m_ScanResultListGeneral.results[listSize - 1].aruid  = aruid;
            m_ScanResultListGeneral.results[listSize - 1].result = result;
            m_ScanResultListGeneral.results[listSize - 1].scannedTick = nn::os::GetSystemTick();
        }
        else
        {
            // 追加できないのでfalse
            return false;
        }
    }

    // RSSI の降順にソート
    for (int i = 0; i < listSize; ++i)
    {
        for (int j = listSize - 1; j > i; j--)
        {
            // 比較先が未発見デバイスなら無視
            if (m_ScanResultListGeneral.results[j].result.rssi == 0)
            {
                continue;
            }
            // 自身が未発見デバイスなら無条件に交換
            if (m_ScanResultListGeneral.results[i].result.rssi == 0)
            {
                TimedScanResult tempResult = m_ScanResultListGeneral.results[i];
                m_ScanResultListGeneral.results[i] = m_ScanResultListGeneral.results[j];
                m_ScanResultListGeneral.results[j] = tempResult;
                // 元々ソートされているのでbreak
                break;
            }
            // 自身が発見済みデバイスならRSSI を比較して、交換
            else if (m_ScanResultListGeneral.results[i].result.rssi < m_ScanResultListGeneral.results[j].result.rssi)
            {
                TimedScanResult tempResult = m_ScanResultListGeneral.results[i];
                m_ScanResultListGeneral.results[i] = m_ScanResultListGeneral.results[j];
                m_ScanResultListGeneral.results[j] = tempResult;
                // 元々ソートされているのでbreak
                break;
            }
        }
    }

    return true;
}

void BleScannerCondition::RefreshScanResultGeneral()
{
    size_t listSize = NN_ARRAY_SIZE(m_ScanResultListGeneral.results);
    nn::os::Tick currentTick = nn::os::GetSystemTick();

    for (int i = 0; i < listSize; ++i)
    {
        if (m_ScanResultListGeneral.results[i].scannedTick != nn::os::Tick(0) &&
            (currentTick - m_ScanResultListGeneral.results[i].scannedTick).ToTimeSpan().GetMilliSeconds() > BLE_SCAN_RESULT_TIMEOUT)
        {
            // 上詰めする
            if (i < listSize - 1)
            {
                for (int j = i; j < listSize - 1; ++j)
                {
                    m_ScanResultListGeneral.results[j] = m_ScanResultListGeneral.results[j + 1];
                }
            }

            m_ScanResultListGeneral.results[listSize - 1].result.rssi = 0;
            m_ScanResultListGeneral.results[listSize - 1].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
            memset(m_ScanResultListGeneral.results[listSize - 1].result.address.address, 0x00, SIZE_OF_BDADDRESS);
            m_ScanResultListGeneral.results[listSize - 1].scannedTick = nn::os::Tick(0);

            m_ScanResultListGeneral.count--;
        }
    }
}

const BleScannerCondition::ScanResultList* BleScannerCondition::GetScanResultsGeneral()
{
    return &m_ScanResultListGeneral;
}

void BleScannerCondition::ClearScanResultsSd()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResultListSd.results); ++i)
    {
        m_ScanResultListSd.results[i].result.rssi = 0;
        m_ScanResultListSd.results[i].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
        memset(m_ScanResultListSd.results[i].result.address.address, 0x00, SIZE_OF_BDADDRESS);
        m_ScanResultListSd.results[i].scannedTick = nn::os::Tick(0);
    }

    m_ScanResultListSd.count = 0;
}

bool BleScannerCondition::AddScanResultsSd(const user::ScanResult result)
{
    // 追加時にもタイムアウトチェックする
    RefreshScanResultSd();

    nn::bluetooth::BleAdvertiseFilter   condition;
    nn::applet::AppletResourceUserId    aruid;

    // スキャンフィルタに一致するかチェック
    if (GetIfMatchScanFilter(&condition, &aruid, BLE_SCAN_FILTER_INDEX_FOR_SD_GENERAL, result) == false)
    {
        return false;
    }

    NN_UNUSED(condition);

    size_t listSize = NN_ARRAY_SIZE(m_ScanResultListSd.results);
    bool isAlreadyFoundDevice = false;

    // 同じアドレスのデータがあれば上書き
    for (int i = 0; i < listSize; ++i)
    {
        if (m_ScanResultListSd.results[i].result.address == result.address)
        {
            m_ScanResultListSd.results[i].aruid  = aruid;
            m_ScanResultListSd.results[i].result = result;
            m_ScanResultListSd.results[i].scannedTick = nn::os::GetSystemTick();

            isAlreadyFoundDevice = true;
            break;
        }
    }

    // 見つかっていない場合、
    if(!isAlreadyFoundDevice)
    {
        // 末尾の結果のRSSI = 0 (埋まっていない）なら、とりあえず末尾に追加
        // 末尾の結果のRSSI != 0（埋まっている）なら、RSSI を比較して、大きければ末尾を上書き
        if(m_ScanResultListSd.results[listSize - 1].result.rssi == 0)
        {
            m_ScanResultListSd.results[listSize - 1].aruid  = aruid;
            m_ScanResultListSd.results[listSize - 1].result = result;
            m_ScanResultListSd.results[listSize - 1].scannedTick = nn::os::GetSystemTick();
            m_ScanResultListSd.count++;
        }
        else if(m_ScanResultListSd.results[listSize - 1].result.rssi < result.rssi)
        {
            m_ScanResultListSd.results[listSize - 1].aruid  = aruid;
            m_ScanResultListSd.results[listSize - 1].result = result;
            m_ScanResultListSd.results[listSize - 1].scannedTick = nn::os::GetSystemTick();
        }
        else
        {
            // 追加できないのでfalse
            return false;
        }
    }

    // RSSI の降順にソート(要素数が少ないのでバブルソート)
    for (int i = 0; i < listSize; ++i)
    {
        for(int j = listSize - 1; j > i; j--)
        {
            // 自身が未発見デバイスなら無視
            if(m_ScanResultListSd.results[j].result.rssi == 0)
            {
                continue;
            }
            // 移動先が未発見デバイスなら無条件に交換
            if(m_ScanResultListSd.results[i].result.rssi == 0)
            {
                TimedScanResult tempResult = m_ScanResultListSd.results[i];
                m_ScanResultListSd.results[i] = m_ScanResultListSd.results[j];
                m_ScanResultListSd.results[j] = tempResult;
                // 元々ソートされているのでbreak
                break;
            }
            // 移動先が発見済みデバイスならRSSI を比較して、交換
            else if(m_ScanResultListSd.results[i].result.rssi < m_ScanResultListSd.results[j].result.rssi)
            {
                TimedScanResult tempResult = m_ScanResultListSd.results[i];
                m_ScanResultListSd.results[i] = m_ScanResultListSd.results[j];
                m_ScanResultListSd.results[j] = tempResult;
                // 元々ソートされているのでbreak
                break;
            }
        }
    }

    return true;
}

void BleScannerCondition::RefreshScanResultSd()
{
    size_t listSize = NN_ARRAY_SIZE(m_ScanResultListSd.results);
    nn::os::Tick currentTick = nn::os::GetSystemTick();

    for (int i = 0; i < listSize; ++i)
    {
        if (m_ScanResultListSd.results[i].scannedTick != nn::os::Tick(0) &&
            (currentTick - m_ScanResultListSd.results[i].scannedTick).ToTimeSpan().GetMilliSeconds() > BLE_SCAN_RESULT_TIMEOUT)
        {
            // 上詰めする
            if (i < listSize - 1)
            {
                for (int j = i; j < listSize - 1; ++j)
                {
                    m_ScanResultListSd.results[j] = m_ScanResultListSd.results[j + 1];
                }
            }

            m_ScanResultListSd.results[listSize - 1].result.rssi = 0;
            m_ScanResultListSd.results[listSize - 1].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
            memset(m_ScanResultListSd.results[listSize - 1].result.address.address, 0x00, SIZE_OF_BDADDRESS);
            m_ScanResultListSd.results[listSize - 1].scannedTick = nn::os::Tick(0);

            m_ScanResultListSd.count--;
        }
    }
}

const BleScannerCondition::ScanResultList* BleScannerCondition::GetScanResultsSd()
{
    return &m_ScanResultListSd;
}

void BleScannerCondition::ClearUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition)
{

}

void BleScannerCondition::AddUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition, const user::ScanResult result)
{

}

const BleScannerCondition::UnconnectableScanResult* BleScannerCondition::GetUnconnectableScanResultGeneral(const nn::bluetooth::BleAdvertiseFilter& condition) const
{
    return nullptr;
}

void BleScannerCondition::ClearUnconnectableScanResultSd(const nn::bluetooth::BleAdvertiseFilter& condition)
{

}

void BleScannerCondition::AddUnconnectableResultSd(const nn::bluetooth::BleAdvertiseFilter& condition, const user::ScanResult result)
{

}

const BleScannerCondition::UnconnectableScanResult* BleScannerCondition::GetUnconnectableScanResultSd(const nn::bluetooth::BleAdvertiseFilter& condition) const
{
    return nullptr;
}

bool BleScannerCondition::GetScanFilter(nn::btm::user::BleAdvFilterForGeneral* pFilter, uint16_t parameterId) const
{
    for (int i = 0; i < sizeof(BleScanFilterParameterList) / sizeof(BleScanFilterParameterList[0]); ++i)
    {
        if (BleScanFilterParameterList[i].filterType == BLE_SCAN_FILTER_TYPE_GENERAL && BleScanFilterParameterList[i].parameterId == parameterId)
        {
            memcpy(pFilter->manufacturerId, BleScanFilterParameterList[i].generalFilter.manufacturerId, sizeof(BleScanFilterParameterList[i].generalFilter.manufacturerId));
            memcpy(pFilter->clientId, BleScanFilterParameterList[i].generalFilter.clientId, sizeof(BleScanFilterParameterList[i].generalFilter.clientId));
            memcpy(pFilter->serverId, BleScanFilterParameterList[i].generalFilter.serverId, sizeof(BleScanFilterParameterList[i].generalFilter.serverId));

            return true;
       }
    }

    return false;
}

bool BleScannerCondition::GetScanFilter(nn::btm::user::BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) const
{
    for (int i = 0; i < sizeof(BleScanFilterParameterList) / sizeof(BleScanFilterParameterList[0]); ++i)
    {
        if (BleScanFilterParameterList[i].filterType == BLE_SCAN_FILTER_TYPE_SMART_DEVICE && BleScanFilterParameterList[i].parameterId == parameterId)
        {
            memcpy(pFilter->uu.uuid128, BleScanFilterParameterList[i].smartDeviceFilter.uu.uuid128, nn::bluetooth::GattAttributeUuidLength_128);

            return true;
        }
    }

    return false;
}

//--------------------------------------------------
// BlePairingInfoContainer
//--------------------------------------------------
const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_SERVICE_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x6C, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_COMMAND_CHARACTERISTIC_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x6D, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_WRITER_CHARACTERISTIC_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x6E, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_ADDRESS_READER_CHARACTERISTIC_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x6F, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_ADV_DATA_READER_CHARACTERISTIC_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x70, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

const nn::bluetooth::GattAttributeUuid BlePairingInfoContainer::NN_GATT_PAIRING_ERROR_HANDLER_CHARACTERISTIC_UUID =
{
    .length = nn::bluetooth::GattAttributeUuidLength_128,
    .uu.uuid128 = { 0x71, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
};

BlePairingInfoContainer::BlePairingInfoContainer()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_BlePairingInfoList); ++i)
    {
        memset(&m_BlePairingInfoList[i], 0x00, sizeof(BlePairingInfo));
    }
}

void BlePairingInfoContainer::InitializeBlePairingState()
{
    memset(&m_PairingDeviceInfo, 0x00, sizeof(BlePairingInfo));

    m_Stage = BlePairingStage_Init;
}

bool BlePairingInfoContainer::SetPairingDevice(const BlePairingInfo& info)
{
    if (m_Stage == BlePairingStage_Init)
    {
        m_PairingDeviceInfo = info;
        return true;
    }

    return false;
}

const BlePairingInfoContainer::BlePairingInfo* BlePairingInfoContainer::GetPairingDevice() const
{
    if (m_Stage != BlePairingStage_Init)
    {
        return &m_PairingDeviceInfo;
    }

    return nullptr;
}

bool BlePairingInfoContainer::AddPairingInfo(const BlePairingInfo& info)
{
    // ペアリング情報がすでにある場合、一旦削除して、再度追加する
    RemovePairingInfo(info);

    // 先頭ほど新しい
    for (int i = NN_ARRAY_SIZE(m_BlePairingInfoList) - 1; i > 0; --i)
    {
        m_BlePairingInfoList[i] = m_BlePairingInfoList[i - 1];
    }
    m_BlePairingInfoList[0] = info;

    return true;
}

bool BlePairingInfoContainer::RemovePairingInfo(const BlePairingInfo& info)
{
    size_t listSize = NN_ARRAY_SIZE(m_BlePairingInfoList);

    for (int i = 0; i < listSize; ++i)
    {
        if (m_BlePairingInfoList[i] == info)
        {
            memset(&m_BlePairingInfoList[i], 0x00, sizeof(BlePairingInfo));

            // 前詰め
            for (int j = i; j < listSize - 1; ++j)
            {
                m_BlePairingInfoList[j] = m_BlePairingInfoList[j + 1];
            }
            memset(&m_BlePairingInfoList[listSize - 1], 0x00, sizeof(BlePairingInfo));

            return true;
        }
    }

    return false;
}

bool BlePairingInfoContainer::IsPaired(const BlePairingInfo& info) const
{
    for (auto tempInfo : m_BlePairingInfoList)
    {
        if (tempInfo == info)
        {
            return true;
        }
    }

    return false;
}

void BlePairingInfoContainer::Load()
{
    nn::settings::system::BlePairingSettings saveData[nn::settings::system::BlePairingSettingsCountMax];

    int count = nn::settings::system::GetBlePairingSettings(saveData, NN_ARRAY_SIZE(saveData));

    for (int i = 0; i < count; ++i)
    {
        memcpy(m_BlePairingInfoList[i].address.address, saveData[i].bd_addr, SIZE_OF_BDADDRESS);
        m_BlePairingInfoList[i].type = static_cast<BlePairingAdvDataType>(saveData[i].pairing_type);

        switch (saveData[i].pairing_type)
        {
        case BlePairingAdvDataType_ManuCliSerId:
            memcpy(m_BlePairingInfoList[i].advInfo.manufacturerId, &saveData[i].manufacturer_id, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy(m_BlePairingInfoList[i].advInfo.clientId, saveData[i].client_id, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy(m_BlePairingInfoList[i].advInfo.serverId, saveData[i].server_id, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            break;
        case BlePairingAdvDataType_Invalid:
        default:
            BTM_LOG("Invalid BLE pairing save data. Ignore.\n");

            memset(m_BlePairingInfoList[i].address.address, 0x00, SIZE_OF_BDADDRESS);
            memset(&m_BlePairingInfoList[i].advInfo, 0x00, sizeof(user::BleAdvFilterForGeneral));
            break;
        }
    }
}

void BlePairingInfoContainer::Save() const
{
    nn::settings::system::BlePairingSettings saveData[nn::settings::system::BlePairingSettingsCountMax];

    for (int i = 0; i < NN_ARRAY_SIZE(m_BlePairingInfoList); ++i)
    {
        memcpy(saveData[i].bd_addr, m_BlePairingInfoList[i].address.address, SIZE_OF_BDADDRESS);
        saveData[i].pairing_type = static_cast<uint8_t>(m_BlePairingInfoList[i].type);

        switch (m_BlePairingInfoList[i].type)
        {
        case BlePairingAdvDataType_ManuCliSerId:
            memcpy(&saveData[i].manufacturer_id, m_BlePairingInfoList[i].advInfo.manufacturerId, nn::bluetooth::BleNnAdvertiseManufacturerIdSize);
            memcpy(&saveData[i].client_id, m_BlePairingInfoList[i].advInfo.clientId, nn::bluetooth::BleNnAdvertiseManufactureClientIdSize);
            memcpy(&saveData[i].server_id, m_BlePairingInfoList[i].advInfo.serverId, nn::bluetooth::BleNnAdvertiseManufactureServerIdSize);
            break;
        case BlePairingAdvDataType_Invalid:
        default:
            BTM_LOG("Invalid BLE pairing save data. Ignore.\n");

            memset(&saveData[i], 0x00, sizeof(nn::settings::system::BlePairingSettings));
            break;
        }
    }

    nn::settings::system::SetBlePairingSettings(saveData, NN_ARRAY_SIZE(saveData));
}

}}  // namespace nn::btm
