﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "boot2_Boot.h"
#include "boot2_Maintenance.h"

#include <nn/util/util_ScopeExit.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/pm/pm_ShellApi.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/svc/svc_Result.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#if !defined(NN_SIGLO_BOOT_LIST_FILE)
#error "NN_SIGLO_BOOT_LIST_FILE が定義されていません。"
#endif

namespace nn { namespace boot2 {

    namespace
    {
        const Bit64 BootProgramIds[] =
        {
            NN_SIGLO_BOOT_PROGRAM_IDS
        };

        void LaunchEachBootProgram(const ncm::ProgramId& Id)
        {
            os::ProcessId processId;
            ncm::ProgramId programId = Id;
            auto location = ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, programId);

            if (programId == ncm::ProgramId::GetInvalidId())
            {
                NN_SDK_LOG("[boot2] Program ID '%016llx' is invalid.\n", programId.value);
                return;
            }

            nn::Result result = nn::pm::LaunchProgram(&processId, location, 0);

            // リソース不足による失敗はアボートします
            NN_ABORT_UNLESS(!(result <= nn::svc::ResultOutOfResource()));
            NN_ABORT_UNLESS(!(result <= nn::svc::ResultOutOfMemory()));
            NN_ABORT_UNLESS(!(result <= nn::svc::ResultLimit()));

            if (result.IsFailure())
            {
                NN_SDK_LOG("[boot2] Failed to run '%016llx'(0x%08x). Skip it.\n", programId.value, result.GetInnerValueForDebug());
            }
        }

        bool NeedsBootingDevMenuAppInstaller()
        {
            bool needsBooting = false;
            settings::fwdbg::GetSettingsItemValue(&needsBooting, sizeof(needsBooting), "boot", "boot_devmenuapp_installer");
            return needsBooting;
        }
    }   // anonymous namespace


    void LaunchBootPrograms()
    {
        for( auto id : BootProgramIds )
        {
            ncm::ProgramId programId = { id };

            LaunchEachBootProgram(programId);
        }

#ifdef NN_SIGLO_SECOND_BOOT_PROGRAM_IDS
        static const Bit64 SecondBootProgramIds[] =
        {
            NN_SIGLO_SECOND_BOOT_PROGRAM_IDS
        };

        static const Bit64 MaintenanceSecondBootProgramIds[] =
        {
            NN_SIGLO_MAINTENANCE_SECOND_BOOT_PROGRAM_IDS
        };

        if(IsMaintenanceBoot())
        {
            pm::SetMaintenanceBoot();

            for( auto id : MaintenanceSecondBootProgramIds )
            {
                ncm::ProgramId programId = { id };

                LaunchEachBootProgram(programId);
            }
        }
        else
        {
            for( auto id : SecondBootProgramIds )
            {
                ncm::ProgramId programId = { id };

                LaunchEachBootProgram(programId);
            }
        }
#endif

#ifdef NN_SIGLO_DEVMENUAPP_INSTALLER_PROGRAM_ID
        if(NeedsBootingDevMenuAppInstaller())
        {
            ncm::ProgramId programId = { NN_SIGLO_DEVMENUAPP_INSTALLER_PROGRAM_ID };

            LaunchEachBootProgram(programId);
        }
#endif
    }

}}  // namespace nn::boot2

