﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// boot プロセスで I2C を直で叩くための pcv ライブラリ
#include <nn/nn_Abort.h>

#include <nn/os.h>
#include <nn/dd.h>

#include <nn/pcv/pcv.h>

namespace
{

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOn32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp |= flags;
    *reg = temp;
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOff32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~flags;
    *reg = temp;
}

// Overwrite partly on specified register
// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void Modify32(volatile uint32_t* reg, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~mask;
    temp |= value;
    *reg = temp;
}

NN_FORCEINLINE uintptr_t GetVirtualAddress(nn::dd::PhysicalAddress physicalAddress, size_t addressSize) NN_NOEXCEPT
{
    uintptr_t virtualAddress = nn::dd::QueryIoMappingAddress(physicalAddress, addressSize);
    NN_ABORT_UNLESS(virtualAddress != 0, "I/O registers for 0x%llx are not mapped. "
        "Make sure the capability setting is properly set for this process.\n", physicalAddress);
    return virtualAddress;
}

/**
 * @brief CLK_RST の、バスに対応するレジスタアドレスおよびビットシフトを保持するための構造体(I2C からコピー)
 */
struct ClkRstRegInfo
{
public:
    void Set(int busIdx) NN_NOEXCEPT
    {
        /**
         * @brief Clock / Reset レジスタのマッピング対象物理アドレス ([Ref1] 2.0)
         */
        static const nn::dd::PhysicalAddress   ClkRstPhysicalAddress  = 0x060006000ull;
        static const size_t                    ClkRstAddressSize      = 0x1000;

        // Clock source register <=> CLK_RST_CONTROLLER_CLK_SOURCE_I2C# ([Ref1] 5.7.52(I2C1), 5.7.68(I2C2), 5.7.73(I2C3), 5.7.126(I2C4), 5.7.53(I2C5), 5.7.248(I2C6), )
        static const uintptr_t ClockSourceRegOffsetTable[] =
        {
            0x0124, // I2C1
            0x0198, // I2C2
            0x01b8, // I2C3
            0x03c4, // I2C4
            0x0128, // I2C5
            0x065c  // I2C6
        };

        // Clock enable register <=> CLK_RST_CONTROLLER_CLK_OUT_ENB_{L,H,U} ([Ref1] 5.7.7 - 5.7.9)
        static const uintptr_t ClockEnableRegOffsetTable[] =
        {
            0x0010, // I2C1
            0x0014, // I2C2
            0x0018, // I2C3
            0x0360, // I2C4
            0x0014, // I2C5
            0x0280  // I2C6
        };
        static const int ClockEnableBitShiftTable[] =
        {
            12,     // I2C1
            22,     // I2C2
            3,      // I2C3
            7,      // I2C4
            15,     // I2C5
            6       // I2C6
        };

        // Reset register <=> CLK_RST_CONTROLLER_RST_DEVICES_{L,H,U} ([Ref1] 5.7.4 - 5.7.6)
        static const uintptr_t ResetRegOffsetTable[] =
        {
            0x0004, // I2C1
            0x0008, // I2C2
            0x000c, // I2C3
            0x0358, // I2C4
            0x0008, // I2C5
            0x028c  // I2C6
        };
        static const int ResetBitShiftTable[] =
        {
            12,     // I2C1
            22,     // I2C2
            3,      // I2C3
            7,      // I2C4
            15,     // I2C5
            6       // I2C6
        };

        NN_SDK_ASSERT(0 <= busIdx && busIdx < static_cast<int>( sizeof(ClockSourceRegOffsetTable) / sizeof(ClockSourceRegOffsetTable[0]) ) );
        // IO マッピングを取得する
        uintptr_t baseAddr  = GetVirtualAddress(ClkRstPhysicalAddress,  ClkRstAddressSize);  // CLK_RST

        pClockSource       = reinterpret_cast<volatile uint32_t*>(baseAddr + ClockSourceRegOffsetTable[busIdx]);
        pClockEnable       = reinterpret_cast<volatile uint32_t*>(baseAddr + ClockEnableRegOffsetTable[busIdx]);
        clockEnableBitMask = (1 << ClockEnableBitShiftTable[busIdx]);
        pReset             = reinterpret_cast<volatile uint32_t*>(baseAddr + ResetRegOffsetTable[busIdx]);
        resetBitMask       = (1 << ResetBitShiftTable[busIdx]);
    }

public:
    volatile uint32_t*  pClockSource;
    volatile uint32_t*  pClockEnable;
    uint32_t            clockEnableBitMask;
    volatile uint32_t*  pReset;
    uint32_t            resetBitMask;
};

int GetBusIdx(nn::pcv::Module moduleId)
{
    switch(moduleId)
    {
    case nn::pcv::Module_I2c1:
        return 0;

    case nn::pcv::Module_I2c2:
        return 1;

    case nn::pcv::Module_I2c3:
        return 2;

    case nn::pcv::Module_I2c4:
        return 3;

    case nn::pcv::Module_I2c5:
        return 4;

    case nn::pcv::Module_I2c6:
        return 5;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

}

namespace nn { namespace pcv {

void Initialize() NN_NOEXCEPT
{
}

void Finalize() NN_NOEXCEPT
{
}


Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    ClkRstRegInfo regInfo;

    // boot で使うのは今のところ I2C1 のみ
    NN_SDK_REQUIRES_EQUAL(moduleId, nn::pcv::Module_I2c1);

    // ModuleId から BusIdx を逆算し、レジスタ情報をセット
    regInfo.Set(GetBusIdx(moduleId));

    // 以下 I2C のソースからコピペ

    // 2. Enable clock
    // [Ref1] p.2189 : The clock enable must also be given to the I2C controller, before any of registers are written.
    SetBitOn32  (regInfo.pClockEnable, regInfo.clockEnableBitMask);

    // 3. Change Clock divisor
    Modify32  ( regInfo.pClockSource, (0x04 << 0),   (0xFF) );      // [5.7.68] I2C#_CLK_DIVISOR = 0x04

    // 4. Wait 1 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );

    // 5. Change the clock source
    Modify32  ( regInfo.pClockSource, (0 << 29),     (0x7 << 29) ); // [5.7.68] CLK_M = PLLP_OUT0

    // 6. Wait 2 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(2) );

    return nn::ResultSuccess();
}

Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    // TORIAEZU : 何もしない (リンクエラー対策)。実際に使うときには内容を書くこと
    return nn::ResultSuccess();
}

Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    ClkRstRegInfo regInfo;

    // boot で使うのは今のところ I2C1 のみ
    NN_SDK_REQUIRES_EQUAL(moduleId, nn::pcv::Module_I2c1);

    // ModuleId から BusIdx を逆算し、レジスタ情報をセット
    regInfo.Set(GetBusIdx(moduleId));

    // 以下 I2C のソースからコピペ
    if(asserted)
    {
        SetBitOn32  (regInfo.pReset, regInfo.resetBitMask);
    }
    else
    {
        SetBitOff32  (regInfo.pReset, regInfo.resetBitMask);
    }

    return nn::ResultSuccess();
}

Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT
{
    // 何もしない

    return nn::ResultSuccess();
}

Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT
{
    // 何もしない

    return nn::ResultSuccess();
}

}} // namespace nn::pcv
