﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/psm/driver/detail/psm_BatteryChargePercentage.h>
#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "boot_SplAccessor.h"
#include "boot_BatteryChargeChecker.h"
#include "boot_BootReason.h"
#include "boot_ChargerDriver.h"
#include "boot_FuelgaugeDriver.h"
#include "boot_PmicDriver.h"
#include "boot_Display.h"
#include "boot_DisplayImageBinary.h"

namespace nn { namespace boot {

namespace {

const auto BatteryLevelThresholdForBoot = 3;
const auto BatteryLevelThresholdForFullCharge = 99.0f;

const auto BatteryVoltageThresholdConnected = 4000;
const auto BatteryVoltageThresholdDisconnected = 3650;

enum class BatteryCheckResult
{
    Success,
    NeedShutdown,
    NeedReboot,
};

// 電池温度と電池電圧を取得し、充電電流上限、充電電圧上限、充電可否を設定する
void UpdateCharger(
    PmicDriver* pPmicDriver,
    ChargerDriver* pChargerDriver,
    FuelgaugeDriver* pFuelgaugeDriver) NN_NOEXCEPT
{
    // 電池温度の取得
    double temperature;
    auto result = pFuelgaugeDriver->GetTemperature(&temperature);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] GetTemperature failure. (%08x)\n", result);
        pPmicDriver->ShutdownSystem();
    }

    // 電池電圧の取得
    uint32_t batteryVoltage;
    result = pFuelgaugeDriver->GetAverageVCell(&batteryVoltage);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] GetAverageVCell failure. (%08x)\n", result);
        pPmicDriver->ShutdownSystem();
    }

    // 充電可否、充電電圧上限の決定
    auto enableCharge = true;
    static decltype(enableCharge) s_LastEnableCharge = false; // 変更時のみログを出すようにする用。初回は必ずログ出すように反転。

    auto chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltDefault;
    static decltype(chargeVoltageLimit) s_LastChargeVoltageLimit = 0; // 変更時のみログを出すようにする用。初回は必ずログ出すように 0。

    if (temperature < nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooLow)
    {
        enableCharge = false;
        chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltDefault;
    }
    else if ((nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh <= temperature)
        && (temperature < nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh))
    {
        if (batteryVoltage < nn::psm::driver::detail::AllowChargeOnHighTemperatureThresholdMilliVoltage)
        {
            enableCharge = true;
            chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltHighTemperature;
        }
        else
        {
            enableCharge = false;
            chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltDefault;
        }
    }
    else if (nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh <= temperature)
    {
        if (batteryVoltage < nn::psm::driver::detail::AllowChargeOnHighTemperatureThresholdMilliVoltage)
        {
            enableCharge = false;
            chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltHighTemperature;
        }
        else
        {
            enableCharge = false;
            chargeVoltageLimit = nn::psm::driver::detail::ChargeVoltageLimitMilliVoltDefault;
        }
    }

    // 充電電流上限の決定
    auto fastChargeCurrentLimit = nn::psm::driver::detail::FastChargeCurrentLimitMilliAmpereAwakeStateDefault;
    static decltype(fastChargeCurrentLimit) s_LastFastChargeCurrentLimit = 0; // 変更時のみログを出すようにする用。初回は必ずログ出すように 0。
    if (temperature < nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow)
    {
        fastChargeCurrentLimit = std::min(nn::psm::driver::detail::FastChargeCurrentLimitMilliAmpereLowTemperature, fastChargeCurrentLimit);
    }
    if (batteryVoltage < nn::psm::driver::detail::DisallowFastSpeedChargeThresholdMilliVoltage)
    {
        fastChargeCurrentLimit = std::min(nn::psm::driver::detail::FastChargeCurrentLimitMilliAmpereAwakeStateLow, fastChargeCurrentLimit);
    }

    // 充電可否の設定
    result = pChargerDriver->SetChargeEnabled(enableCharge);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] SetChargeEnabled failure. (%08x)\n", result);
        pPmicDriver->ShutdownSystem();
    }
    if (s_LastEnableCharge != enableCharge) // 変更時のみログ表示
    {
        NN_SDK_LOG("[boot] EnableCharge changed to: %s.\n", enableCharge? "true": "false");
    }
    s_LastEnableCharge = enableCharge;

    // 充電電流上限の設定
    result = pChargerDriver->SetFastChargeCurrentLimit(fastChargeCurrentLimit);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] SetFastChargeCurrentLimit failure. (%08x)\n", result);
        pPmicDriver->ShutdownSystem();
    }
    if (s_LastFastChargeCurrentLimit != fastChargeCurrentLimit) // 変更時のみログ表示
    {
        NN_SDK_LOG("[boot] FastChargeCurrentLimit changed to: %d mA.\n", fastChargeCurrentLimit);
    }
    s_LastFastChargeCurrentLimit = fastChargeCurrentLimit;

    // 充電電圧上限の設定
    result = pChargerDriver->SetChargeVoltageLimit(chargeVoltageLimit);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] SetChargeVoltageLimit failure. (%08x)\n", result);
        pPmicDriver->ShutdownSystem();
    }
    if (s_LastChargeVoltageLimit != chargeVoltageLimit) // 変更時のみログ表示
    {
        NN_SDK_LOG("[boot] ChargeVoltageLimit changed to: %d mV.\n", chargeVoltageLimit);
    }
    s_LastChargeVoltageLimit = chargeVoltageLimit;
}

void PrintBatteryCharge(double batteryCharge) NN_NOEXCEPT
{
    if (batteryCharge >= BatteryLevelThresholdForBoot)
    {
        NN_SDK_LOG(
            "[boot] Battery charge is %.2f %% (>= %d %%).\n",
            std::floor(batteryCharge * 100) / 100,
            BatteryLevelThresholdForBoot);
    }
    else
    {
        NN_SDK_LOG(
            "[boot] Battery charge is %.2f %% (< %d %%).\n",
            std::floor(batteryCharge * 100) / 100,
            BatteryLevelThresholdForBoot);
    }
}

void PrintBatteryVoltage(uint32_t batteryVoltage, const uint32_t BatteryVoltageThreshold) NN_NOEXCEPT
{
    if (batteryVoltage >= BatteryVoltageThreshold)
    {
        NN_SDK_LOG(
            "[boot] Battery voltage is %d.%03d V (>= %d.%03d V).\n",
            batteryVoltage / 1000, batteryVoltage % 1000,
            BatteryVoltageThreshold / 1000, BatteryVoltageThreshold % 1000);
    }
    else
    {
        NN_SDK_LOG(
            "[boot] Battery voltage is %d.%03d V (< %d.%03d V).\n",
            batteryVoltage / 1000, batteryVoltage % 1000,
            BatteryVoltageThreshold / 1000, BatteryVoltageThreshold % 1000);
    }
}

void ShowLowBattery() NN_NOEXCEPT
{
    const auto LowBatterySignageSpan = nn::TimeSpan::FromSeconds(5);

    NN_SDK_LOG("[boot] %s.\n", __FUNCTION__);
    nn::boot::SetupDisplay();
    nn::boot::ShowDisplay(LowBatteryPointX, LowBatteryPointY, LowBatteryWidth, LowBatteryHeight, reinterpret_cast<const Bit32*>(LowBatteryData), LowBatteryDataSize);
    nn::os::SleepThread(LowBatterySignageSpan);
    nn::boot::ShutdownDisplay();
}

void FillInBlack(Bit32 *pBuffer, int bufferWidth, int bufferHeight, int width, int height, int offsetX, int offsetY) NN_NOEXCEPT
{
    if (width + offsetX > bufferWidth)
    {
        return;
    }
    if (height + offsetY > bufferHeight)
    {
        return;
    }

    Bit32 *p = pBuffer + offsetY * bufferWidth + offsetX;
    for (int h = 0; h < height; h++)
    {
        for (int w = 0; w < width; w++)
        {
            *(p + w) = 0xff000000;
        }
        p += bufferWidth;
    }
}

// Programs/Eris/Sources/Libraries/omm/srv/omm_AwakeModeManager.cpp の処理をコピー
void StartShowCharging(int batteryPercentage, bool waitForChargingSignageSpan) NN_NOEXCEPT
{
    const int BatteryStartX = 6;
    const int BatteryStartY = 6;
    const int BatteryWidth = 36;
    const int BatteryFillHeight = 16;

    auto isChargingRed = batteryPercentage <= 15;

    const int pointX = (isChargingRed) ? ChargingRedPointX : ChargingPointX;
    const int pointY = (isChargingRed) ? ChargingRedPointY : ChargingPointY;
    const int width = (isChargingRed) ? ChargingRedWidth : ChargingWidth;
    const int height = (isChargingRed) ? ChargingRedHeight : ChargingHeight;
    const size_t dataSize = (isChargingRed) ? ChargingRedDataSize : ChargingDataSize;
    const Bit8* pData = (isChargingRed) ? ChargingRedData : ChargingData;

    Bit32 buffer[dataSize / sizeof(Bit32)];
    std::memcpy(buffer, pData, dataSize);

    // OverDispTest におけるバッテリー残量表示スケールの近似式 (100% のとき 1.00、1% のとき 0.05)
    auto scale = 0.0096 * batteryPercentage + 0.0404;
    auto batteryFillWidth = static_cast<int>(BatteryWidth * (1 - scale) + 0.5);
    auto offsetX = BatteryStartX + BatteryWidth - batteryFillWidth;

    NN_SDK_LOG("[boot] fillWidth=%d, offsetX=%d\n", batteryFillWidth, offsetX);

    FillInBlack(buffer, width, height, batteryFillWidth, BatteryFillHeight, offsetX, BatteryStartY);

    nn::boot::SetupDisplay();
    nn::boot::ShowDisplay(
        pointX, pointY,
        width, height,
        buffer, sizeof(buffer));
    if (waitForChargingSignageSpan)
    {
        const auto ChargingSignageSpan = nn::TimeSpan::FromSeconds(2);
        nn::os::SleepThread(ChargingSignageSpan);
    }
}

void StartShowCharging(int batteryPercentage) NN_NOEXCEPT
{
    NN_SDK_LOG("[boot] %s.\n", __FUNCTION__);
    StartShowCharging(batteryPercentage, true);
}

void StartShowLowBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_LOG("[boot] %s.\n", __FUNCTION__);
    StartShowCharging(1, false);
}

void EndShowCharging() NN_NOEXCEPT
{
    NN_SDK_LOG("[boot] %s.\n", __FUNCTION__);
    nn::boot::ShutdownDisplay();
}

bool IsBatteryVoltageSufficient(uint32_t batteryVoltage, bool isAcOk) NN_NOEXCEPT
{
    uint32_t batteryVoltageThreshold = BatteryVoltageThresholdConnected;

    if (isAcOk)
    {
        batteryVoltageThreshold = BatteryVoltageThresholdConnected;
    }
    else
    {
        batteryVoltageThreshold = BatteryVoltageThresholdDisconnected;
    }

    if (batteryVoltage >= batteryVoltageThreshold)
    {
        // 起動許可
        PrintBatteryVoltage(batteryVoltage, batteryVoltageThreshold);
        return true;
    }
    else
    {
        // 定期的に電池電圧を表示する
        const auto progressInterval = nn::TimeSpan::FromMilliSeconds(10000);
        static auto s_LastProgressTime = nn::TimeSpan(0);
        auto currentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        if (currentTime - s_LastProgressTime >= progressInterval)
        {
            PrintBatteryVoltage(batteryVoltage, batteryVoltageThreshold);
            s_LastProgressTime = currentTime;
        }
        return false;
    }
}

BatteryCheckResult InvokeBatteryChargeCheckLoop(
    PmicDriver* pPmicDriver,
    ChargerDriver* pChargerDriver,
    FuelgaugeDriver* pFuelgaugeDriver,
    bool rebootOnPowerButtonPress,
    bool returnOnEnoughBattery,
    bool shutdownOnFullBattery,
    bool showDisplay,
    bool showChargingDisplay) NN_NOEXCEPT
{
    const uint8_t onOffIrqEn0Mask  = 0x08;
    const auto batteryChargeCheckInterval = nn::TimeSpan::FromMilliSeconds(20);
    const auto progressInterval = nn::TimeSpan::FromMilliSeconds(10000);
    auto lastProgressTime = nn::TimeSpan(0);
    auto isShowingChargingSign = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (isShowingChargingSign)
        {
            EndShowCharging();
            isShowingChargingSign = false;
        }
    };

    if (showChargingDisplay)
    {
        double batteryChargeRaw;
        auto result = pFuelgaugeDriver->GetSocRep(&batteryChargeRaw);
        if (result.IsSuccess())
        {
            auto batteryCharge = nn::psm::driver::detail::ConvertBatteryChargePercentage(batteryChargeRaw);
            StartShowCharging(batteryCharge);
            isShowingChargingSign = true;
        }
        else
        {
            NN_SDK_LOG("[boot] GetSocRep() failure. (%08x)\n", result);
            return BatteryCheckResult::NeedShutdown;
        }
    }

    while (true)
    {
// 擬似オフ中にはいくら充電しても満充電ステータスにならないので不採用 (SIGLO-43076)
#if 0
        // 満充電になったらシャットダウンするモード
        if (shutdownOnFullBattery)
        {
            nne::bq2419x::ChargeStatusType chargeStatus;
            auto result = pChargerDriver->GetChargeStatus(&chargeStatus);
            if (result.IsSuccess())
            {
                if (chargeStatus == nne::bq2419x::ChargeStatusType_ChargeTerminationDone)
                {
                    // 満充電と判定されたのでシャットダウンする
                    NN_SDK_LOG("[boot] Charge termination done. Shutdown now.\n");
                    return BatteryCheckResult::NeedShutdown;
                }
            }
            else
            {
                NN_SDK_LOG("[boot] GetChargeStatus() failure. (%08x)\n", result);
                return BatteryCheckResult::NeedShutdown;
            }
        }
#endif

        double batteryCharge;
        auto result = pFuelgaugeDriver->GetSocRep(&batteryCharge);
        if (result.IsSuccess())
        {
            if (returnOnEnoughBattery && batteryCharge >= BatteryLevelThresholdForBoot)
            {
                // 起動許可
                PrintBatteryCharge(batteryCharge);
                return BatteryCheckResult::Success;
            }
            else if (shutdownOnFullBattery && batteryCharge >= BatteryLevelThresholdForFullCharge)
            {
                // 満充電でシャットダウン
                NN_SDK_LOG("[boot] Charge termination done. Shutdown now.\n");
                return BatteryCheckResult::NeedShutdown;
            }
            else
            {
                // 定期的に電池残量を表示する
                auto currentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
                if (currentTime - lastProgressTime >= progressInterval)
                {
                    PrintBatteryCharge(batteryCharge);
                    lastProgressTime = currentTime;
                }
            }
        }
        else
        {
            NN_SDK_LOG("[boot] GetSocRep() failure. (%08x)\n", result);
            return BatteryCheckResult::NeedShutdown;
        }

        bool isAcOk;
        result = pPmicDriver->GetAcOk(&isAcOk);
        if (!(result.IsSuccess()))
        {
            NN_SDK_LOG("[boot] GetAcOk() failure. (%08x)\n", result);
            return BatteryCheckResult::NeedShutdown;
        }

        // 十分に電池電圧が高ければ起動を許可する

        uint32_t batteryVoltage;
        result = pFuelgaugeDriver->GetAverageVCell(&batteryVoltage);
        if (result.IsSuccess())
        {
            if (returnOnEnoughBattery && IsBatteryVoltageSufficient(batteryVoltage, isAcOk))
            {
                return BatteryCheckResult::Success;
            }
        }
        else
        {
            NN_SDK_LOG("[boot] GetAverageVCell() failure. (%08x)\n", result);
            return BatteryCheckResult::NeedShutdown;
        }

        // --- 以下、電池残量が閾値未満のときにのみ通るパス（擬似オフ状態は常時こちらに来る） ---

        // 充電器が接続されていない場合、まだ充電中画面を出してなければ要充電画面を出してからシャットダウンする

        if (!isAcOk)
        {
            PrintBatteryCharge(batteryCharge);
            NN_SDK_LOG("[boot] Charger disconnected while battery charge is too low.\n");
            // 充電中画面をまだ表示していないとき（最初のチェック時）に限り要充電画面を出す
            if (showDisplay && !isShowingChargingSign)
            {
                ShowLowBattery();
            }
            return BatteryCheckResult::NeedShutdown;
        }

        // （給電起動時、擬似オフ状態のみ）
        // PowerButton が押された場合にはリブートする
        // ただし、起動時から押されたままの場合にはリブートしない
        if (rebootOnPowerButtonPress)
        {
            uint8_t onOffIrq;

            // boot プロセス起床時に ONOFFIRQ は Clear On Read されている前提
            // Clear On Read 後に再度 PowerButton が押されたかを取得する
            result = pPmicDriver->GetOnOffIrq(&onOffIrq);
            if (result.IsSuccess())
            {
                if ((onOffIrq & onOffIrqEn0Mask) == onOffIrqEn0Mask)
                {
                    NN_SDK_LOG("[boot] Power button pressed.\n");
                    return BatteryCheckResult::NeedReboot;
                }
            }
            else
            {
                NN_SDK_LOG("[boot] GetPowerButtonPressed() failure. (%08x)\n", result);
                return BatteryCheckResult::NeedShutdown;
            }
        }

        // 充電待ちで待機し始めることが決まったので、（まだ表示してなければ）充電中画面を表示
        if (showDisplay && !isShowingChargingSign)
        {
            StartShowLowBatteryCharging();
            isShowingChargingSign = true;
        }

        nn::os::SleepThread(batteryChargeCheckInterval);
        UpdateCharger(pPmicDriver, pChargerDriver, pFuelgaugeDriver);
    }

    // NEVER REACHED HERE
} // NOLINT(impl/function_size)

} // namespace

void CheckBatteryCharge() NN_NOEXCEPT
{
    ChargerDriver chargerDriver;
    FuelgaugeDriver fuelgaugeDriver;
    PmicDriver pmicDriver;

    nn::Result result;

    result = fuelgaugeDriver.InitializeParameter();
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] Fuelgauge initialization failure. (%08x)\n", result);
        pmicDriver.ShutdownSystem();
    }

    // 電池が装着されていないと異常温度が検出される
    bool isBatteryRemoved;
    result = fuelgaugeDriver.IsBatteryRemoved(&isBatteryRemoved);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] IsBatteryRemoved failure. (%08x)\n", result);
        pmicDriver.ShutdownSystem();
    }
    if (isBatteryRemoved)
    {
        NN_SDK_LOG("[boot] Battery removed.\n");
        pmicDriver.ShutdownSystem();
    }

    auto bootReason = nn::boot::GetBootReason();

    // 擬似オフの場合は InputCurrentLimit を初期化しない
    if (bootReason == nn::spl::BootReason_RtcAlarm2)
    {
        result = chargerDriver.Initialize(false);
    }
    else
    {
        result = chargerDriver.Initialize(true);
    }

    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] Initialize failure. (%08x)\n", result);
        pmicDriver.ShutdownSystem();
    }

    UpdateCharger(&pmicDriver, &chargerDriver, &fuelgaugeDriver);

    BatteryCheckResult batteryCheckResult;
    if (bootReason == nn::spl::BootReason_AcOk)
    {
        // 給電起動
        NN_SDK_LOG("[boot] ACOK boot\n");
        batteryCheckResult = InvokeBatteryChargeCheckLoop(
            &pmicDriver,
            &chargerDriver,
            &fuelgaugeDriver,
            true,   // 充電待ち中の電源ボタン押しでリブートする
            true,   // 残量が十分になったら起動
            false,  // 満充電でもシャットダウンしない
            true,   // 充電中・要充電の画面表示あり
            true);  // 充電中の画面を必ず表示
    }
    else if (bootReason == nn::spl::BootReason_RtcAlarm2)
    {
        // 擬似オフ (QuasiOff)
        NN_SDK_LOG("[boot] RTC Alarm 2 boot (QuasiOff)\n");
        batteryCheckResult = InvokeBatteryChargeCheckLoop(
            &pmicDriver,
            &chargerDriver,
            &fuelgaugeDriver,
            true,   // 充電待ち中の電源ボタン押しでリブートする
            false,  // 残量が十分になっても滞留（永久に充電）
            true,   // 満充電でシャットダウンする
            false,  // 画面表示なし
            false); // 充電中の画面表示なし
    }
    else
    {
        NN_SDK_LOG("[boot] Normal boot\n");
        batteryCheckResult = InvokeBatteryChargeCheckLoop(
            &pmicDriver,
            &chargerDriver,
            &fuelgaugeDriver,
            false,  // 充電待ち中の電源ボタン押しは無視
            true,   // 残量が十分になったら起動
            false,  // 満充電でもシャットダウンしない
            true,   // 充電中・要充電の画面表示あり
            false); // 充電中の画面表示なし
    }

    switch (batteryCheckResult)
    {
        case BatteryCheckResult::Success:
        {
            return;
        }
        case BatteryCheckResult::NeedShutdown:
        {
            pmicDriver.ShutdownSystem();
        }
        case BatteryCheckResult::NeedReboot:
        {
            pmicDriver.RebootSystem();
        }
    } // NOLINT(style/switch_default)

    NN_ABORT("NEVER REACHED HERE");
} // NOLINT(impl/function_size)

NN_NORETURN void RebootSystem() NN_NOEXCEPT
{
    PmicDriver pmicDriver;
    pmicDriver.RebootSystem();
}

NN_NORETURN void ShutdownSystem() NN_NOEXCEPT
{
    PmicDriver pmicDriver;
    pmicDriver.ShutdownSystem();
}

}} // namespace nn::boot
