﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "bluetooth_InternalTypes.h"

#ifdef __cplusplus
#include <nn/bluetooth/bluetooth_Types.h>
#include "bsa_api.h"
#include "xml_param.h"
#include "bt_types.h"

extern "C"
{
#endif // ifdef __cplusplus

#define BLE_CLIENT_MAX BSA_BLE_CLIENT_MAX
#define BLE_SERVER_MAX BSA_BLE_SERVER_MAX

typedef struct
{
    tBT_UUID        attr_UUID;
    uint16_t        service_handle;
    uint16_t        attr_handle;
    uint8_t         attr_type;
    uint8_t         prop;
    uint16_t        length;
    uint8_t         value[BSA_BLE_MAX_ATTR_LEN];
    bool            is_pri;
    bool            wait_flag;
} tAPP_BLE_ATTRIBUTE;

typedef struct
{
    tBT_UUID    service_uuid;
    tBT_UUID    char_uuid;
    uint8_t     service_inst_id;
    uint8_t     char_inst_id;
    bool        is_primary_service;
    bool        is_registered;
} tAPP_BLE_REGISTERED_NOTIFICATION;

typedef struct
{
    bool                                enabled;
    tBSA_BLE_IF                         client_if;
    uint16_t                            bsa_conn_id;
    uint32_t                            conn_id;
    BD_ADDR                             server_addr;
    tAPP_BLE_REGISTERED_NOTIFICATION    registered_notification[BTA_GATTC_NOTIF_REG_MAX];
    bool                                write_pending;
    bool                                read_pending;
    bool                                congested;
} tAPP_BLE_CLIENT;

typedef struct
{
    bool                enabled;
    tBSA_BLE_IF         server_if;
    uint16_t            conn_id;
    tAPP_BLE_ATTRIBUTE  attr[BSA_BLE_ATTRIBUTE_MAX];
} tAPP_BLE_SERVER;

typedef struct
{
    tAPP_BLE_CLIENT ble_client[BLE_CLIENT_MAX];
    tAPP_BLE_SERVER ble_server[BLE_SERVER_MAX];
} tAPP_BLE_CB;

struct GattDataPathFilter
{
    GattAttributeUuid   uuid;
    uint8_t             type;
    bool                inUse;
};

/*
 * Global Variables
 */
extern tAPP_BLE_CB app_ble_cb;

extern GattDataPathFilter g_GattDataPathFilter[GATT_DATA_PATH_FILTER_NUM_MAX];


#ifdef __cplusplus
}
#endif // ifdef __cplusplus

/*******************************************************************************
**
** Function:        BtHalLeInitialize
**
** Description:     Clear BLE client/server application database.
**
** Parameters:      None
**
** Returns:         void
**
*******************************************************************************/
void BtHalLeInitialize();

/*******************************************************************************
**
** Function:        BtHalLeEnable
**
** Description:     Enable BLE features. Deregister all the client/server application.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeEnable();

/*******************************************************************************
**
** Function:        BtHalLeDisable
**
** Description:     Disable BLE features. Deregister all the client/server application.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeDisable();

/*******************************************************************************
**
** Function:        BtHalLeSetVisibility
**
** Description:     Start/Stop BLE advertising.
**
** Parameters:      [in] bool discoverable : Start/Stop BLE advertising as discoverable device
** Parameters:      [in] bool connectable : Start/Stop BLE advertising as connectable device
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetVisibility(bool discoverable, bool connectable);

/*******************************************************************************
**
** Function:        BtHalLeSetConnectionParameter
**
** Description:     Start BLE connection parameter update for a connected device with prefferred parameters.
**
** Parameters:      [in] tBSA_DM_BLE_CONN_PARAM* pParam : prefferred connection parameters.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetConnectionParameter(const tBSA_DM_BLE_CONN_PARAM* pParam);

/*******************************************************************************
**
** Function:        BtHalLeSetDefaultConnectionParameter
**
** Description:     Set default BLE connection parameter for incoming connection. Actual update will start when connection established.
**
** Parameters:      [in] tBSA_DM_BLE_CONN_PARAM* pParam : prefferred connection parameters.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetDefaultConnectionParameter(const tBSA_DM_BLE_CONN_PARAM *pParam);

/*******************************************************************************
**
** Function:        BtHalLeSetAdvertiseData
**
** Description:     Set BLE advertising packet data.
**
** Parameters:      [in] tBSA_DM_BLE_ADV_CONFIG* pConfig : contens of advertising data.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetAdvertiseData(const tBSA_DM_BLE_ADV_CONFIG* pConfig);

/*******************************************************************************
**
** Function:        BtHalLeSetAdvertiseParam
**
** Description:     Set BLE advertising parameters.
**
** Parameters:      [in] tBSA_DM_BLE_ADV_PARAM* pParam : parameters for BLE advertising.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetAdvertiseParam(const tBSA_DM_BLE_ADV_PARAM* pParam);

/*******************************************************************************
**
** Function:        BtHalLeSetScanParameter
**
** Description:     Set BLE scan parameters.
**
** Parameters:      [in] uint16_t scanInterval : Scan interval. Actual interval would be (scanInterval * 0.625) msec
** Parameters:      [in] uint16_t scanWindow : Scan Window. Actual window would be (scanWindow * 0.625) msec
**
** Conditions:      scanWindow <= scanInterval
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetScanParameter(uint16_t scanInterval, uint16_t scanWindow);

