﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>

#include "bluetooth_hal_utils.h"
#include "bluetooth_hal_hh.h"
#include "fs_wrapper.h"
#include "bluetooth_queue.h"
#include <nn/bluetooth/bluetooth_TypesHal.h>
#include "bluetooth_InternalTypes.h"
#include "bluetooth.h"
#include "bluetooth_hid.h"

#include "bsa_tm_api.h"

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/
#define CONNECT_AFTER_PAIRING

/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
 *
 *    File scope functions/symbols defined for this file
 *    -- Function Prototypes, File Scope Data --
 *
 *---------------------------------------------------------------------------*/
static tUIPC_CH_ID app_hh_uipc_chan;


// ////////////////////////////////////////// HAL HID INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL HID INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL HID INTERFACE ////////////////////////////////////////// //
/*******************************************************************************
 **
 ** Function        BtHalCallbackReportData
 **
 ** Description     report data callback
 **
 ** Parameters      p_msg - pointer to the report data
 **
 ** Returns         void
 **
 *******************************************************************************/
void BtHalCallbackReportData(BT_HDR *p_msg)
{
    tBSA_HH_UIPC_REPORT *p_uipc_report = (tBSA_HH_UIPC_REPORT *)p_msg;

    if (bonding_connect)
    {
        return;
    }

    if (p_uipc_report!=NULL) {
        BluetoothHalInsertToQueue(
                p_uipc_report,
                sizeof(*p_uipc_report) - sizeof(p_uipc_report->report.report_data.data) + p_uipc_report->report.report_data.length,
                HAL_QUEUE_HID,
                CALLBACK_TYPE_HID_DATA);
    }

    GKI_freebuf(p_msg);
}

/*******************************************************************************
 **
 ** Function        BtHalBsaGetDscpinfo
 **
 ** Description     Function to connect DHCP Info of an HID device
 **
 ** Parameters
 **
 ** Returns         void
 **
 *******************************************************************************/
int BtHalBsaGetDscpinfo(uint8_t handle)
{
    int status;
    tBSA_HH_GET get_dscpinfo;

    BTHAL_BSA_DEBUG("Get HID DSCP Info");
    if (handle == BSA_HH_INVALID_HANDLE)
    {
        NN_SDK_LOG("[bluetooth] BSA_HhGet: Invalid handle\n");
        return -1;
    }
    BSA_HhGetInit(&get_dscpinfo);
    get_dscpinfo.request = BSA_HH_DSCP_REQ;
    get_dscpinfo.handle = handle;
    status = BSA_HhGet(&get_dscpinfo);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_HhGet: Unable to get DSCP Info status:%d\n", status);
        return -1;
    }

    return 0;
}
/*******************************************************************************
 **
 ** Function        BtHalCallbackConnectionState
 **
 ** Description     Connection callback
 **
 ** Parameters      event   - type of connection event
 **                 p_data  - connect data (e.g. bd_addr)
 **
 ** Returns         void
 **
 *******************************************************************************/
void BtHalCallbackConnectionState(tBSA_HH_EVT event, tBSA_HH_MSG *p_data)
{
    BTHAL_BSA_DEBUG("event=%d",event);

    switch (event)
    {
    case BSA_HH_OPEN_EVT:
        NN_SDK_LOG("[bluetooth] BSA_HH_OPEN_EVT: status:%d, handle:%d mode:%s (%d) "
                     "BdAddr :%02X-%02X-%02X-%02X-%02X-%02X\n",
                p_data->open.status, p_data->open.handle,
                p_data->open.mode==BTA_HH_PROTO_RPT_MODE?"Report":"Boot",
                p_data->open.mode,
                p_data->open.bd_addr[0], p_data->open.bd_addr[1], p_data->open.bd_addr[2],
                p_data->open.bd_addr[3], p_data->open.bd_addr[4], p_data->open.bd_addr[5]);

#ifndef CONNECT_AFTER_PAIRING
        if (!bonding_connect){
            /* only return this connection state when not bonding */
            BluetoothHalInsertToQueue(&p_data->open,
                    sizeof(p_data->open),
                    HAL_QUEUE_CB,
                    CALLBACK_TYPE_CONNECTION_STATE_CONNECTED);
        }else{
            /*
             * Disconnect after bonding has completed.
             * This is required for master/slaver role switch in superpiconet.
             */
            BluetoothHhStatus status = BTHH_OK;
            tBSA_HH_CLOSE hh_close_param;

            // TODO: move out of callback
            BSA_HhCloseInit(&hh_close_param);
            hh_close_param.handle = p_data->open.handle;
            status = BSA_HhClose(&hh_close_param);
            if (status != BSA_SUCCESS)
            {
                // FIXME: Do something
                NN_SDK_LOG("[bluetooth] %s: ERROR disconnecting\n",__func__);
            }
        }
#else
        BluetoothHalInsertToQueue(&p_data->open,
                sizeof(p_data->open),
                HAL_QUEUE_CB,
                CALLBACK_TYPE_CONNECTION_STATE_CONNECTED);

        bonding_connect = false;
#endif
        break;

    /* coverity[MISSING BREAK] False-positive: Unplug event is handled as a close event */
    case BSA_HH_VC_UNPLUG_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_VC_UNPLUG_EVT is handled like a close event, redirecting");
        /* NOT SUPPORTED BY THE HAL */
        break;

    case BSA_HH_CLOSE_EVT:
        NN_SDK_LOG("[bluetooth] BSA_HH_CLOSE_EVT: status:%d  handle:%d\n",
                p_data->close.status, p_data->close.handle);
#ifndef CONNECT_AFTER_PAIRING
        if (!bonding_connect) {
            BluetoothHalInsertToQueue(&p_data->close,
                    sizeof(p_data->close),
                    HAL_QUEUE_CB,
                    CALLBACK_TYPE_CONNECTION_STATE_DISCONNECTED);
        }
        bonding_connect = false;
#else
        BluetoothHalInsertToQueue(&p_data->close,
                sizeof(p_data->close),
                HAL_QUEUE_CB,
                CALLBACK_TYPE_CONNECTION_STATE_DISCONNECTED);
        bonding_connect = false;
#endif
        break;

    case BSA_HH_GET_DSCPINFO_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_GET_DSCPINFO_EVT: status:%d handle:%d",
                p_data->dscpinfo.status, p_data->dscpinfo.handle);
        BluetoothHalInsertToQueue(&p_data->dscpinfo,
                sizeof(p_data->dscpinfo),
                HAL_QUEUE_CB,
                CALLBACK_TYPE_HID_INFO);
        if (p_data->dscpinfo.status == BSA_SUCCESS)
        {
            scru_dump_hex(p_data->dscpinfo.dscpdata.data, NULL,
                                            p_data->dscpinfo.dscpdata.length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
        }
        break;

    case BSA_HH_GET_REPORT_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_GET_REPORT_EVT: status:%d handle:%d len:%d",
                p_data->get_report.status, p_data->get_report.handle,
                p_data->get_report.report.length);
        BluetoothHalInsertToQueue(&p_data->get_report,
                sizeof(p_data->get_report),
                HAL_QUEUE_HID,
                CALLBACK_TYPE_GET_REPORT);
        break;

    case BSA_HH_SET_REPORT_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_SET_REPORT_EVT: status:%d handle:%d",
                p_data->set_report.status, p_data->set_report.handle);
        BluetoothHalInsertToQueue(&p_data->set_report,
                sizeof(p_data->set_report),
                HAL_QUEUE_HID,
                CALLBACK_TYPE_SET_REPORT);
        break;

    case BSA_HH_GET_PROTO_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_GET_PROTO_EVT: status:%d handle:%d mode:%d",
                p_data->get_proto.status, p_data->get_proto.handle,
                p_data->get_proto.mode);
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_GET_PROTO_EVT\n", __func__);
        break;

    case BSA_HH_SET_PROTO_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_SET_PROTO_EVT: status:%d, handle:%d",
                p_data->set_proto.status, p_data->set_proto.handle);
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_SET_PROTO_EVT\n", __func__);
        break;

    case BSA_HH_GET_IDLE_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_GET_IDLE_EVT: status:%d, handle:%d, idle:%d",
                p_data->get_idle.status, p_data->get_idle.handle, p_data->get_idle.idle);
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_GET_IDLE_EVT\n", __func__);
        break;

    case BSA_HH_SET_IDLE_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_SET_IDLE_EVT: status:%d, handle:%d",
                p_data->set_idle.status, p_data->set_idle.handle);
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_SET_IDLE_EVT\n", __func__);
        break;

    case BSA_HH_MIP_START_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_MIP_START_EVT");
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_MIP_START_EVT\n", __func__);
        break;

    case BSA_HH_MIP_STOP_EVT:
        BTHAL_BSA_DEBUG("BSA_HH_MIP_STOP_EVT");
        NN_SDK_LOG("[bluetooth] %s: un-handled event: BSA_HH_MIP_STOP_EVT\n", __func__);
        break;

    default:
        NN_SDK_LOG("[bluetooth] %s: bad event:%d\n", __func__, event);
        break;
    }
} //NOLINT(impl/function_size)

/***********************************************************************************************
 **
 ** Function         BtHalConnect
 **
 ** Description      Connect a device after bonding
 **
 ** Parameters       pBdAddr - BD_ADDR of the device to which we are opening a connection with
 **
 ** Returns          BluetoothHhStatus
 **
 ***********************************************************************************************/
BluetoothHhStatus BtHalConnect(const Btbdaddr *pBdAddr)
{
     NN_SDK_LOG("[bluetooth] %s: API available but feature not supported\n",__func__);
     BluetoothHhStatus status = BTHH_OK;

#if 0
     /* Connect HID Device in Standard Report mode*/
     tBSA_HH_OPEN hh_open_param;

    BSA_HhOpenInit(&hh_open_param);
    copyBdAddr(hh_open_param.bd_addr, pBdAddr->address);

    hh_open_param.mode = BSA_HH_PROTO_RPT_MODE; // only report mode is currently supported

    hh_open_param.sec_mask = BTA_SEC_NONE;

    app_hh_cb.sec_mask_out = hh_open_param.sec_mask;

    status = BSA_HhOpen(&hh_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to disconnect %d\n", __func__, status);
    }
#endif
    return status;
}

/****************************************************************************************
 **
 ** Function        BtHalDisconnect
 **
 ** Description     Disconnect an open connection with the device specified in pBdAddr
 **
 ** Parameters      pBdAddr - BD_ADDR of the device being disconnected
 **
 ** Returns         BluetoothHhStatus
 **
 ****************************************************************************************/
BluetoothHhStatus BtHalDisconnect(const Btbdaddr *pBdAddr)
{
    /* Example how to disconnect HID Device */
    tAPP_XML_REM_DEVICE *p_hh_dev;

    BTHAL_BSA_DEBUG("Called");

    p_hh_dev = app_xml_find_dev_db(pBdAddr->address);
    if (p_hh_dev == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    tBSA_HH_CLOSE hh_close_param;
    BSA_HhCloseInit(&hh_close_param);
    hh_close_param.handle = p_hh_dev->handle;
    tBSA_STATUS status = BSA_HhClose(&hh_close_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to disconnect %d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalSetInfo
 **
 ** Description     Sets device info (e.g. vendor_id)
 **
 ** Parameters      pBdAddr - BD_ADDR of the target device
 **                 HidInfo - HID device info (e.g. vendor_id/product_id)
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetInfo()
{
    // FIXME: @implement
    BTHAL_BSA_DEBUG("Called");

    return BTHH_OK;
}

/*******************************************************************************
 **
 ** Function        BtHalGetReport
 **
 ** Description     Get the report type
 **
 ** Parameters      pBdAddr     - BD_ADDR of the target device
 **                 reportType  - report type (input/output/feature)
 **                 reportId    - report ID
 **                 bufferSize  - size of the report
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalGetReport(const Btbdaddr *pBdAddr,
        BluetoothHhReportType reportType,
        uint8_t reportId,
        int bufferSize)
{
    tAPP_XML_REM_DEVICE *p_hh_dev;

    BTHAL_BSA_DEBUG("Called");

    p_hh_dev = app_xml_find_dev_db(pBdAddr->address);
    if (p_hh_dev == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    tBSA_HH_GET hh_get_param;
    BSA_HhGetInit(&hh_get_param);

    //hh_get_param.request = BSA_HH_REPORT_REQ;
    hh_get_param.request = BSA_HH_REPORT_NO_SIZE_REQ;
    hh_get_param.handle = p_hh_dev->handle;
    hh_get_param.param.get_report.report_type = reportType;
    hh_get_param.param.get_report.report_id = reportId;

    tBSA_STATUS status = BSA_HhGet(&hh_get_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get report, status: %d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalSetReport
 **
 ** Description     Sets a report type
 **
 ** Parameters      pBdAddr     - BD_ADDR of the target device
 **                 reportType  - report type (input/output/feature)
 **                 report      - pointer to the report data
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetReport(const Btbdaddr *pBdAddr, BluetoothHhReportType reportType, const char* report)
{
    tBSA_HH_REPORT_DATA *reportData = (tBSA_HH_REPORT_DATA *)report;

    BTHAL_BSA_DEBUG("Called, size=%d", reportData->length);

    tAPP_XML_REM_DEVICE *p_hh_dev = app_xml_find_dev_db(pBdAddr->address);
    if (p_hh_dev == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    tBSA_HH_SET hh_set_param;
    BSA_HhSetInit(&hh_set_param);
    hh_set_param.request = BSA_HH_REPORT_REQ;
    hh_set_param.handle = p_hh_dev->handle;
    hh_set_param.param.set_report.report_type = reportType;
    hh_set_param.param.set_report.report.length = reportData->length;
    memset(hh_set_param.param.set_report.report.data, 0, sizeof(hh_set_param.param.set_report.report.data));
    memcpy(hh_set_param.param.set_report.report.data, reportData->data, reportData->length);

    tBSA_STATUS status = BSA_HhSet(&hh_set_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set report, status: %d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalSendData
 **
 ** Description     Send HID data to the device specified in pBdAddr
 **
 ** Parameters      pBdAddr - Target device
 **                 data    - pointer to the data being sent to the target device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSendData(const Btbdaddr *pBdAddr, const char* data)
{
    tBSA_HH_REPORT_DATA *reportData = (tBSA_HH_REPORT_DATA *)data;
    tAPP_XML_REM_DEVICE *p_hh_dev;

    if (bonding_connect)
    {
       NN_SDK_LOG("[bluetooth]: %s ERROR cannot send HID data while bonding\n",__func__);
       return BTHH_HS_HID_NOT_READY;
    }

    p_hh_dev = app_xml_find_dev_db(pBdAddr->address);
    if (p_hh_dev == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    BTHAL_BSA_DEBUG("rptId=0x%x length=%d handle=%d", reportData->data[0],reportData->length,p_hh_dev->handle);

    tBSA_HH_SEND_DATA hh_sent_data_param;
    BSA_HhSendDataInit(&hh_sent_data_param);
    hh_sent_data_param.handle = p_hh_dev->handle;
    memset(hh_sent_data_param.data.data, 0, sizeof(hh_sent_data_param.data.data));
    memcpy(hh_sent_data_param.data.data,reportData->data,reportData->length);

    hh_sent_data_param.data.length = reportData->length;

    tBSA_STATUS status = BSA_HhSendData(&hh_sent_data_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: BtHalSendData unable to send data to RC:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalAddPairedDevice
 **
 ** Description     Adds paired device settings
 **
 ** Parameters      nn::settings::system::BluetoothDevicesSettings
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalAddPairedDevice(const nn::settings::system::BluetoothDevicesSettings* pDevice)
{
    if (pDevice == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: NULL param\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    BTHAL_IF_DEBUG("[bluetooth] %s: %s", __FUNCTION__, toString(pDevice->bd_addr, 6) );

    if (app_xml_find_dev_db(pDevice->bd_addr) != NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Re-pairing for device %02x:%02x:%02x:%02x:%02x:%02x\n", __func__,
                pDevice->bd_addr[0], pDevice->bd_addr[1], pDevice->bd_addr[2],
                pDevice->bd_addr[3], pDevice->bd_addr[4], pDevice->bd_addr[5]);
        BtHalRemoveBond((const Btbdaddr*)pDevice->bd_addr);
    }

    int result = app_xml_add_db(pDevice);
    //app_xml_display_devices();

    if(result != 0)
    {
        NN_SDK_LOG("[bluetooth] app_xml_add_db failed status:%d\n", result);
        return BTHH_ERR_NO_RES;
    }

    for (int index = 0 ; index < APP_NUM_ELEMENTS(app_xml_remote_devices_db) ; index++)
    {
        tAPP_XML_REM_DEVICE *p_xml_dev = &app_xml_remote_devices_db[index];

        if(!isSameBdAddr(p_xml_dev->bd_addr, pDevice->bd_addr) || !p_xml_dev->in_use)
        {
            continue;
        }

        BTHAL_BSA_DEBUG("Adding SEC Device:%s", p_xml_dev->name);
        tBSA_STATUS bsa_status = BtHalBsaIntSecAddDev(p_xml_dev);
        if (bsa_status != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] BSA_SecAddDevice failed status:%d\n", bsa_status);
            return BTHH_ERR;
        }

        /* Add every HID devices found in remote device database */
        /* They will be able to connect to our device */
        BTHAL_BSA_DEBUG("Add HID device found in database");
        if (p_xml_dev->trusted_services & BSA_HID_SERVICE_MASK)
        {
            BTHAL_BSA_DEBUG("Adding HID Device:%s", p_xml_dev->name);
            BtHalBsaIntHhAddDev(p_xml_dev);
        }
    }

    return BTHH_OK;
}

/*******************************************************************************
 **
 ** Function        BtHalGetPairedDevice
 **
 ** Description     Get paired device settings
 **
 ** Parameters      pBdAddr - Target device to get info from
 **                 pDeviceOut - Settings info desination
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalGetPairedDevice(const Btbdaddr *pBdAddr, nn::settings::system::BluetoothDevicesSettings* pDeviceOut)
{
    BTHAL_IF_DEBUG("[bluetooth] %s: %s", __FUNCTION__, toString(pBdAddr->address, 6) );
    //app_xml_display_devices();

    tAPP_XML_REM_DEVICE *pXmlDevice;

    if (pDeviceOut == NULL || pBdAddr == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: NULL param\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    pXmlDevice = app_xml_find_dev_db(pBdAddr->address);

    if (pXmlDevice == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    /* If this element is in use (not empty) */
    if (pXmlDevice->in_use != FALSE)
    {
        memcpy(pDeviceOut->bd_addr,         pXmlDevice->bd_addr,         sizeof(pDeviceOut->bd_addr));
        memcpy(pDeviceOut->device_name,     pXmlDevice->name,            sizeof(pDeviceOut->device_name));
        memcpy(pDeviceOut->class_of_device, pXmlDevice->class_of_device, sizeof(pDeviceOut->class_of_device));
        memcpy(pDeviceOut->link_key,        pXmlDevice->link_key,        sizeof(pDeviceOut->link_key));
        memcpy(pDeviceOut->brr,             pXmlDevice->brr,             sizeof(pDeviceOut->brr));
        pDeviceOut->link_key_present = pXmlDevice->link_key_present;
        pDeviceOut->trusted_services = pXmlDevice->trusted_services;
        pDeviceOut->vid              = pXmlDevice->vid;
        pDeviceOut->pid              = pXmlDevice->pid;
        pDeviceOut->version          = pXmlDevice->version;
        pDeviceOut->key_type         = pXmlDevice->key_type;
        pDeviceOut->device_type      = pXmlDevice->device_type;
        pDeviceOut->sub_class        = pXmlDevice->sub_class;
        pDeviceOut->attribute_mask   = pXmlDevice->attr_mask;
        pDeviceOut->brr_size         = pXmlDevice->brr_size;

        if (pXmlDevice->descriptor_size > sizeof(pDeviceOut->descriptor))
        {
            NN_SDK_LOG("[bluetooth] %s: invalid desciptor_size (%d)\n", __func__, pXmlDevice->descriptor_size);
            pDeviceOut->descriptor_length = 0;
            memset(pDeviceOut->descriptor, 0, sizeof(pDeviceOut->descriptor));
        }
        else
        {
            pDeviceOut->descriptor_length = pXmlDevice->descriptor_size;
            memcpy(pDeviceOut->descriptor, pXmlDevice->descriptor, pXmlDevice->descriptor_size);
        }
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: Xml info empty \n", __func__);
        return BTHH_ERR_BAD_RESPONSE;
    }

    return BTHH_OK;
}
// ////////////////////////////////////////// BSA HH PROFILE ////////////////////////////////////////// //
// ////////////////////////////////////////// BSA HH PROFILE ////////////////////////////////////////// //
// ////////////////////////////////////////// BSA HH PROFILE ////////////////////////////////////////// //
/*******************************************************************************
 **
 ** Function        InitHh
 **
 ** Description     ROBSON HH initialization function
 **
 ** Parameters      none
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int BtHalBsaInitHh()
{
    app_hh_uipc_chan = UIPC_CH_ID_BAD;
    return 0;
}

/*******************************************************************************
 **
 ** Function        StartHh
 **
 ** Description     ROBSON HH start function
 **
 ** Parameters      none
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int BtHalBsaStartHh()
{
    int status;
    tBSA_HH_ENABLE enable_param;

    BTHAL_BSA_DEBUG("app_hh_start");

    BSA_HhEnableInit(&enable_param);
    enable_param.sec_mask = APP_HH_SEC_DEFAULT;
    enable_param.p_cback = BtHalCallbackConnectionState;

    status = BSA_HhEnable(&enable_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to enable HID Host status:%d\n", __func__, status);
        return -1;
    }

    app_hh_uipc_chan = enable_param.uipc_channel; /* Save UIPC channel */
    UIPC_Open(app_hh_uipc_chan, BtHalCallbackReportData);  /* Open UIPC channel */

    return 0;
}

/*******************************************************************************
 **
 ** Function         BtHalBsaExitHh
 **
 ** Description      Exit HH application
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
int BtHalBsaExitHh()
{
    tBSA_HH_DISABLE dis_param;

    UIPC_Close(app_hh_uipc_chan);  /* Close UIPC channel */
    app_hh_uipc_chan = UIPC_CH_ID_BAD;

    BSA_HhDisableInit(&dis_param);
    BSA_HhDisable(&dis_param);

    return 0;
}

// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //

/*******************************************************************************
 **
 ** Function         _BtHalIntSecAddDev
 **
 ** Description      Add an HID device in the internal tables of BSA
 **
 ** Parameters       index: index of the device in the remote entries XML table
 **
 ** Returns          Pointer to HH device structure (NULL if failed)
 **
 *******************************************************************************/
tBSA_STATUS BtHalBsaIntSecAddDev(tAPP_XML_REM_DEVICE *p_xml_dev)
{
    tBSA_STATUS bsa_status;
    tBSA_SEC_ADD_DEV bsa_add_dev_param;

    BTHAL_BSA_DEBUG("Adding:%s", p_xml_dev->name);
    BSA_SecAddDeviceInit(&bsa_add_dev_param);
    copyBdAddr(bsa_add_dev_param.bd_addr, p_xml_dev->bd_addr);
    memcpy(bsa_add_dev_param.class_of_device, p_xml_dev->class_of_device, sizeof(DEV_CLASS));
    memcpy(bsa_add_dev_param.link_key,        p_xml_dev->link_key,        sizeof(LINK_KEY));
    bsa_add_dev_param.link_key_present = p_xml_dev->link_key_present;
    bsa_add_dev_param.trusted_services = p_xml_dev->trusted_services;
    bsa_add_dev_param.is_trusted = TRUE;
    bsa_add_dev_param.key_type = p_xml_dev->key_type;
    bsa_add_dev_param.io_cap = p_xml_dev->io_cap;
    bsa_status = BSA_SecAddDevice(&bsa_add_dev_param);
    if (bsa_status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to add SEC device:%d\n", __func__, bsa_status);
    }
    return bsa_status;
}
/*******************************************************************************
 **
 ** Function         _BtHalIntHhAddDev
 **
 ** Description      Add an HID device in the internal tables of BSA
 **
 ** Parameters       index: index of the device in the remote entries XML table
 **
 ** Returns          Pointer to HH device structure (NULL if failed)
 **
 *******************************************************************************/
tBSA_STATUS BtHalBsaIntHhAddDev(tAPP_XML_REM_DEVICE *p_xml_dev)
{
    DEV_CLASS class_of_device;
    UINT8 *p_cod = &class_of_device[0];
    UINT8 major;
    UINT8 minor;
    UINT16 services;
    int status;
    tBSA_HH_ADD_DEV hh_add_dev_param;

    /* Check if device is already in internal tables */
    //p_xml_dev->info_mask = APP_HH_DEV_USED;
    p_xml_dev->handle = BSA_HH_INVALID_HANDLE;

    /* Add device in HH tables */
    BSA_HhAddDevInit(&hh_add_dev_param);

    copyBdAddr(hh_add_dev_param.bd_addr, p_xml_dev->bd_addr);

    BTHAL_BSA_DEBUG("Attribute mask = %x - sub_class = %x",p_xml_dev->attr_mask,p_xml_dev->sub_class);
    hh_add_dev_param.attr_mask = p_xml_dev->attr_mask;
    hh_add_dev_param.sub_class = p_xml_dev->sub_class;
    hh_add_dev_param.peerinfo.ssr_max_latency = p_xml_dev->ssr_max_latency;
    hh_add_dev_param.peerinfo.ssr_min_tout = p_xml_dev->ssr_min_tout;
    hh_add_dev_param.peerinfo.supervision_tout = p_xml_dev->supervision_tout;
    hh_add_dev_param.attr_mask |= HID_SUP_TOUT_AVLBL;

    hh_add_dev_param.peerinfo.vendor_id = p_xml_dev->vid;
    hh_add_dev_param.peerinfo.product_id = p_xml_dev->pid;
    hh_add_dev_param.peerinfo.version = p_xml_dev->version;

    /* Copy the descriptor if available */
    if ((p_xml_dev->descriptor_size != 0) &&
        (p_xml_dev->descriptor_size <= sizeof(hh_add_dev_param.dscp_data.data)))
    {
        hh_add_dev_param.dscp_data.length = p_xml_dev->descriptor_size;
        memcpy(hh_add_dev_param.dscp_data.data, p_xml_dev->descriptor,
                p_xml_dev->descriptor_size);
    }

    status = BSA_HhAddDev(&hh_add_dev_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to add HID device:%d\n", __func__, status);
        //p_xml_dev->in_use = FALSE;
        //p_xml_dev->info_mask = 0;
        //p_xml_dev->trusted_services &= ~BSA_HID_SERVICE_MASK;
        //app_xml_write_db();
        return status;
    }

    /* Save the HH handle in the internal table */
    p_xml_dev->handle = hh_add_dev_param.handle;

    /* Add HID to security trusted services */
    p_xml_dev->trusted_services |= BSA_HID_SERVICE_MASK;
    BTHAL_BSA_DEBUG("Services = x%x", p_xml_dev->trusted_services);

    /* Extract Minor and Major numbers from COD */
    p_cod = p_xml_dev->class_of_device;
    BTM_COD_MINOR_CLASS(minor, p_cod);
    BTM_COD_MAJOR_CLASS(major, p_cod);
    BTHAL_BSA_DEBUG("COD:%02X-%02X-%02X", p_cod[0], p_cod[1], p_cod[2]);

    /* Check if the COD means RemoteControl */
    if ((major == BTM_COD_MAJOR_PERIPHERAL) &&
        ((minor & ~BTM_COD_MINOR_COMBO) == BTM_COD_MINOR_REMOTE_CONTROL))
    {
        BTM_COD_SERVICE_CLASS(services, p_cod);
        BTHAL_BSA_DEBUG("This device is a RemoteControl");
    }
    return BSA_SUCCESS;
}


BluetoothHhStatus BtHalGetAfhChannelMap(const Btbdaddr *pBdAddr, ChannelInfo* pOutChannelInfo)
{
    tBSA_STATUS status = BSA_SUCCESS;

    tBSA_TM_GET_HANDLE connectionHandle;
    tBSA_TM_HCI_CMD hciCmd;
    uint8_t* pData;
    uint8_t afhMode;
    uint8_t afhChannelMap[ARRAY_SIZE_OF_CHANNEL_INFO];
    uint8_t count = 0;
    uint16_t handle = 0;

    status = BSA_TmGetHandleInit(&connectionHandle);

    connectionHandle.transport = BT_TRANSPORT_BR_EDR;   // Should add BR/EDR or LE to the arguments?

    for(int i = 0; i < BD_ADDR_LEN; ++i)
    {
        connectionHandle.bd_addr[i] = pBdAddr->address[i];
    }

    BSA_TmGetHandle(&connectionHandle);

    if(status != BSA_SUCCESS)
    {
        BTHAL_BSA_DEBUG("[bluetooth] BSA_TmGetHandle error. (status = %d)\n", status);
        return BtHalBsaConvertResult(status);
    }

    status = BSA_TmHciInit(&hciCmd);

    if(status != BSA_SUCCESS)
    {
        BTHAL_BSA_DEBUG("[bluetooth] BSA_TmHciInit error. (status = %d)\n", status);
        return BtHalBsaConvertResult(status);
    }

    hciCmd.opcode = HCI_READ_AFH_CH_MAP;
    hciCmd.length = 2;
    hciCmd.no_opcode_swap = TRUE;
    pData = hciCmd.data;

    // copy handle
    *(pData)++ = (uint8_t)connectionHandle.handle;
    *(pData)++ = (uint8_t)(connectionHandle.handle >> 8);

    status = BSA_TmHciCmd(&hciCmd);
    if(status != BSA_SUCCESS)
    {
        BTHAL_BSA_DEBUG("[bluetooth] BSA_TmHciInit error. (status = %d)\n", status);
        return BtHalBsaConvertResult(status);
    }

    pData = hciCmd.data;
    handle = (uint16_t)(pData[0]) + (((uint16_t)(pData[1]) << 8));

    afhMode = (uint8_t)pData[2];

    for(int i = 0; i < ARRAY_SIZE_OF_CHANNEL_INFO; ++i)
    {
        afhChannelMap[i] = (uint8_t)pData[i + 3];

        // count bits "1"
        uint8_t tempCount = afhChannelMap[i];
        tempCount = (tempCount & 0x55) + ((tempCount >> 1) & 0x55);
        tempCount = (tempCount & 0x33) + ((tempCount >> 2) & 0x33);
        tempCount = (tempCount & 0x0f) + ((tempCount >> 4) & 0x0f);

        count += tempCount;
    }

    BTHAL_BSA_DEBUG("Status: %d", hciCmd.status);
    BTHAL_BSA_DEBUG("Current AFH channel MAP:");
    BTHAL_BSA_DEBUG("Addr: %02X-%02X-%02X-%02X-%02X-%02X handle:%d mode:%d",
                    pBdAddr->address[0], pBdAddr->address[1], pBdAddr->address[2], pBdAddr->address[3], pBdAddr->address[4], pBdAddr->address[5], handle, afhMode);
    BTHAL_BSA_DEBUG("Ch: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X count: %d",
                    afhChannelMap[0], afhChannelMap[1], afhChannelMap[2], afhChannelMap[3], afhChannelMap[4],
                    afhChannelMap[5], afhChannelMap[6], afhChannelMap[7], afhChannelMap[8], afhChannelMap[9],
                    count);

    memcpy(pOutChannelInfo->channels, afhChannelMap, ARRAY_SIZE_OF_CHANNEL_INFO);
    pOutChannelInfo->count = count;

    return BtHalBsaConvertResult(status);
}


/*******************************************************************************
 **
 ** Function         BtHalBsaConvertResult
 **
 ** Description      Convert result from tBSA_STATUS to BluetoothHhStatus
 **
 ** Parameters       resultFromBsa: tBSA_STATUS to be converted
 **
 ** Returns          BluetoothHhStatus result
 **
 *******************************************************************************/
BluetoothHhStatus BtHalBsaConvertResult(tBSA_STATUS resultFromBsa)
{
    BluetoothHhStatus result;

    if(resultFromBsa == BSA_SUCCESS)
    {
        return BTHH_OK;
    }

    switch(resultFromBsa)
    {
        /* Bad parameter errors */
        case BSA_ERROR_CLI_BAD_PARAM:                // 104
        case BSA_ERROR_CLI_NAME_TOO_LONG:            // 111
        case BSA_ERROR_SRV_BAD_PARAM:                // 200
        case BSA_ERROR_SRV_BAD_REQ_SIZE:             // 201
        case BSA_ERROR_SRV_DG_FIND_SERVICE_FAIL:     // 1502
        case BSA_ERROR_SRV_BLE_INVALID_PARAM:        // 1603
            result = BTHH_ERR_BAD_PARAM;
        break;

        /* Bad message/response errors */
        case BSA_ERROR_CLI_BAD_RSP_SIZE:             // 100
        case BSA_ERROR_CLI_BAD_MSG:                  // 109
        case BSA_ERROR_CLI_BAD_PING_RSP:             // 110
        case BSA_ERROR_SRV_BAD_MSG_ID:               // 202
        case BSA_ERROR_SRV_HD_UNKNOWN_KEY_TYPE:      // 1702
        case BSA_ERROR_SRV_HD_WRONG_RPT_LEN:         // 1704
            result = BTHH_ERR_BAD_RESPONSE;
        break;

        /* Busy/not ready errors */
        case BSA_ERROR_CLI_DISC_BUSY:                // 102
        case BSA_ERROR_CLI_BUSY:                     // 106
        case BSA_ERROR_CLI_ALREADY_WAITING:          // 112
            result = BTHH_ERR_BUSY;
        break;

        /* Connection errors */
        case BSA_ERROR_CLI_NOT_CONNECTED:            // 105
        case BSA_ERROR_CLI_ALREADY_CONNECTED:        // 115
        case BSA_ERROR_SRV_ALREADY_ACTIVE:           // 203
        case BSA_ERROR_SRV_DG_CONNECTED:             // 1501
        case BSA_ERROR_SRV_HD_AREADY_CONNECTED:      // 1703
            result = BTHH_ERR_CONNECTION;
        break;

        /* Open/Enable errors */
        case BSA_ERROR_SRV_DG_OPEN_FAIL:             // 1500
        case BSA_ERROR_SRV_BLE_FAIL:                 // 1600
        case BSA_ERROR_SRV_BLE_NOT_ENABLED:          // 1601
        case BSA_ERROR_SRV_HD_FAIL:                  // 1700
        case BSA_ERROR_SRV_HD_NOT_ENABLED:           // 1701
            result = BTHH_ERR_OPEN;
        break;

        /* No resource/memory full errors */
        case BSA_ERROR_CLI_RES_ERROR:                // 103
        case BSA_ERROR_CLI_MEM_FULL:                 // 116
        case BSA_ERROR_SRV_MEM_FULL:                 // 206
        case BSA_ERROR_SRV_BLE_NO_RESOURCE:          // 1602
            result = BTHH_ERR_NO_RES;
        break;

        /* General client errors */
        case BSA_ERROR_CLI_NYI:                      // 101
        case BSA_ERROR_CLI_TASK_ERROR:               // 113
        case BSA_ERROR_CLI_INTERNAL:                 // 114
            result = BTHH_ERR_CLIENT;
        break;

        /* General server errors */
        case BSA_ERROR_SRV_NYI:                      // 204
        case BSA_ERROR_SRV_BAD_CLIENT:               // 205
        case BSA_ERROR_SRV_HW_ERROR:                 // 207
        case BSA_ERROR_SRV_BLUETOOTH_DISABLE:        // 208
        case BSA_ERROR_SRV_INTERNAL:                 // 209
        case BSA_ERROR_SRV_NOT_COMPILED:             // 210
        case BSA_ERROR_SRV_WLAN_RESET:               // 211
            result = BTHH_ERR_SERVER;
        break;

        /* Security errors */
        case BSA_ERROR_SRV_SEC_BOND_ACTIVE:          // 300
        case BSA_ERROR_SRV_SEC_RM_DEV:               // 301
        case BSA_ERROR_SRV_SEC_BOND_CANCEL_ERROR:    // 302
            result = BTHH_ERR_SECURITY;
        break;

        /* General hid host errors */
        case BSA_ERROR_SRV_HH_DSCP_TOO_BIG:          // 400
        case BSA_ERROR_SRV_HH_DSCP_PENDING:          // 401
        case BSA_ERROR_SRV_HH_NOT_ENABLED:           // 402
        case BSA_ERROR_SRV_HH_NOT_SUPPORTED:         // 403
        case BSA_ERROR_SRV_HH_NOT_ALLOWED:           // 404
        case BSA_ERROR_SRV_HH_UNKNOWN_DEVICE:        // 405
        case BSA_ERROR_SRV_HH_ALREADY_ADDED:         // 406
        case BSA_ERROR_SRV_HH_ERR:                   // 413
            result = BTHH_HH_ERR;
        break;

        /* Specific hid host errors */
        case BSA_ERROR_SRV_HH_HS_HID_NOT_READY:      // 408
            result = BTHH_HS_HID_NOT_READY;
        break;
        case BSA_ERROR_SRV_HH_HS_INVALID_RPT_ID:     // 409
            result = BTHH_HS_INVALID_RPT_ID;
        break;
        case BSA_ERROR_SRV_HH_HS_TRANS_NOT_SPT:      // 410
            result = BTHH_HS_TRANS_NOT_SPT;
        break;
        case BSA_ERROR_SRV_HH_HS_INVALID_PARAM:      // 411
            result = BTHH_HS_INVALID_PARAM;
        break;
        case BSA_ERROR_SRV_HH_HS_ERROR:              // 412
            result = BTHH_HS_ERROR;
        break;
        case BSA_ERROR_SRV_HH_ERR_SDP:               // 414
            result = BTHH_ERR_SDP;
        break;
        case BSA_ERROR_SRV_HH_ERR_PROTO:             // 415
            result = BTHH_ERR_PROTO;
        break;
        case BSA_ERROR_SRV_HH_ERR_DB_FULL:           // 416
            result = BTHH_ERR_DB_FULL;
        break;
        case BSA_ERROR_SRV_HH_ERR_TOD_UNSPT:         // 417
            result = BTHH_ERR_TOD_UNSPT;
        break;
        case BSA_ERROR_SRV_HH_ERR_NO_RES:            // 418
            result = BTHH_ERR_NO_RES;
        break;
        case BSA_ERROR_SRV_HH_ERR_AUTH_FAILED:       // 419
            result = BTHH_ERR_AUTH_FAILED;
        break;
        case BSA_ERROR_SRV_HH_ERR_HDL:               // 420
            result = BTHH_ERR_HDL;
        break;

        /* File system errors */
        case BSA_ERROR_SRV_FS_PERMISSION:            // 600
        case BSA_ERROR_SRV_FS_NO_FILE:               // 601
            result = BTHH_ERR_FILE_SYS;
        break;

        /* Audio video errors */
        case BSA_ERROR_SRV_AV_NOT_ENABLED:           // 700
        case BSA_ERROR_SRV_AV_FEEDING_NOT_SUPPORTED: // 701
        case BSA_ERROR_SRV_AV_BUSY:                  // 702
        case BSA_ERROR_SRV_AV_NOT_OPENED:            // 703
        case BSA_ERROR_SRV_AV_NOT_STARTED:           // 704
        case BSA_ERROR_SRV_AV_NOT_STOPPED:           // 705
        case BSA_ERROR_SRV_AV_CP_NOT_SUPPORTED:      // 706
        case BSA_ERROR_SRV_AV_BCST_NOT_SUPPORTED:    // 707
        case BSA_ERROR_SRV_AV_BR_CT_NOT_SUPPORTED:   // 708
        case BSA_ERROR_SRV_AV_FAIL:                  // 709
        case BSA_ERROR_SRV_AV_FAIL_SDP:              // 710
        case BSA_ERROR_SRV_AV_FAIL_STREAM:           // 711
        case BSA_ERROR_SRV_AV_FAIL_RESOURCES:        // 712
        case BSA_ERROR_SRV_AV_FAIL_ROLE:             // 713
            result = BTHH_ERR_AUDIO_VIDEO;
        break;

        /* Discovery errors */
        case BSA_ERROR_SRV_DEV_DISC_BUSY:            // 900
        case BSA_ERROR_SRV_DEV_INFO_BUSY:            // 901
        case BSA_ERROR_SRV_DISC_ABORT_ERROR:         // 902
            result = BTHH_ERR_DISCOVERY;
        break;

        /* Test mode errors */
        case BSA_ERROR_SRV_TM_VSC_BUSY:              // 1100
        case BSA_ERROR_SRV_TM_HCI_ERROR:             // 1101
        case BSA_ERROR_SRV_TM_HCI_CMD_ERROR:         // 1102
            result = BTHH_ERR_TEST_MODE;
        break;

        /* DM errors */
        case BSA_ERROR_SRV_DM_3DTV_LOCK_FAIL:        // 1200
        case BSA_ERROR_SRV_DM_TBFC_UNSUPPORTED:      // 1201
        case BSA_ERROR_SRV_DM_TBFC_NOT_COMPILED:     // 1202
        case BSA_ERROR_SRV_DM_SWITCH_STACK_FAIL:     // 1203
            result = BTHH_ERR_DM;
        break;

        /* Robson errors */
        case BSA_ERROR_ROBSON_MEM_FULL:                       // 2100
        case BSA_ERROR_ROBSON_SET_TSI_NOT_ALLOWED:            // 2101
        case BSA_ERROR_ROBSON_EXIT_TSI_NOT_ALLOWED:           // 2102
        case BSA_ERROR_ROBSON_TSI_FAIL:                       // 2103
        case BSA_ERROR_ROBSON_SET_BURST_NOT_ALLOWED:          // 2104
        case BSA_ERROR_ROBSON_EXIT_BURST_NOT_ALLOWED:         // 2105
        case BSA_ERROR_ROBSON_BURST_FAIL:                     // 2106
        case BSA_ERROR_ROBSON_SWITCH_MODE_FAIL:               // 2107
        case BSA_ERROR_ROBSON_SET_PACKET_TYPE_FAIL:           // 2108
        case BSA_ERROR_ROBSON_READ_0RETRAN_FAIL:              // 2109
        case BSA_ERROR_ROBSON_SET_0RETRAN_MODE_FAIL:          // 2110
        case BSA_ERROR_ROBSON_SUPER_PICONET_FAIL:             // 2111
        case BSA_ERROR_ROBSON_SET_ENCRYPTION_FAIL:            // 2112
        case BSA_ERROR_ROBSON_LINK_DOWN:                      // 2113
        case BSA_ERROR_ROBSON_LLR_START_FAIL_OVER_MAX:        // 2114
        case BSA_ERROR_ROBSON_LLR_SET_TBFC_PARAM_FAIL:        // 2115
        case BSA_ERROR_ROBSON_LLR_SET_SLEEP_MODE_FAIL:        // 2116
        case BSA_ERROR_ROBSON_LLR_WRITE_STORED_LINK_KEY_FAIL: // 2117
        case BSA_ERROR_ROBSON_LLR_SET_REMOTE_ONOFF_FAIL:      // 2118
        case BSA_ERROR_ROBSON_SET_RADIO_FAIL:                 // 2119
        case BSA_ERROR_ROBSON_SET_FW_CONTROL_HOPPING_FAIL:    // 2120
        case BSA_ERROR_ROBSON_SET_MWS_CHANNEL_PARAMETER_FAIL: // 2121
        case BSA_ERROR_ROBSON_LLR_START_FAIL_CONNECTION_EXIST: // 2122
        case BSA_ERROR_ROBSON_DEV_STATUS_DOWN:                // 2128    /* Device Down */
        case BSA_ERROR_ROBSON_DEV_STATUS_CMD_TOUT:            // 2129    /* HCI Command Timeout */
        case BSA_ERROR_ROBSON_DEV_STATUS_HW_ERR_EVT:          // 2130    /* HW Err Evt */
            result = BTHH_ERR_ROBSON;
        break;

        default:
            result = BTHH_ERR;
            NN_SDK_LOG("[bluetooth] %s: Unknown BSA status returned: %d \n", __func__, resultFromBsa);
        break;
    }
    return result;
} // NOLINT(impl/function_size)
