﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include "bluetooth.h"
#include "bluetooth_ext.h"
#include "bluetooth_InternalTypes.h"
#include "bluetooth_queue.h"

namespace nn {
namespace bluetooth {

/*******************************************************************************
 **
 ** Function        SendData
 **
 ** Description     Send HID data to the device specified in pBdAddr
 **
 ** Parameters      pBdAddr - Target device
 **                 data    - pointer to the data being sent to the target device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::InitializeExtension(BluetoothExtCallbacks* callbacks)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    ::BtHalCallbackRegisterExtension(callbacks);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetTSI
 **
 ** Description     Set TSI mode
 **
 ** Parameters      pBdAddr     - Target device
 **                 tsi_mode    - TSI mode
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetTSI(Btbdaddr *pBdAddr, uint8_t tsiMode)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. tsiMode=%d", tsiMode);
    status = BtHalSetTsi(pBdAddr,tsiMode);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfExitTSI
 **
 ** Description     Exit TSI mode
 **
 ** Parameters      pBdAddr     - Target device
 **
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfExitTSI(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalExitTsi(pBdAddr);
    return status;
}


/*******************************************************************************
 **
 ** Function        HalIfSetBurstMode
 **
 ** Description     Enter burst mode
 **
 ** Parameters      pBdAddr     - Target device
 **
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetBurstMode(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalSetBurstMode(pBdAddr);
    return status;
}


/*******************************************************************************
 **
 ** Function        HalIfExitBurstMode
 **
 ** Description     Exit Burst mode
 **
 ** Parameters      pBdAddr     - Target device
 **
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfExitBurstMode(Btbdaddr *pBdAddr)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalExitBurstMode(pBdAddr);
    return status;
}


/*******************************************************************************
 **
 ** Function        HalIfSetSfh
 **
 ** Description     Set Static AFH
 **
 ** Parameters      pBdAddr     - Target device
 **                 tsi_mode    - TSI mode
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetSfh(uint8_t sfh_on, uint8_t *pChannels,uint32_t length, uint32_t sfh_offset)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. sfh_on:%d length:%d", sfh_on, length);
    status = BtHalSetSfh(sfh_on,pChannels,length,sfh_offset);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetZeroRetran
 **
 ** Description     Set 0-re-transmission parameters
 **
 ** Parameters      pBdAddr     - Target device
 **
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetZeroRetran(Btbdaddr *pBdAddr, uint8_t numIds, const uint8_t *pIds)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. numIds:%d", numIds);
    status = BtHalSetZeroRetran(pBdAddr, numIds, pIds);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetMcMode
 **
 ** Description     Set MC mode parameters
 **
 ** Parameters      isMcModeEnabled
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetMcMode(bool isMcModeEnabled)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. enabled=%d", isMcModeEnabled);
    status = BtHalSetMcMode(isMcModeEnabled);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfStartLlrMode
 **
 ** Description     Start LLR mode by setting the GPIO and calling the BSA function
 **
 ** Parameters
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfStartLlrMode()
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalStartLlrMode();
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfStartLlrMode
 **
 ** Description     Exit LLR mode by setting the GPIO and calling the BSA function
 **
 ** Parameters
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfExitLlrMode()
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalExitLlrMode();
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetRadio
 **
 ** Description      Set the Device Visibility parameters (InquiryScan & PageScan)
 **
 ** Parameters       enable: FALSE for off, TRUE for on
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetRadio(bool enable)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. enable=%d", enable);
    status = BtHalSetRadio(enable);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetVisibility
 **
 ** Description      Set the Device Visibility parameters (InquiryScan & PageScan)
 **
 ** Parameters       discoverable: FALSE if not discoverable
 **                  connectable: FALSE if not connectable
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetVisibility(bool discoverable, bool connectable)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. disc=%d conn=%d", discoverable, connectable);
    status = BtHalSetVisibility(discoverable, connectable);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfSetTbfcScan
 **
 ** Description      Set the Device TBFC scan ability
 **
 ** Parameters       enable: FALSE to disable TBFC scan
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfSetTbfcScan(bool enable)
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called. enable=%d", enable);
    status = BtHalSetTbfcScan(enable);
    return status;
}

/*******************************************************************************
 **
 ** Function        HalIfGetPendingConnections
 **
 ** Description      Set the Device Visibility parameters (InquiryScan & PageScan)
 **
 ** Parameters       discoverable: FALSE if not discoverable
 **                  connectable: FALSE if not connectable
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BluetoothExtC::HalIfGetPendingConnections()
{
    BluetoothHhStatus status = BTHH_OK;
    BTHAL_IF_DEBUG("Called");
    status = BtHalGetNumPendingConnections();
    return status;
}


} // bluetooth
} // nn

