﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bcat_EventWatcherThread.h"
#include "bcat_NintendoAccountCacheUpdateThread.h"

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/npns/npns_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/bcat/service/bcat_Server.h>
#include <nn/news/service/news_Server.h>
#include <nn/prepo/service/prepo_Server.h>
#include <nn/result/result_HandlingUtility.h>

namespace
{
    const nn::ApplicationId ProcessAppId = {0x010000000000000C};
}

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[20 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    void ReceiveNotification() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            nn::npns::NotificationData data;

            nn::Result result = nn::npns::Receive(&data);

            if (nn::npns::ResultNotReceived::Includes(result))
            {
                break;
            }
            if (result.IsFailure())
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::npns::Receive() failed. code = %03d-%04d\n",
                    result.GetModule(), result.GetDescription());
                break;
            }

            nn::bcat::service::NotificationPayload payload(data.GetPayload(), data.GetPayloadSize());

            if (payload.IsValid())
            {
                if (nn::bcat::service::NotifyNotificationReceived(payload))
                {
                    continue;
                }
                if (nn::news::service::NotifyNotificationReceived(payload))
                {
                    continue;
                }

                NN_DETAIL_BCAT_INFO("[bcat] The payload was unhandled.\n");
            }
            else
            {
                NN_DETAIL_BCAT_INFO("[bcat] The invalid format payload was received.\n");
            }
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::ListenTo(ProcessAppId));

        nn::os::SystemEvent npnsNotificationReceiveEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::GetReceiveEvent(npnsNotificationReceiveEvent));

        nn::os::SystemEvent npnsStateChangeEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::GetStateChangeEvent(npnsStateChangeEvent));

        nn::account::Notifier nsaAvailabilityChangeNotifier;
        nn::os::SystemEvent nsaAvailabilityChangeEvent;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAvailabilityChangeNotifier(&nsaAvailabilityChangeNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nsaAvailabilityChangeNotifier.GetSystemEvent(&nsaAvailabilityChangeEvent));

        nn::nifm::NetworkConnection connectionMonitor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connectionMonitor.GetRequestHandle(),
            nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));

        connectionMonitor.SubmitRequest();

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitAny(g_StopEvent.GetBase(),
                npnsNotificationReceiveEvent.GetBase(),
                npnsStateChangeEvent.GetBase(),
                nsaAvailabilityChangeEvent.GetBase(),
                connectionMonitor.GetSystemEvent().GetBase());

            if (g_StopEvent.TryWait())
            {
                break;
            }

            if (npnsNotificationReceiveEvent.TryWait())
            {
                npnsNotificationReceiveEvent.Clear();

                ReceiveNotification();
            }
            if (npnsStateChangeEvent.TryWait())
            {
                npnsStateChangeEvent.Clear();

                nn::bcat::service::NotifyNpnsStateChanged();
                nn::news::service::NotifyNpnsStateChanged();
            }
            if (nsaAvailabilityChangeEvent.TryWait())
            {
                nsaAvailabilityChangeEvent.Clear();

                nn::bcat::service::NotifyNetworkServiceAccountAvailabilityChanged();
                nn::news::service::NotifyNetworkServiceAccountAvailabilityChanged();

                NintendoAccountCacheUpdateThread::RequestNintendoAccountCacheUpdate();
            }
            if (connectionMonitor.GetSystemEvent().TryWait())
            {
                connectionMonitor.GetSystemEvent().Clear();

                if (connectionMonitor.IsAvailable())
                {
                    nn::bcat::service::NotifyNetworkConnected();
                    nn::news::service::NotifyNetworkConnected();
                    nn::prepo::service::NotifyNetworkConnected();
                }
            }
        }
    }
}

void EventWatcherThread::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(bcat, EventWatcher)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(bcat, EventWatcher));
    nn::os::StartThread(&g_Thread);
}

void EventWatcherThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}
