﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Thread.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>
#include <nn/idle/detail/idle_Log.h>
#include <nn/idle/idle_ServiceName.h>
#include <nn/idle/idle_ShimInternal.h>
#include <nn/idle/server/idle_HandlerManager.h>
#include <nn/idle/server/idle_PolicyManagerSystemImpl.h>
#include "am_IdleStateManager.h"

namespace nn { namespace am {

namespace
{
    // サーバー側から発行する全セッションの Max
    const int IdleSessionCountMax = 8;
    class IdleServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<IdleSessionCountMax, 1>
    {};

    IdleServerManager g_IdleServerManager;

    sf::UnmanagedServiceObject<nn::idle::detail::IPolicyManagerSystem, nn::idle::server::PolicyManagerSystemImpl> g_PolicyManagerSystemServer;

    nn::os::ThreadType              g_IdleStateHandlerThread;
    NN_OS_ALIGNAS_THREAD_STACK char g_IdleStateHandlerThreadStack[0x2000];

    // スリープ等の通知を受け取るオブジェクト
    nn::psc::PmModule g_PmModule;

    nn::Result HandlePmEvent(nn::psc::PmState state, nn::psc::PmFlagSet flags)
    {
        NN_UNUSED(flags);
        switch (state)
        {
        case nn::psc::PmState_FullAwake:
            nn::idle::server::EnableHandlers();
            break;
        default:
            nn::idle::server::DisableHandlers();
            break;
        }
        NN_RESULT_SUCCESS;
    }

    void IdleStateHandlerThread(void* arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);

        // idle サーバの初期化
        NN_DETAIL_IDLE_INFO("Starting idle server\n");
        nn::idle::server::InitializeHandlerManager(); // 内部で settings アクセスあり

        // サーバマネージャの開始
        // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
        g_IdleServerManager.Start();

        // PSC へのモジュール登録
        const nn::psc::PmModuleId PmDependencies[] =
        {
            nn::psc::PmModuleId_Display,
            nn::psc::PmModuleId_Lbl,
            nn::psc::PmModuleId_Hid,
        };
        auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Idle,
            PmDependencies,
            NN_ARRAY_SIZE(PmDependencies),
            nn::os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_DETAIL_IDLE_INFO("Started idle server\n");

        const uintptr_t UpdateIdleHandlerStatusTag = 0x80000001;
        const uintptr_t PmModuleEventStatusTag = 0x80000002;

        nn::os::MultiWaitHolderType pmEventHolder;
        nn::os::InitializeMultiWaitHolder(&pmEventHolder, g_PmModule.GetEventPointer()->GetBase());
        nn::os::SetMultiWaitHolderUserData(&pmEventHolder, PmModuleEventStatusTag);
        g_IdleServerManager.AddUserWaitHolder(&pmEventHolder);

        nn::os::TimerEventType timerEvent;
        nn::os::MultiWaitHolderType timerEventHolder;

        nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
        nn::os::SetMultiWaitHolderUserData(&timerEventHolder, UpdateIdleHandlerStatusTag);

        // 25ms 毎に無操作状態ハンドラの設定を更新するタイマーの有効化
        // ポーリングベースで低輝度化などを十分すばやく解除できる程度の間隔であること
        const auto UpdateIdleHandlerInterval = nn::TimeSpan::FromMilliSeconds(25);
        nn::os::StartPeriodicTimerEvent(&timerEvent, UpdateIdleHandlerInterval, UpdateIdleHandlerInterval);

        g_IdleServerManager.AddUserWaitHolder(&timerEventHolder);

        while ( auto p = g_IdleServerManager.Wait() )
        {
            uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);
            switch ( userData )
            {
            case IdleServerManager::InvokeTag:
            case IdleServerManager::AcceptTag:
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_IdleServerManager.ProcessAuto(p));
                break;
            }
            case UpdateIdleHandlerStatusTag:
            {
                nn::os::ClearTimerEvent(&timerEvent);
                g_IdleServerManager.AddUserWaitHolder(p);
                // NN_DETAIL_IDLE_INFO_V2("Updating handler status\n");
                nn::idle::server::UpdateHandlerStatus(UpdateIdleHandlerInterval);
                break;
            }
            case PmModuleEventStatusTag:
            {
                bool tryWaitResult = g_PmModule.GetEventPointer()->TryWait();
                NN_SDK_ASSERT(tryWaitResult);
                NN_UNUSED(tryWaitResult);
                g_PmModule.GetEventPointer()->Clear();
                g_IdleServerManager.AddUserWaitHolder(&pmEventHolder);

                nn::psc::PmState    state;
                nn::psc::PmFlagSet  flags;
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));
                auto pmEventResult = HandlePmEvent(state, flags);

                g_PmModule.Acknowledge(state, pmEventResult);
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        nn::os::StopTimerEvent(&timerEvent);
        nn::idle::server::DisableHandlers();
        nn::os::FinalizeMultiWaitHolder(&timerEventHolder);
        nn::os::FinalizeTimerEvent(&timerEvent);
    }

}   // anonymous namespace

void StartIdleStateManager() NN_NOEXCEPT
{
    nn::idle::InitializeWith(g_PolicyManagerSystemServer.GetShared());

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    g_IdleServerManager.RegisterObjectForPort(nn::idle::GetPolicyManagerSystem(), IdleSessionCountMax, nn::idle::ServiceNameForSystem);

    auto result = nn::os::CreateThread( &g_IdleStateHandlerThread,
                                        IdleStateHandlerThread,
                                        nullptr,
                                        g_IdleStateHandlerThreadStack,
                                        sizeof(g_IdleStateHandlerThreadStack),
                                        NN_SYSTEM_THREAD_PRIORITY(idle, IpcServer));
    nn::os::SetThreadNamePointer(&g_IdleStateHandlerThread, NN_SYSTEM_THREAD_NAME(idle, IpcServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread(&g_IdleStateHandlerThread);
}

}}  // namespace nn::am

