﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>

#include <nn/nn_Common.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>

namespace nn {

class CancelPoint
{
    friend Result ConnectToInternet(nifm::NetworkConnection& con, CancelPoint& cp) NN_NOEXCEPT;

private:
    mutable os::Mutex m_Lock;
    bool m_IsActive;
    os::Event m_IsCancelled;
    std::pair<bool, account::AsyncContext*> m_Attachment;

    void Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;
    void AttachAsyncContext(account::AsyncContext& ctx) NN_NOEXCEPT;
    void DetachAsyncContext() NN_NOEXCEPT;

    os::EventType* GetEventBase() NN_NOEXCEPT;

public:
    CancelPoint() NN_NOEXCEPT;
    void Signal() NN_NOEXCEPT;
    bool Test() NN_NOEXCEPT;

    class ScopedActivation
    {
        NN_DISALLOW_COPY(ScopedActivation);

    private:
        CancelPoint& m_Cp;

    public:
        explicit ScopedActivation(CancelPoint& cp) NN_NOEXCEPT
            : m_Cp(cp)
        {
            m_Cp.Activate();
        }
        ~ScopedActivation() NN_NOEXCEPT
        {
            m_Cp.Deactivate();
        }
    };

    class Attachment
    {
        NN_DISALLOW_COPY(Attachment);

    private:
        CancelPoint& m_Cp;

    public:
        explicit Attachment(CancelPoint& cp, account::AsyncContext& ctx) NN_NOEXCEPT
            : m_Cp(cp)
        {
            m_Cp.AttachAsyncContext(ctx);
        }
        ~Attachment() NN_NOEXCEPT
        {
            m_Cp.DetachAsyncContext();
        }
    };
};

void ReportError(Result result, bool isAborting) NN_NOEXCEPT;
Result ConnectToInternet(nifm::NetworkConnection& con, CancelPoint& cp) NN_NOEXCEPT;

#if defined(NN_SDK_BUILD_RELEASE)
#define NN_ACCOUNT_INFO_WITH_TIMESTAMP(...)
#define NN_ACCOUNT_WARN_WITH_TIMESTAMP(...)
#define NN_ACCOUNT_ERROR_WITH_TIMESTAMP(...)
#else
enum PrintWithTimeStampKind
{
    PrintWithTimeStampKind_Info,
    PrintWithTimeStampKind_Warn,
    PrintWithTimeStampKind_Error,
};
void PrintWithTimeStamp(PrintWithTimeStampKind kind, const char* pFormat, ...) NN_NOEXCEPT;
#define NN_ACCOUNT_INFO_WITH_TIMESTAMP(...) PrintWithTimeStamp(::nn::PrintWithTimeStampKind_Info, __VA_ARGS__)
#define NN_ACCOUNT_WARN_WITH_TIMESTAMP(...) PrintWithTimeStamp(::nn::PrintWithTimeStampKind_Warn, __VA_ARGS__)
#define NN_ACCOUNT_ERROR_WITH_TIMESTAMP(...) PrintWithTimeStamp(::nn::PrintWithTimeStampKind_Error, __VA_ARGS__)
#endif

} // ~namespace nn
