﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_Util.h"

#include <mutex>

#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/detail/account_Log.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/err/err_SystemApi.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_FormatString.h>

namespace nn {

/* ---------------------------------------------------------------------------------------------
    CancelPoint
*/
void CancelPoint::Activate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_ABORT_UNLESS(!m_IsActive);
    m_IsActive = true;
    m_IsCancelled.Clear();
}
void CancelPoint::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsActive);
    m_IsActive = false;
}
void CancelPoint::AttachAsyncContext(account::AsyncContext& ctx) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsActive);
    NN_ABORT_UNLESS(!m_Attachment.first);

    m_Attachment.first = true;
    m_Attachment.second = &ctx;

    if (m_IsCancelled.TryWait())
    {
        auto r = m_Attachment.second->Cancel();
        NN_UNUSED(r);
#if !defined(NN_SDK_BUILD_RELEASE)
        if (!r.IsSuccess())
        {
            NN_DETAIL_ACCOUNT_ERROR(
                "[account] m_Attachment.second->Cancel() failed with %03d-%04d\n",
                r.GetModule(), r.GetDescription());
        }
#endif
    }
}
void CancelPoint::DetachAsyncContext() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsActive);
    NN_ABORT_UNLESS(m_Attachment.first);

    m_Attachment.first = false;
}
CancelPoint::CancelPoint() NN_NOEXCEPT
    : m_Lock(false)
    , m_IsActive(false)
    , m_IsCancelled(os::EventClearMode_ManualClear)
    , m_Attachment(false, nullptr)
{
}
void CancelPoint::Signal() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    if (!(m_IsActive && !m_IsCancelled.TryWait()))
    {
        return;
    }
    m_IsCancelled.Signal();
    if (m_Attachment.first)
    {
        auto r = m_Attachment.second->Cancel();
        NN_UNUSED(r);
#if !defined(NN_SDK_BUILD_RELEASE)
        if (!r.IsSuccess())
        {
            NN_DETAIL_ACCOUNT_ERROR(
                "[account] m_Attachment.second->Cancel() failed with %03d-%04d\n",
                r.GetModule(), r.GetDescription());
        }
#endif
    }
}
bool CancelPoint::Test() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_IsActive && m_IsCancelled.TryWait();
}
os::EventType* CancelPoint::GetEventBase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsActive);
    return m_IsCancelled.GetBase();
}

/* ---------------------------------------------------------------------------------------------
    Functions
*/

namespace {
inline Result ReportErrorImpl(Result result, bool isAborting) NN_NOEXCEPT
{
    // エラーコード
    auto errorCode = err::ConvertResultToErrorCode(result);
    char errorString[nn::err::ErrorCode::StringLengthMax];
    err::GetErrorCodeString(errorString, sizeof(errorString), errorCode);
    auto errorStringLength = static_cast<uint32_t>(strnlen(errorString, sizeof(errorString)));

    erpt::Context context(erpt::ErrorInfo);
    NN_RESULT_DO(context.Add(erpt::FieldId::AbortFlag, isAborting));
    NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, errorString, errorStringLength));
    NN_RESULT_DO(context.AddCurrentThreadName());
    NN_RESULT_DO(context.CreateReport(erpt::ReportType_Invisible));
    NN_RESULT_SUCCESS;
}
} // ~namespace

void ReportError(Result result, bool isAborting) NN_NOEXCEPT
{
    auto r = ReportErrorImpl(result, isAborting);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_ERROR(
            "[account] ReportError() failed with %03d-%04d\n",
            r.GetModule(), r.GetDescription());
    }
}

Result ConnectToInternet(nifm::NetworkConnection& con, CancelPoint& cp) NN_NOEXCEPT
{
    NN_RESULT_DO(nifm::SetRequestRequirementPreset(
        con.GetRequestHandle(),
        nifm::RequirementPreset_InternetForSystemProcessSharable));
    con.SubmitRequest();

    auto& e = con.GetSystemEvent();
    auto index = os::WaitAny(cp.GetEventBase(), e.GetBase());
    NN_UNUSED(index);

    bool isCancelled = false;
    if (cp.Test())
    {
        con.CancelRequest();
        isCancelled = true;
    }

    NN_SDK_ASSERT(e.TryWait());
    e.Clear();
    NN_SDK_ASSERT(!con.IsRequestOnHold());

    NN_RESULT_THROW_UNLESS(!isCancelled, account::ResultCancelled());
    NN_RESULT_THROW_UNLESS(con.IsAvailable(), account::ResultInternetConnectionUnavailable());
    NN_RESULT_SUCCESS;
}

#if !defined(NN_SDK_BUILD_RELEASE)
namespace {
int GetTimeStamp(char* str, size_t size) NN_NOEXCEPT
{
    time::PosixTime posix;
    auto r = time::StandardUserSystemClock::GetCurrentTime(&posix);
    if (!r.IsSuccess())
    {
        auto span = os::GetSystemTick().ToTimeSpan();
        auto l = util::SNPrintf(
            str, size,
            "[account] Uptime=%lldh%02lldm%02llds | ",
            span.GetHours(),
            span.GetMinutes() % 60,
            span.GetSeconds() % 60);
        NN_SDK_ASSERT(l < size);
        return l;
    }

    auto calendar = time::ToCalendarTimeInUtc(posix);
    auto l = util::SNPrintf(
        str, size,
        "[account] %04hd-%02hhd-%02hhd %02hhd:%02hhd:%02hhd | ",
        calendar.year, calendar.month, calendar.day,
        calendar.hour, calendar.minute, calendar.second);
    NN_SDK_ASSERT(l < size);
    return l;
}
} // ~namespace nn::<anonymous>
void PrintWithTimeStamp(PrintWithTimeStampKind kind, const char* pFormat, ...) NN_NOEXCEPT
{
    char buffer[256];
    auto l = GetTimeStamp(buffer, sizeof(buffer));

    std::va_list vaList;
    va_start(vaList, pFormat);
    auto ret = util::VSNPrintf(buffer + l, sizeof(buffer) - l, pFormat, vaList);
    va_end(vaList);
    NN_UNUSED(ret);

    switch (kind)
    {
    case nn::PrintWithTimeStampKind_Info:
        NN_DETAIL_ACCOUNT_INFO(buffer);
        break;
    case nn::PrintWithTimeStampKind_Warn:
        NN_DETAIL_ACCOUNT_WARN(buffer);
        break;
    case nn::PrintWithTimeStampKind_Error:
        NN_DETAIL_ACCOUNT_ERROR(buffer);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
#endif

} // ~namespace nn
