﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/dbg/dbg_Api.h>

#include <nn/fs/fs_Directory.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>
#include <algorithm>
#include <cstring>

#include "dmnt_Server.h"
#include "dmnt_GdbServer.h"

namespace
{
nn::lr::LocationResolver g_LocationResolver;
nn::Result GetProgramPath(char* pBuffer, size_t size, nn::ncm::ProgramId programId) NN_NOEXCEPT
{
    nn::lr::Path path;
    auto result = g_LocationResolver.ResolveProgramPath(&path, programId);
    std::memset(pBuffer, 0, size);
    std::memcpy(pBuffer, path.string, std::min(size, sizeof(path)));
    return result;
}
}

void Server::Initialize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(nn::pm::InitializeForDebugMonitor().IsSuccess());
    NN_ABORT_UNLESS(nn::ns::InitializeForDevelop().IsSuccess());
    nn::lr::Initialize();
    NN_ABORT_UNLESS(nn::lr::OpenLocationResolver(&g_LocationResolver, nn::ncm::StorageId::Host).IsSuccess());
}

nn::Result Server::DebugNewProcess(nn::sf::Out<nn::sf::NativeHandle> out, nn::sf::Out<nn::os::ProcessId> outId, nn::ncm::ProgramId programId, int32_t port) NN_NOEXCEPT
{
    nn::Result result;
    char path[nn::fs::EntryNameLengthMax];
    const char* fileName = path;

    result = GetProgramPath(path, sizeof(path), programId);
    if (result.IsFailure())
    {
        return result;
    }
    if (std::strlen(fileName) <= 4 ||
            (std::strcmp(fileName + std::strlen(fileName) - 4, ".nca") != 0 &&
             std::strcmp(fileName + std::strlen(fileName) - 4, ".NCA") != 0))
    {
        // NCA以外は nullptr
        fileName = nullptr;
    }

    nn::ns::ProgramLaunchProperty launchProperty;
    result = nn::ns::PrepareLaunchProgramFromHost(&launchProperty, fileName);
    if (result.IsFailure())
    {
        return result;
    }

    // Start なしでプログラム起動
    nn::os::ProcessId processId;
    result = nn::ns::LaunchProgram(&processId, launchProperty, nn::ns::LaunchProgramFlags_NotStart);
    if( result.IsFailure() )
    {
        return result;
    }

    // アタッチ
    nn::svc::Handle debugHandle;
    result = nn::dbg::DebugActiveProcess(&debugHandle, processId.value);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    // Start
    result = nn::pm::StartProcess(processId);
    if( result.IsFailure() )
    {
        return result;
    }

    nn::svc::Handle eventHandle;
    nn::dmnt::gdbserver::AttachProcess(&eventHandle, debugHandle, port, fileName);

    // 返り値セット
    outId.Set(processId);
    out.Set(nn::sf::NativeHandle(eventHandle.operator nnHandle().value, false));

    return nn::ResultSuccess();
}

nn::Result Server::AttachProcess(nn::sf::Out<nn::sf::NativeHandle> out, nn::os::ProcessId processId, int32_t port) NN_NOEXCEPT
{
    nn::svc::Handle eventHandle;
    nn::dmnt::gdbserver::AttachProcess(&eventHandle, processId.value, port);
    out.Set(nn::sf::NativeHandle(eventHandle.operator nnHandle().value, false));
    return nn::ResultSuccess();
}
