﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <mutex>
#include "dmnt_RspParser.h"
#include "dmnt_RspPacket.h"
#include "dmnt_DebugIO.h"
#include <nn/util/util_FormatString.h>
#include "gdbserver_log.h"

namespace nn { namespace dmnt { namespace rsp {
namespace {
const int CTRL_C = 3;

static const char* hexchars = "0123456789abcdef";
int HexToInteger(char ch)
{
    if (ch >= 'a' && ch <= 'f')
    {
        return ch-'a'+10;
    }
    if (ch >= '0' && ch <= '9')
    {
        return ch-'0';
    }
    if (ch >= 'A' && ch <= 'F')
    {
        return ch-'A'+10;
    }
    return -1;
}

}

void RspPacket::SetNoAck()
{
    mNoAck = true;
}

void RspPacket::SendPacket(bool* pBreak, const char* pSendBuffer, nn::dmnt::io::DebugSession *pSession)
{
    *pBreak = false;

    for (;;)
    {
        std::lock_guard<InternalMutex> lock(m_Mutex);

        size_t count = 0;
        uint8_t checksum = 0;
        int ch;

        while ((ch = pSendBuffer[count]) != '\0')
        {
            checksum += ch;
            count++;
        }

        char csumBuffer[4];
        csumBuffer[0] = '#';
        csumBuffer[1] = hexchars[checksum >> 4];
        csumBuffer[2] = hexchars[checksum & 0xf];
        csumBuffer[3] = '\0';

        char sendBuffer[BUFFER_SIZE + sizeof(csumBuffer) + 1];
        nn::util::SNPrintf(sendBuffer, sizeof(sendBuffer), "$%s%s", pSendBuffer, csumBuffer);

        GDB_TRACE_N("[dmnt0] RspPacket::SendPacket() line:%d, this=%p. '%s'\n", __LINE__, this, sendBuffer);
        if (pSession->PutDebugString(sendBuffer) == -1)
        {
            GDB_TRACE_E("[dmnt0] RspPacket::SendPacket() line:%d, this=%p. error '%s'\n", __LINE__, this, sendBuffer);
            return;
        }

        bool retransmit = false;
        if (!mNoAck) {
            do {
                ch = pSession->GetDebugChar();
                switch (ch)
                {
                case CTRL_C:
                    {
                        *pBreak = true;
                    }
                    return;
                case '+':
                    return;
                case -1:
                    return;
                case '-': // Retransmit
                    retransmit = true;
                    break;
                default:
                    break;
                }
            } while (!retransmit);
        }
        else
        {
            return;
        }
    }
}

char* RspPacket::ReceivePacket(bool* pBreak, char* pBuffer, size_t size, nn::dmnt::io::DebugSession *pSession)
{
    *pBreak = false;
    for (;;)
    {
        int ch;

        if (pSession->RecvPoll() < 0)
        {
            return NULL;
        }

        std::lock_guard<InternalMutex> lock(m_Mutex);
        if (pSession->RecvPoll(0) <= 0)
        {
            continue;
        }

        for (;;)
        {
            ch = pSession->GetDebugChar();
            if (ch == CTRL_C)
            {
                *pBreak = true;
                return NULL;
            }
            if (ch < 0)
            {
                return NULL;
            }
            if (ch == '$')
            {
                break;
            }
        }

        uint8_t checksum = 0;
        int count = 0;

        while (count < size - 1)
        {
            ch = pSession->GetDebugChar();
            if (ch == '$')
            {
                GDB_TRACE_N("[dmnt0] RspPacket::ReceivePacket() line:%d, this=%p. Double $$'\n", __LINE__, this);
                checksum = 0;
                count = 0;
                continue;
            }
            if (ch == '#')
            {
                break;
            }
            checksum = checksum + ch;
            pBuffer[count] = ch;
            count++;
        }

        pBuffer[count] = 0;

        if (ch == '#')
        {
            ch = pSession->GetDebugChar();
            int xmitcsumHigh = HexToInteger(ch);
            ch = pSession->GetDebugChar();
            int xmitcsumLow = HexToInteger(ch);

            uint8_t xmitcsum = (xmitcsumHigh << 4) + xmitcsumLow;

            if (mNoAck)
            {
                break;
            }
            else
            {
                if (xmitcsumHigh < 0 || xmitcsumLow < 0 || checksum != xmitcsum)
                {
                    pSession->PutDebugChar('-');
                }
                else
                {
                    pSession->PutDebugChar('+');
                    break;
                }
            }
        }
    }
    return pBuffer;
}

}}}

