﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <mutex>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/dmnt/dmnt_Result.h>
#include "dmnt_Rsp.h"
#include "dmnt_DebugMonitor.h"
#include <alloca.h>
#include <cstring>
#include "gdbserver_log.h"

#include "HardwareWatchPoints.h"
#include "HardwareBreakPoints.h"
#include "dmnt_ArraySize.h"

namespace nn { namespace dmnt {

bool HardwareWatchPoints::IsValidWatch(nn::Bit64 address, nn::Bit64 len)
{
    if (len == 0)
    {
        GDB_TRACE_E("[SetDataBreakpoint] Line:%d, Invalid len == 0\n", __LINE__);
        return false;
    }

    if (len <= 8)
    {
        nn::Bit64 base = address & 0xFFFFFFFFFFFFFFF8; // Double-word alignment is required
        if (base != ((address + len - 1) & 0xFFFFFFFFFFFFFFF8))
        {
            // Don't allow "len" bytes to span a double-word boundary
            GDB_TRACE_E("[SetDataBreakpoint] Line:%d, Misalignment Addrs:0x%llx, Base:0x%llx, len:0x%llx\n", __LINE__, address, base, len);
            return false;
        }
    }
    else
    {
        if (len > 0x80000000)
        {
            GDB_TRACE_E("[SetDataBreakpoint] Line:%d, Invalid len too large: 0x%llx == 0\n", __LINE__, len);
            return false;
        }

        // Len must be a power of 2.
        if ((len & (len - 1)) != 0)
        {   // not a power of two.
            GDB_TRACE_E("[SetDataBreakpoint] Line:%d, Invalid len must be a power of 2: 0x%llx == 0\n", __LINE__, len);
            return false;
        }

        // Address and length must align
        if ((address & (len - 1)) != 0)
        {
            GDB_TRACE_E("[SetDataBreakpoint] Line:%d, Misalignment Addrs:0x%llx, len:0x%llx\n", __LINE__, address, len);
            return false;
        }
    }

    return true;
}

nn::Result SetDataBreakpoint(
    nn::svc::HardwareBreakPointRegisterName regNo,
    nn::svc::HardwareBreakPointRegisterName regContext,
    nn::Bit64 address,
    nn::Bit64 len,
    bool readAccess,
    bool writeAccess) NN_NOEXCEPT
{
    uint8_t fieldLSC = (readAccess ? 1 : 0) | (writeAccess ? 2 : 0);
    uint8_t fieldBAS = 0;  // 8 bits (byte address select)
    uint8_t fieldMask = 0; // 5 bits (masking 3 to 31 bits of the address)

    if (fieldLSC != 0 && !HardwareWatchPoints::IsValidWatch(address, len))
    {
        return nn::svc::ResultInvalidArgument();
    }

    if (len <= 8)
    {
        nn::Bit64 base = address & 0xFFFFFFFFFFFFFFF8; // Double-word alignment is required
        // Set "len" bits
        fieldBAS = (1 << len) - 1;
        // Offset BAS
        int offset = (address & 7);
        fieldBAS = fieldBAS << offset;
        address = base;
    }
    else
    {
        // Len must be a power of 2.
        fieldBAS = 0xff;
        fieldMask = __builtin_popcount(len - 1);
    }
    // DBGBCR
    nn::Bit64 control =
        (fieldMask << 24)       | // MASK
        (regContext << 16) | // LBN
        (fieldBAS << 5)         |  // BAS
        (fieldLSC << 3)         | // LSC
        ((fieldLSC != 0) ? 1 : 0); // Enable

    //char bMask[32];
    //char bBAS[32];
    //char bLSC[32];
    //char bControl[34];
    //GDB_TRACE_T("[SetDataBreakpoint] Line:%d, reg:%d, address=0x%llx, Mask:%s, BAS:%s, LSC:%s, Control:%s\n", __LINE__, regNo, address, toBin(fieldMask, 5, bMask), toBin(fieldBAS, 8, bBAS), toBin(fieldLSC, 2, bLSC), toBin((uint32_t)control, 32, bControl));

    Result result = HardwareBreakPoints::SetHardwareBreakPoint(regNo, control, address);

    if (!result.IsSuccess())
    {
        GDB_TRACE_E("[SetDataBreakpoint] Line:%d, FAILED:0x%x, reg:%d, address=0x%llx\n", __LINE__, result.GetInnerValueForDebug(), regNo, address);
        return result;
    }
    return result;
}

nn::Result WatchPoint::Set(DebugProcess *debugProcess, uintptr_t addr, size_t len, bool readAccess, bool writeAccess)
{
    m_Addr = addr;
    m_Length = len;
    m_ReadAccess = readAccess;
    m_WriteAccess = writeAccess;

    Result result = HardwareBreakPoints::SetContextBreakpoint(m_RegContext, debugProcess->GetHandle());

    if (result.IsSuccess())
    {
        result = SetDataBreakpoint(m_RegNumber, m_RegContext, addr, len, readAccess, writeAccess);
    }

    if (result.IsSuccess())
    {
        m_InUse = true;
    }
    return result;
}

nn::Result WatchPoint::Clear(DebugProcess *debugProcess)
{
    nn::Result result;
    if (m_InUse)
    {
        GDB_TRACE_T("WatchPoint::ClearBreakPoint @0x%llx\n", m_Addr);
        result = SetDataBreakpoint(m_RegNumber, m_RegContext, 0, 0, false, false);
        Reset();
    }
    return result;
}

HardwareWatchPoints::HardwareWatchPoints(DebugProcess *debugProcess)
: BreakPointManagerBase(debugProcess)
{
    const nn::svc::HardwareBreakPointRegisterName contextReg = HardwareBreakPoints::GetWatchPointContextRegister();
    for (int i = 0; i < ARRAYSIZE(m_BreakPoints); ++i)
    {
        m_BreakPoints[i].Initialize( (nn::svc::HardwareBreakPointRegisterName) (nn::svc::HardwareBreakPointRegisterName_D0 + i), contextReg);
    }
}

BreakPointBase *HardwareWatchPoints::GetBreakPoint(unsigned index)
{
    if (index < ARRAYSIZE(m_BreakPoints))
    {
        return &m_BreakPoints[index];
    }
    return nullptr;
}

WatchPoint *HardwareWatchPoints::GetFreeBreak()
{
    for (unsigned i = 0; GetBreakPoint(i) != nullptr; ++i)
    {
        WatchPoint *breakPoint((WatchPoint*)GetBreakPoint(i));
        if (!breakPoint->m_InUse)
        {
            return breakPoint;
        }
    }
    return nullptr;
}

nn::Result HardwareWatchPoints::GetWatchPointInfo(nn::Bit64 addr, bool &readAccess, bool &writeAccess) NN_NOEXCEPT
{
    for (unsigned i = 0; GetBreakPoint(i) != nullptr; ++i)
    {
        WatchPoint *breakPoint((WatchPoint*)GetBreakPoint(i));
        if (breakPoint->m_InUse)
        {
            if (addr >= breakPoint->m_Addr && addr < (breakPoint->m_Addr + breakPoint->m_Length))
            {
                readAccess = breakPoint->m_ReadAccess;
                writeAccess = breakPoint->m_WriteAccess;
                return nn::ResultSuccess();
            }
        }
    }
    readAccess = false;
    writeAccess = false;
    GDB_TRACE_E("[HardwareWatchPoints::GetWatchPointInfo] Failed, addr: 0x%llx\n", addr);
    #if defined(NN_DETAIL_ENABLE_SDK_LOG)
    for (unsigned i = 0; GetBreakPoint(i) != nullptr; ++i)
    {
        WatchPoint *breakPoint((WatchPoint*)GetBreakPoint(i));
        GDB_TRACE_E("[HardwareWatchPoints::GetWatchPointInfo] Failed, i:%d, inUse:%d, addr:0x%llx, end:0x%llx, len:0x%llx\n", i, breakPoint->m_InUse, breakPoint->m_Addr, (breakPoint->m_Addr + breakPoint->m_Length), breakPoint->m_Length);
    }
    #endif
    return nn::svc::ResultInvalidArgument();
}

nn::Result HardwareWatchPoints::SetWatchPoint(nn::Bit64 addr, nn::Bit64 len, bool readAccess, bool writeAccess) NN_NOEXCEPT
{
    auto breakPoint = GetFreeBreak();

    if (breakPoint)
    {
        return breakPoint->Set(m_pDebugProcess, addr, len, readAccess, writeAccess);
    }

    // When there are too many break points
    return nn::svc::ResultMaxHandle();
}

}}


