﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/pwm/pwm_ShimInternal.h>
#include <nn/pwm/driver/pwm_Lib.h>
#include <nn/pwm/server/pwm_Server.h>

#include "../detail/Bus_Log.h"
#include "Bus_PwmServer.h"

namespace nn { namespace bus { namespace pwm {

namespace
{
    const size_t ThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    //! サーバのループスレッド
    void ServerThread(void *)
    {
        // 追加のモジュール初期化処理（他モジュールへの依存のためにメインスレッド内では呼ばないほうが良いもの）
        nn::pwm::driver::Initialize();
        nn::pwm::server::StartHipcServer();

        NN_DETAIL_BUS_INFO("[Bus] Entering Pwm Server Loop\n");
        nn::pwm::server::LoopHipcServer();

        // モジュール終了処理
        nn::pwm::driver::Finalize();
    };
}

//! プロセスメインスレッド内で早い段階で同期的に済ませておく必要がある処理（プロセス内他スレッドが起動直後に API を叩いて来たら競合しうるもの）
void InitializeServer() NN_NOEXCEPT
{
    // sm への HIPC サーバの登録
    nn::pwm::server::RegisterHipcServer();

    // プロセス内シムへのサービスオブジェクト登録
    nn::pwm::InitializeWith(nn::pwm::server::GetServiceObject());
}

void StartServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_Thread, ServerThread, nullptr, g_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(pwm, IpcServer))
    );
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(pwm, IpcServer));
    nn::os::StartThread(&g_Thread);
}

void StopServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    nn::pwm::server::StopHipcServer();
}

void WaitAndFinalizeServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}
