﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_VibratorBase.h"

#ifdef VIBRATION_LOG_ENABLE

#include <nn/xcd/detail/xcd_Log.h>

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_VIB_TRACE(...)            NN_DETAIL_XCD_TRACE("VibratorBase: " ##__VA_ARGS__)
#else
#define NN_VIB_TRACE(format, ...)    NN_DETAIL_XCD_TRACE("VibratorBase: " format, ##__VA_ARGS__)
#endif  // ifdef NN_BUILD_CONFIG_COMPILER_VC

#else

#define NN_VIB_TRACE(...)            static_cast<void>(0)

#endif

namespace nn { namespace xcd {

const size_t VibratorBase::InputReportSize = 1;
const size_t VibratorBase::OutputReportSize = 8;
const int VibratorBase::InputReportBitShiftLeft = 4;
const int VibratorBase::InputReportBitShiftRight = 0;
const int VibratorBase::OutputReportOffsetLeft = 0;
const int VibratorBase::OutputReportOffsetRight = 4;

VibratorBase::~VibratorBase() NN_NOEXCEPT
{
    // 何もしない
}

void VibratorBase::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    NN_VIB_TRACE("%s Type=%d FW=%d P=%p\n", NN_CURRENT_FUNCTION_NAME, type, firmwareVersion, this);

    // Joy-Conの場合、左右それぞれにVibratorBaseが割り当てられる。
    // ProConの場合、左右で１つVibratorBaseが割り当てられる。

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_Activated = true;
    m_Type = type;
    m_FirmwareVersion = firmwareVersion;

    // 7bit AMFM 符号を利用可能なバージョンか否かの判別
    bool isAmFm7bitCodeAvailable = m_FirmwareVersion.bluetooth >= FirmwareVersionBt_01f4;

    // 電源状態
    m_PowerStatusLeft = PowerStatus::Disabled;
    m_PowerStatusRight = PowerStatus::Disabled;
    m_IsPowerDisableScheduled = true;

    // 左手用振動子の初期化
    if(m_Type == DeviceType_Left || m_Type == DeviceType_FullKey)
    {
        m_VibratorAgentLeft.SetVibrationOnConnect(&m_VibrationOnConnectLeft);
        m_VibratorAgentLeft.Activate(m_Type);
        m_VibratorAgentLeft.SetAmFm7bitCodeAvailable(isAmFm7bitCodeAvailable);
    }

    // 右手用振動子の初期化
    if(m_Type == DeviceType_Right || m_Type == DeviceType_FullKey)
    {
        m_VibratorAgentRight.SetVibrationOnConnect(&m_VibrationOnConnectRight);
        m_VibratorAgentRight.Activate(m_Type);
        m_VibratorAgentRight.SetAmFm7bitCodeAvailable(isAmFm7bitCodeAvailable);
    }

    if(m_Type == DeviceType_MiyabiLeft)
    {
        m_VibratorAgentLeft.SetVibrationOnConnect(&m_VibrationOnConnectLeft);
        m_VibratorAgentLeft.Activate(m_Type);
        m_VibratorAgentLeft.SetAmFm7bitCodeAvailable(true);
    }

    if(m_Type == DeviceType_MiyabiRight)
    {
        m_VibratorAgentRight.SetVibrationOnConnect(&m_VibrationOnConnectRight);
        m_VibratorAgentRight.Activate(m_Type);
        m_VibratorAgentRight.SetAmFm7bitCodeAvailable(true);
    }
}

void VibratorBase::Deactivate() NN_NOEXCEPT
{
    NN_VIB_TRACE("%s Type=%d p=%p \n", NN_CURRENT_FUNCTION_NAME, m_Type,this);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    // 左手用振動子の終了処理
    if(m_Type == DeviceType_Left || m_Type == DeviceType_FullKey || m_Type == DeviceType_MiyabiLeft)
    {
        m_VibratorAgentLeft.Deactivate();
    }

    // 右手用振動子の終了処理
    if(m_Type == DeviceType_Right || m_Type == DeviceType_FullKey || m_Type == DeviceType_MiyabiRight)
    {
        m_VibratorAgentRight.Deactivate();
    }

    // 電源 OFF
    m_pCommand->MotorEnable(false, this);
    m_PowerStatusLeft = PowerStatus::Disabled;
    m_PowerStatusRight = PowerStatus::Disabled;
}

void VibratorBase::ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_ASSERT_EQUAL(size, InputReportSize);
    NN_UNUSED(sampleNumber);    // TDB: サンプル番号が前後した場合の処理
    NN_UNUSED(size);

    uint8_t report = *pBuffer;
    if(m_Type == DeviceType_Left || m_Type == DeviceType_MiyabiLeft)
    {
        // Right 用の 4 ビットを Left 用の 4 ビット領域にコピー（古いファームへの対策）
        report |= (report << InputReportBitShiftLeft);
    }

    m_VibratorAgentLeft.ParseInputReport(report >> InputReportBitShiftLeft);
    m_VibratorAgentRight.ParseInputReport(report >> InputReportBitShiftRight);
}

size_t VibratorBase::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_ASSERT_EQUAL(size, OutputReportSize);

    int reportSizeL = m_VibratorAgentLeft.GetOutputReport(pOutValue + OutputReportOffsetLeft);
    int reportSizeR = m_VibratorAgentRight.GetOutputReport(pOutValue + OutputReportOffsetRight);

    // もう一方の 4 バイトにもデータをコピーしておく（古いファームへの対策）
    if(m_Type == DeviceType_Left || m_Type == DeviceType_MiyabiLeft)
    {
        memcpy(
            pOutValue + OutputReportOffsetRight,
            pOutValue + OutputReportOffsetLeft,
            OutputReportSize / 2);
    }
    else if(m_Type == DeviceType_Right || m_Type == DeviceType_MiyabiRight)
    {
        memcpy(
            pOutValue + OutputReportOffsetLeft,
            pOutValue + OutputReportOffsetRight,
            OutputReportSize / 2);
    }

    if(reportSizeL + reportSizeR > 0)
    {
        return size;
    }

    return 0;
}

void VibratorBase::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    NN_VIB_TRACE("S %s Result=%d id=%d Type=%d Left=%d Right=%d p=%p\n", NN_CURRENT_FUNCTION_NAME, result, id, m_Type, m_PowerStatusLeft, m_PowerStatusRight, this);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(result.IsSuccess())
    {
        if(id == Command_MotorEnable.Id)
        {
            // 電源 ON コマンドの Ack を受け取った
            if(m_PowerStatusLeft == PowerStatus::EnableRequested)
            {
                m_PowerStatusLeft = PowerStatus::Enabled;
                m_VibratorAgentLeft.Activate(m_Type);
                m_VibrationOnConnectLeft.SetDevicePowerStatus(true);
            }
            if(m_PowerStatusRight == PowerStatus::EnableRequested)
            {
                m_PowerStatusRight = PowerStatus::Enabled;
                m_VibratorAgentRight.Activate(m_Type);
                m_VibrationOnConnectRight.SetDevicePowerStatus(true);
            }

            // 電源 OFF コマンドの Ack を受け取った
            if(m_PowerStatusLeft == PowerStatus::DisableRequested)
            {
                m_PowerStatusLeft = PowerStatus::Disabled;
                m_VibratorAgentLeft.Deactivate();
                m_VibrationOnConnectLeft.SetDevicePowerStatus(false);
            }
            if(m_PowerStatusRight == PowerStatus::DisableRequested)
            {
                m_PowerStatusRight = PowerStatus::Disabled;
                m_VibratorAgentRight.Deactivate();
                m_VibrationOnConnectRight.SetDevicePowerStatus(false);
            }
        }
    }

    NN_VIB_TRACE("E %s Result=%d id=%d Type=%d Left=%d Right=%d p=%p\n", NN_CURRENT_FUNCTION_NAME, result, id, m_Type, m_PowerStatusLeft, m_PowerStatusRight, this);
}

Result VibratorBase::CheckExistence(VibratorPosition position) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(position == VibratorPosition_Left && (m_Type == DeviceType_Left || m_Type == DeviceType_FullKey || m_Type == DeviceType_MiyabiLeft))
    {
        NN_RESULT_SUCCESS;
    }

    if(position == VibratorPosition_Right && (m_Type == DeviceType_Right || m_Type == DeviceType_FullKey || m_Type == DeviceType_MiyabiRight))
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultVibratorNotExist());
}

Result VibratorBase::SetEnabled(bool isEnabled, VibratorPosition position) NN_NOEXCEPT
{
    NN_VIB_TRACE("S %s %s pos=%d Type=%d Left=%d Right=%d　p=%p\n", NN_CURRENT_FUNCTION_NAME, isEnabled ? "True" : "False", position, m_Type, m_PowerStatusLeft, m_PowerStatusRight,this);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_RESULT_DO(CheckExistence(position));

    VibratorPosition opposite = (position == VibratorPosition_Left) ? VibratorPosition_Right : VibratorPosition_Left;
    PowerStatus& refPower = RefPowerStatus(position);           // 指定された振動子の電源状態
    PowerStatus& refPowerOpposite = RefPowerStatus(opposite);   // もう一方の振動子の電源状態

    if(isEnabled == true)
    {
        m_IsPowerDisableScheduled = false;

        if(refPower != PowerStatus::Enabled && refPower != PowerStatus::EnableRequested)
        {
            //アンプの電源状態がOFFの時のみ受け付ける
            if(refPowerOpposite == PowerStatus::Enabled)
            {
                // 両方の振動子でアンプ電源は共通なので、もう一方の振動子で Enabled ならば実際のアンプ電源は既に投入済み。
                // HidCommand を新規に発行する必要はないので内部状態だけ変更しておく。
                refPower = PowerStatus::Enabled;
            }
            else
            {
                // 電源 ON コマンド送信
                m_pCommand->MotorEnable(true, this);
                refPower = PowerStatus::EnableRequested;
                NN_VIB_TRACE("%s MotorEnable Enable\n", NN_CURRENT_FUNCTION_NAME);
            }
        }
    }
    else
    {
        if (refPower != PowerStatus::Disabled && refPower != PowerStatus::DisableRequested)
        {
            //自身のアンプの電源状態がONの時のみ受け付ける
            if(refPowerOpposite == PowerStatus::Disabled || refPowerOpposite == PowerStatus::DisableRequested)
            {
                // アンプの電源を切るのは両方の振動子で Disable or DisableRequested が指定されたときのみ
                // 注：Proコンの場合、左右の振動子のアンプ制御は HW 的に連動している
                if (m_VibrationOnConnectLeft.IsPlaying() || m_VibrationOnConnectRight.IsPlaying())
                {
                    // 接続時振動を再生中の場合は、後で電源 OFF する予定だけを設定
                    m_IsPowerDisableScheduled = true;
                }
                else
                {
                    // 電源 OFF コマンド送信
                    m_pCommand->MotorEnable(false, this);
                    refPower = PowerStatus::DisableRequested;
                    NN_VIB_TRACE("%s MotorEnable Disable\n", NN_CURRENT_FUNCTION_NAME);
                }
            }
            else
            {
                // 反対側の振動子がONの場合、自分は DisableRequested に遷移して、反対側の振動子（左の場合は右）で電源 Off コマンド送信を行う
                refPower = PowerStatus::DisableRequested;
            }
        }
    }

    NN_VIB_TRACE("E %s %s pos=%d Type=%d Left=%d Right=%d　p=%p\n", NN_CURRENT_FUNCTION_NAME, isEnabled ? "True" : "False", position, m_Type, m_PowerStatusLeft, m_PowerStatusRight, this);

    NN_RESULT_SUCCESS;
}

Result VibratorBase::IsEnabled(bool* pOutValue, VibratorPosition position) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(CheckExistence(position));

    *pOutValue = (RefPowerStatus(position) == PowerStatus::Enabled);

    if (m_VibrationOnConnectLeft.IsPlaying() || m_VibrationOnConnectRight.IsPlaying())
    {
        // 接続時振動を再生中の場合は、再生終了後に予定されている電源ステータスを考慮
        *pOutValue &= !m_IsPowerDisableScheduled;
    }

    NN_RESULT_SUCCESS;
}

Result VibratorBase::GetVibratorAgent(nn::xcd::VibratorAgent** pOutValue, VibratorPosition position) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = nullptr;
    NN_RESULT_DO(CheckExistence(position));
    NN_RESULT_THROW_UNLESS(RefPowerStatus(position) == PowerStatus::Enabled, ResultVibratorNotEnabled());

    *pOutValue = &RefVibratorAgent(position);
    NN_RESULT_SUCCESS;
}

void VibratorBase::SetInterfaceType(InterfaceType interfaceType) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_InterfaceType = interfaceType;
}

void VibratorBase::StartVibrationOnConnect() NN_NOEXCEPT
{
    if (m_Type == DeviceType_Left || m_Type == DeviceType_MiyabiLeft)
    {
        m_pCommand->MotorEnable(true, this);
        m_PowerStatusLeft = PowerStatus::EnableRequested;
        m_VibrationOnConnectLeft.StartPlaying(VibrationPatternOnConnect_JoyCon);
    }
    else if (m_Type == DeviceType_Right || m_Type == DeviceType_MiyabiRight)
    {
        m_pCommand->MotorEnable(true, this);
        m_PowerStatusRight = PowerStatus::EnableRequested;
        m_VibrationOnConnectRight.StartPlaying(VibrationPatternOnConnect_JoyCon);
    }
    else if (m_Type == DeviceType_FullKey)
    {
        m_pCommand->MotorEnable(true, this);
        m_PowerStatusLeft = PowerStatus::EnableRequested;
        m_PowerStatusRight = PowerStatus::EnableRequested;
        m_VibrationOnConnectLeft.StartPlaying(VibrationPatternOnConnect_FullConLeft);
        m_VibrationOnConnectRight.StartPlaying(VibrationPatternOnConnect_FullConRight);
    }
}

void VibratorBase::NotifyVibrationOnConnectFinished() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!m_VibrationOnConnectLeft.IsPlaying() && !m_VibrationOnConnectRight.IsPlaying())
    {
        // 必要に応じてアンプの電源を OFF
        if (m_IsPowerDisableScheduled)
        {
            (void)SetEnabled(false, VibratorPosition_Left);
            (void)SetEnabled(false, VibratorPosition_Right);
        }
    }
}

}} // namespace nn::xcd
