﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_Peripheral.h"
#include "xcd_TeraStateMachine.h"
#include "xcd_ITeraUpdater.h"
#include "xcd_NfcProcessor.h"
#include "xcd_IrsensorBase.h"
#include "detail/xcd_HidAccessor.h"
#include "detail/xcd_TeraCommon.h"
#include "detail/xcd_TeraFirmwareVersion.h"
#include "detail/xcd_TeraUpdater.h"
#include "detail/xcd_TeraUpdaterLegacy.h"
#include "detail/xcd_TimeCounter.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
    #pragma warning(push)
    #pragma warning(disable: 4351)
#endif

namespace nn { namespace xcd {

/**
 * @brief   Tera MCU の実装のためのベースとなるクラス
 */
class TeraBase final : public Peripheral, public ICommandListener
{
    NN_DISALLOW_COPY(TeraBase);
    NN_DISALLOW_MOVE(TeraBase);

public:
    // SetDataFormat の実処理を行う関数
    typedef void (*DataFormatFunctionType)(PeriodicDataFormat, void*);

public:
    TeraBase() NN_NOEXCEPT :
        m_Mutex(true),
        m_CommonResponseStatus(),
        m_pOutputReporter(nullptr),
        m_McuFirmwareVersion(),
        m_IsVersionReading(false),
        m_IsMcuFirmwareUpdating(false),
        m_pUpdateFinishEvent(nullptr),
        m_ReservedAction(),
        m_pSetDataFormatFunction(nullptr),
        m_pSetDataFormatArgument(nullptr),
        m_IapBrokenCheckCounter(),
        m_GetVersionTimeoutCounter(),
        m_pHidAccessor(nullptr),
        m_StateMachine(),
        m_pFirmwareUpdater(nullptr),
        m_NfcProcessor(),
        m_IrsensorBase(),
        m_FirmwareUpdaterStorage() {}

    /**
     * @brief   MCU 破損エミュレーションの有効状態を設定します。
     */
    static void SetMcuBrokenEmulationEnabled(bool isEnabled) NN_NOEXCEPT;

    /**
     * @brief   FW 更新失敗エミュレーションの有効状態を設定します。
     */
    static void SetFirmwareUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT;

    virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT NN_OVERRIDE;
    virtual size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    void SetHidAccessor(detail::HidAccessor* pHidAccessor) NN_NOEXCEPT;

    /**
     * @brief Firmware Update 中に受け取ったデータをパースします
     */
    void ParseMcuUpdateInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief Firmware Update でコントローラーに送信するデータを取得します。
     */
    size_t GetMcuUpdateOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   TeraMcu の現在の状態を取得します
     */
    McuState GetMcuState() NN_NOEXCEPT;

    /**
     * @brief   TeraMcu の状態を設定します
     */
    nn::Result SetMcuState(McuState mcuState) NN_NOEXCEPT;

    /**
     * @brief   MCU Read/Write を使用して、DataFormat によらず TeraMcu の状態を設定します
     */
    nn::Result SetMcuStateImmediate(McuState mcuState) NN_NOEXCEPT;

    /**
     * @brief   高速 FW 更新が可能か判定
     */
    bool IsAvailableFastFirmwareUpdate() NN_NOEXCEPT;

    /**
     * @brief   FW バージョンの取得要求を出す
     */
    nn::Result RequestMcuFirmwareVersion() NN_NOEXCEPT;

    /**
     * @brief   FW バージョン情報を取得
     */
    nn::Result GetMcuFirmwareVersion(McuVersionData* pOutVersion) NN_NOEXCEPT;

    /**
     * @brief   FW バージョン情報を取得 (NFC 用)
     */
    nn::Result GetMcuFirmwareVersionForNfc(McuVersionDataForNfc* pOutVersion) NN_NOEXCEPT;

    /**
     * @brief   FW アップデートを開始
     */
    nn::Result StartMcuFirmwareUpdate(
        const FirmwareImage& image,
        nn::os::SystemEventType* pFinishEvent,
        ITeraUpdater::UpdateMode updateMode) NN_NOEXCEPT;

    /**
     * @brief   DataFormat を変更するための関数を登録
     */
    void SetChangeDataFormatFunction(DataFormatFunctionType function, void* pArgument) NN_NOEXCEPT;

    /**
     * @brief   NFC の状態通知イベントを登録
     *
     * @param[out]  pOutCommonHandle    NFC のタグ検出関連以外を通知するイベントのハンドル
     * @param[out]  pOutDetectHandle    NFC のタグ検出関連を通知するイベントのハンドル
     */
    void SetNfcEvent(nn::os::NativeHandle* pOutCommonHandle, nn::os::NativeHandle* pOutDetectHandle) NN_NOEXCEPT;

    /**
     * @brief   アップデータへのアクセス
     *
     * @param[in] function  アップデータへの操作を行う関数
     *
     * @return 処理結果 (具体的な値は function に依存)
     */
    template <class ProcessFunction>
    nn::Result ProcessUpdater(ProcessFunction function) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        NN_RESULT_DO(function(m_pFirmwareUpdater));
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   NFC コマンドの実行
     *
     * @param[in] function  コマンド発行処理を行う関数
     *
     * @return コマンド発行結果 (具体的な値は function に依存)
     */
    template <class ProcessFunction>
    nn::Result ProcessNfcCommand(ProcessFunction function) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        NN_RESULT_DO(m_StateMachine.CheckMcuState(McuState_Nfc));
        NN_RESULT_DO(function(&m_NfcProcessor));
        m_pOutputReporter = &m_NfcProcessor;

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   IR センサーコマンドの実行
     *
     * @param[in] function  コマンド発行処理を行う関数
     *
     * @return コマンド発行結果 (具体的な値は function に依存)
     */
    template <class ProcessFunction>
    nn::Result ProcessIrCommand(ProcessFunction function) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        NN_RESULT_DO(m_StateMachine.CheckMcuState(McuState_Ir));
        NN_RESULT_DO(function(&m_IrsensorBase));
        m_pOutputReporter = &m_IrsensorBase;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief   EXT_CONTROL コマンドの実行
    *
    * @param[in] isOn  EXT_CONTROL の ON / OFF
    *
    * @return コマンド発行結果
    */
    nn::Result ExtControl(bool isOn, ICommandListener* pListener) NN_NOEXCEPT;

private:
    /**
     * @brief   優先実行するために予約するアクション
     */
    enum class ReservedActionType
    {
        StartFwUpdate,      //!< FW アップデート開始
    };

    /**
     * @brief   予約アクションの設定
     */
    struct ReservedAction
    {
        bool                isEnabled;      //!< 予約済みフラグ
        ReservedActionType  action;         //!< 予約されたアクション
        detail::TimeCounter timer;          //!< 発動までの待機時間計測カウンタ

        /**
         * @brief   アクションの設定
         */
        void Set(ReservedActionType actionType, nn::TimeSpan delay) NN_NOEXCEPT
        {
            action    = actionType;
            isEnabled = true;
            timer.Start(delay);
        }

        /**
         * @brief   待機時間満了判定
         */
        bool IsTimerExpired() const NN_NOEXCEPT
        {
            return timer.IsExpired();
        }

        /**
         * @brief   アクションの解除
         */
        void Clear() NN_NOEXCEPT
        {
            isEnabled = false;
        }
    };

    /**
     * @brief   受信した Input report の情報
     */
    struct InputReportBasicInfo
    {
        bool isValid;                           //!< 有効なデータか
        bool needsParse;                        //!< プロトコル毎の解析が必要か
        detail::ReceiveProtocolType protocol;   //!< プロトコルの種別
    };

    // デバッグ用の設定
    struct DebugOption
    {
        typedef ::nn::util::BitFlagSet<32, DebugOption>::Flag<0> McuBrokenEmulation;
    };

    // DebugOption の集合を扱う型
    typedef ::nn::util::BitFlagSet<32, DebugOption> DebugOptionSet;

private:
    static DebugOptionSet       g_DebugOptions;             //!< デバッグ用の設定

    nn::os::Mutex               m_Mutex;                    //!< メンバへのアクセスを排他するための Mutex
    detail::CommonResponseType  m_CommonResponseStatus;     //!< 前回の通信タイミングで受信した Common コマンドの最新の Response 状態
    Peripheral*                 m_pOutputReporter;
    detail::TeraFirmwareVersion m_McuFirmwareVersion;       //!< Tera MCU FW のバージョン
    bool                        m_IsVersionReading;         //!< FW バージョン読み出し中
    bool                        m_IsMcuFirmwareUpdating;    //!< FW アップデート中
    nn::os::SystemEventType*    m_pUpdateFinishEvent;       //!< FW アップデート終了を通知するイベント
    ReservedAction              m_ReservedAction;           //!< 優先実行するアクション
    DataFormatFunctionType      m_pSetDataFormatFunction;   //!< データフォーマットを変更する関数のポインタ
    void*                       m_pSetDataFormatArgument;   //!< データフォーマット変更関数に渡すポインタ引数
    detail::TimeCounter         m_IapBrokenCheckCounter;    //!< IAP 破損判定用カウンタ
    detail::TimeCounter         m_GetVersionTimeoutCounter; //!< FW バージョン取得中断判定用カウンタ

    detail::HidAccessor*        m_pHidAccessor;             //!< HidAccessor
    TeraStateMachine            m_StateMachine;             //!< Tera MCU の状態を管理します
    ITeraUpdater*               m_pFirmwareUpdater;         //!< Tera MCU FW を更新するためのモジュール
    NfcProcessor                m_NfcProcessor;             //!< NFC 制御用のインスタンス
    IrsensorBase                m_IrsensorBase;             //!< IR センサー制御用のインスタンス

    char m_FirmwareUpdaterStorage[
        sizeof(detail::TeraUpdater)];                       //!< FW 更新モジュールを格納する領域

    // FW 更新モジュールのインスタンスがバッファ溢れしないことを保証
    NN_STATIC_ASSERT(
        sizeof(detail::TeraUpdater) >= sizeof(detail::TeraUpdaterLegacy));

private:
    /**
     * @brief   ステート遷移通知を受け取るコールバック
     */
    static void StateTransitionCallback(detail::InternalMcuState state, ICommandListener* pListener) NN_NOEXCEPT;

    /**
     * @brief   アップデートの進捗通知を受け取るコールバック
     */
    static void FirmwareUpdateCallback(ITeraUpdater::UpdatePhase phase, void* pArgument) NN_NOEXCEPT;

private:
    /**
     * @brief   TeraMcu に接続されたペリフェラルを起動
     */
    nn::Result ActivatePeripheral(detail::InternalMcuState nextMcuState) NN_NOEXCEPT;

    /**
     * @brief   TeraMcu に接続されたペリフェラルを終了
     */
    void DeactivatePeripheral() NN_NOEXCEPT;

    /**
     * @brief   TeraMcu の FW 更新モジュールを起動
     */
    void ActivateFirmwareUpdater() NN_NOEXCEPT;

    /**
     * @brief   TeraMcu の FW 更新モジュールを終了
     */
    void DeactivateFirmwareUpdater() NN_NOEXCEPT;

    /**
     * @brief   TeraMcu の状態を設定します
     */
    nn::Result SetMcuStateImpl(McuState mcuState, bool isHidCommandOnly) NN_NOEXCEPT;

    /**
     * @brief   McuWrite で TERA_CONTROL NOP を発行
     */
    void SendNopCommandByMcuWrite() NN_NOEXCEPT;

    /**
     * @brief   McuRead で受信したパケットの解析
     */
    void ParseInputByMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   IAP 破損判定の更新
     */
    void UpdateIapBrokenCheck(bool allowStart) NN_NOEXCEPT;

    /**
     * @brief   IAP 破損判定を停止
     */
    void StopIapBrokenCheck() NN_NOEXCEPT
    {
        m_IapBrokenCheckCounter.Stop();
        m_GetVersionTimeoutCounter.Stop();
    }

    /**
     * @brief   受信パケット解析処理
     */
    bool ParseInputReportImpl(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief   受信パケットの基本的な情報を取得
     *
     * @param[out]  pOutInfo        解析した情報
     * @param[in]   bluetoothMode   データサイズを規定する @ref detail::BluetoothMode
     * @param[in]   pBuffer         パケットのデータ
     * @param[in]   size            パケットのデータサイズ
     * @param[in]   sampleNumber    パケットのサンプル番号
     */
    void ParseInputReportBasicInfo(
        InputReportBasicInfo* pOutInfo,
        detail::BluetoothMode bluetoothMode,
        const uint8_t* pBuffer,
        size_t size,
        uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief   Common プロトコルの受信パケットを解析
     */
    void ParseCommonInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief   IAP 動作時の受信データを解析
     */
    void ParseIapInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

    /**
     * @brief   IAP 動作時のリザルトコードに応じた処理
     */
    void DispatchIapResult(uint8_t resultCode) NN_NOEXCEPT;

    /**
     * @brief   IAP 破損時の処理
     */
    void ProcessIapCorruption() NN_NOEXCEPT;

    /**
     * @brief   Common プロトコルの送信パケットを取得
     */
    size_t GetCommonOutput(uint8_t* pOutValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   ステート遷移時の処理
     */
    void ProcessStateTransition(detail::InternalMcuState state) NN_NOEXCEPT;

    /**
     * @brief   FW アップデート進行時の処理
     */
    void ProcessFirmwareUpdateForward(ITeraUpdater::UpdatePhase phase) NN_NOEXCEPT;

    /**
     * @brief   FW アップデートの完了を通知
     */
    void NotifyFirmwareUpdateFinished() NN_NOEXCEPT;

    /**
     * @brief   内部状態をクリアする
     */
    void ClearInternalState() NN_NOEXCEPT;

    /**
     * @brief   FW バージョン取得完了を通知
     */
    void NotifyGotMcuFirmwareVersion() NN_NOEXCEPT;

    /**
     * @brief   FW バージョンの有効状態を確認
     */
    nn::Result CheckFirmwareVersionValidity() NN_NOEXCEPT;

    /**
     * @brief   強制アクションの処理
     */
    bool ProcessReservedAction() NN_NOEXCEPT;
};

}} // namespace nn::xcd

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    #pragma warning(pop)
#endif
