﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include "xcd_IrsensorBase.h"
#include "xcd_IrsensorImageTransferProcessor.h"

namespace nn { namespace xcd {

const IrsensorImageTransferProcessor::ImageSizeInfo IrsensorImageTransferProcessor::TransferSizeInfo[] =
{
    {320, 240, 255},  //!< QVGA (320x240)
    {160, 120, 63},   //!< QQVGA (160x120)
    {80,  60,  15},   //!< QQQVGA (80x60)
    {40,  30,  3},    //!< QQQQVGA (40x30)
    {20,  15,  0},    //!< QQQQQVGA (20x15)
};

IrsensorImageTransferProcessor::~IrsensorImageTransferProcessor() NN_NOEXCEPT
{
}

Result IrsensorImageTransferProcessor::ParseImageTransferData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ヘッダの読み込み
    ParseIrsensorHeader(pBuffer, size, sampleNumber);

    if (m_ResultCode == static_cast<int>(IrsensorMcuPacketResult::DataNotReady))
    {
        // ImageTransfer モードの初期化中の場合は何もしない。
        NN_XCD_IRSENSOR_LOG("[ImageTransfer] Mcu DataNotReady\n");
    }

    m_ImageId = m_IrCommonData[0].frameId;
    NN_XCD_IRSENSOR_LOG("[ImageTransfer] ImageId: %d timestamp: %d diff:%d\n", m_ImageId, m_TimeStamp, m_DiffTimeStamp);

    if (m_TransferProtocol == IrImageTransferProtocol::FastMode)
    {
        // フレーム更新判定
        if (m_DiffTimeStamp != 0)
        {
            // フレームが更新されたら All FF にリセットする
            NN_XCD_IRSENSOR_LOG("[ImageTransfer] Flag Reset To ALL 1 \n");
            m_ReceivedFrameFlag.Set();
            m_IsImageReady = false;
            m_IsCompletionCommandAvailable = true;
            m_ContinuousFrameCount = 0;
        }
        else
        {
            m_ContinuousFrameCount++;
        }

        // 書き込み可能かどうか
        if (m_ReceivedFrameFlag.CountPopulation() != IMAGE_TRANSFER_FRAME_COUNT_MAX - (GetTransferPacketCount(m_ImageTransferFormat) + 1))
        {

            uint8_t* pImageBuffer = static_cast<uint8_t*>(m_pImageTransferProcessorState->pImage);
            if (m_ReceivedFrameFlag.Test(m_ImageId))
            {
                // 未コピーであればコピーする
                memcpy((pImageBuffer + ImageTransferProcessorInputReportSize * m_ImageId), pBuffer + ImageTransferProcessorInputReportOffset, ImageTransferProcessorInputReportSize);
                m_ReceivedFrameFlag.Reset(m_ImageId);
            }

            if (m_ReceivedFrameFlag.CountPopulation() == IMAGE_TRANSFER_FRAME_COUNT_MAX - (GetTransferPacketCount(m_ImageTransferFormat) + 1))
            {
                NN_XCD_IRSENSOR_LOG("[ImageTransfer] Last packet arrived\n");
                m_IsImageReady = true;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(nn::xcd::ResultIrImageTransferBusy());
    }
    else if (m_TransferProtocol == IrImageTransferProtocol::Normal)
    {
        if (m_ImageId == ((m_PrevImageId + 1) % (GetTransferPacketCount(m_ImageTransferFormat) + 1)))
        {
            // 連続したフレームが到着した場合はコピーする
            uint8_t* pImageBuffer = static_cast<uint8_t*>(m_pImageTransferProcessorState->pImage);
            memcpy((pImageBuffer + ImageTransferProcessorInputReportSize * m_ImageId), pBuffer + ImageTransferProcessorInputReportOffset, ImageTransferProcessorInputReportSize);
            NN_XCD_IRSENSOR_LOG("[ImageTransfer] Packet success imageId:%d prevId:%d\n", m_ImageId, m_PrevImageId);
            m_PrevImageId = m_ImageId;
            m_IsFrameDropped = false;
            if (m_ImageId == GetTransferPacketCount(m_ImageTransferFormat))
            {
                // 最終フレームを受け取ったら、成功を返す
                m_IsImageReady = true;
                NN_RESULT_SUCCESS;
            }
        }
        else if (m_ImageId == 0 && m_PrevImageId == 0)
        { // 1フレーム目
            NN_XCD_IRSENSOR_LOG("[ImageTransfer] Copy Start\n");
            uint8_t* pImageBuffer = static_cast<uint8_t*>(m_pImageTransferProcessorState->pImage);
            memcpy((pImageBuffer + ImageTransferProcessorInputReportSize * m_ImageId), pBuffer + ImageTransferProcessorInputReportOffset, ImageTransferProcessorInputReportSize);

            m_IsImageReady = false;
            m_PrevImageId = 0;
        }
        else if (m_ImageId == m_PrevImageId)
        {
            // 既に取得済みなのでスキップする。
        }
        else
        {
            // 非連続の場合は、パケット落ちが発生しているため、再送要求を出す。
            m_IsFrameDropped = true;
            NN_XCD_IRSENSOR_LOG("[ImageTransfer] Packet drop imageId:%d prevId:%d\n", m_ImageId, m_PrevImageId);
        }
        return nn::xcd::ResultIrImageTransferBusy();
    }
    else
    {
        NN_ABORT();
    }
}

Result IrsensorImageTransferProcessor::SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrImageTransferProcessorState* pImageTransferProcessorWorkBuffer, IrImageTransferProcessorFormat size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pIrCommonWorkBuffer);
    m_ImageTransferFormat = size;
    m_ReceivedFrameFlag.Set();
    m_IsImageReady = false;
    // ワークバッファを保持する
    m_pImageTransferProcessorState = pImageTransferProcessorWorkBuffer;
    m_IsProcessorReady = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorImageTransferProcessor::StopProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // 取得済みパケットのインデックスを初期化
    m_PrevImageId = 0;
    m_ReceivedFrameFlag.Set();
    m_IsImageReady = false;
    m_ContinuousFrameCount = 0;

    // ワークバッファを解放する
    m_pImageTransferProcessorState = nullptr;
    m_IsProcessorReady = false;
    NN_RESULT_SUCCESS;
}


Result IrsensorImageTransferProcessor::GetImageTransferState(
    IrCommonData* pOutIrCommonData,
    IrImageTransferProcessorState* pOutImageTransferProcessorState,
    int* pOutAckCount) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pOutImageTransferProcessorState);
    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    if (m_IsImageReady)
    {
        // データコピーする
        *(pOutIrCommonData) = m_IrCommonData[0];
        memcpy(pOutImageTransferProcessorState->pImage, m_pImageTransferProcessorState->pImage, GetTransferSize(m_ImageTransferFormat));
        *pOutAckCount = m_LatestAckCount;
        // コピーが終わったらMutexを解いて、Eventを返す
        NN_RESULT_SUCCESS;
    }
    else
    {
        // TODO: 適切なエラーを返す
        return nn::xcd::ResultIrSamplingTooLate();
    }
}

Result IrsensorImageTransferProcessor::SetDataReadCommand(uint8_t& resendFlag, uint8_t& resendId, uint8_t& imageId, ReceivedFrameFlag& receivedFrameFlag) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    if (m_TransferProtocol == IrImageTransferProtocol::FastMode)
    {
        if (m_ReceivedFrameFlag.CountPopulation() == IMAGE_TRANSFER_FRAME_COUNT_MAX - (GetTransferPacketCount(m_ImageTransferFormat) + 1)
            || m_ContinuousFrameCount > GetTransferPacketCount(m_ImageTransferFormat) + ContinuousLastFrameCountMax)

        {
            if (m_ContinuousFrameCount > GetTransferPacketCount(m_ImageTransferFormat) + ContinuousLastFrameCountMax)
            {
                NN_XCD_IRSENSOR_LOG("Retry completion command due to continuous last frame arrival\n");
                m_IsCompletionCommandAvailable = true;
                m_ContinuousFrameCount = 0;
            }
            // 全てのフレームの受けとりが完了したら、最終パケットID を MCU に通知する
            resendFlag = 0;
            resendId = 0;
            imageId = 0;
            // フラグを不用意に変えないように、ALl FF を送っておく
            receivedFrameFlag.Set();
            if (m_IsCompletionCommandAvailable)
            {
                // 連続で送らないようにする。 連続でここに到着した場合は、ALL FF を McuOut で送る。
                m_IsCompletionCommandAvailable = false;
                imageId = static_cast<uint8_t>(GetTransferPacketCount(m_ImageTransferFormat));
                NN_RESULT_THROW(nn::xcd::ResultIrImageTransferCompleted());
            }
        }
        else
        {
            // 受け取ってコピーが完了したパケットID を Mcuに通知する
            resendFlag = 0;
            resendId = 0;
            receivedFrameFlag = m_ReceivedFrameFlag;
        }
    }
    else if (m_TransferProtocol == IrImageTransferProtocol::Normal)
    {
        if (m_IsFrameDropped)
        {
            // フレーム落ちしていた場合は再送要求をする
            resendFlag = 1;
            resendId = static_cast<uint8_t>(m_PrevImageId + 1) % (GetTransferPacketCount(m_ImageTransferFormat) + 1);
        }
        else if (m_ImageId == GetTransferPacketCount(m_ImageTransferFormat))
        {
            // // 受け取ってコピーが完了したパケットID を Mcuに通知する
            resendFlag = 0;
            resendId = 0;
            imageId = static_cast<uint8_t>(m_ImageId);
        }
        else
        {
            // 受け取ってコピーが完了したパケットID を Mcuに通知する
            resendFlag = 0;
            resendId = 0;
            imageId = static_cast<uint8_t>(m_PrevImageId);
        }
    }
    else
    {
        NN_ABORT();
    }
    NN_XCD_IRSENSOR_LOG("[ImageTransfer] Packet out imageId:%d resendId:%d\n", imageId, resendId);
    NN_XCD_IRSENSOR_LOG("    ");
    for (auto i = 0; i < GetTransferPacketCount(m_ImageTransferFormat) + 1; i++)
    {
        NN_XCD_IRSENSOR_LOG("%d ", m_ReceivedFrameFlag.Test(i));
    }
    NN_XCD_IRSENSOR_LOG("\n");

    NN_RESULT_SUCCESS;
}

int IrsensorImageTransferProcessor::GetTransferPacketCount(IrImageTransferProcessorFormat size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(static_cast<int>(size) < ImageTransferFormatTypeCountMax);
    return TransferSizeInfo[static_cast<int>(size)].packetCnt;
}

size_t IrsensorImageTransferProcessor::GetTransferSize(IrImageTransferProcessorFormat size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(static_cast<int>(size) < ImageTransferFormatTypeCountMax);
    return TransferSizeInfo[static_cast<int>(size)].width * TransferSizeInfo[static_cast<int>(size)].height;
}

int IrsensorImageTransferProcessor::GetTransferFormat(IrImageTransferProcessorFormat* pSize) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    *pSize = m_ImageTransferFormat;
    return 0;
}

Result IrsensorImageTransferProcessor::SetTransferProtocol(IrImageTransferProtocol protocol) NN_NOEXCEPT
{
    m_TransferProtocol = protocol;
    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
