﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_AttachmentDevice.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_DeviceState.h>
#include <nn/xcd/xcd_DataFormat.h>
#include "xcd_AttachmentBase.h"
#include "xcd_BluetoothTypes.h"
#include "xcd_BtFirmwareUpdater.h"
#include "xcd_CommandHandler.h"
#include "xcd_FirmwareVersionTypes.h"
#include "xcd_IEventTask.h"
#include "xcd_MultiWaitEvent.h"
#include "xcd_OutputGenerator.h"
#include "xcd_PadInput.h"
#include "xcd_PairingManager.h"
#include "xcd_TransactionMonitor.h"
#include "xcd_VibratorBase.h"
#include "xcd_SerialFlashAccessor.h"
#include "xcd_TeraBase.h"
#include "xcd_ICommandListener.h"
#include "detail/xcd_HidAccessor.h"

namespace nn { namespace xcd {

const int MaxXcdBufferSize = 374;

//!< コントローラの連続充電可能時間
const nn::TimeSpanType ContinuousChargeIntervalMax = nn::TimeSpanType::FromHours(8);

enum DeviceSequence
{
    DeviceSequence_Nolink,               //!< リンクアップしていない状態
    DeviceSequence_Linkup,               //!< リンクアップしている状態 (通信は開始していない)
    DeviceSequence_ReadDeviceInfo,       //!< デバイスの情報の取得中
    DeviceSequence_ShipmentClear,        //!< Shipment の解除処理中
    DeviceSequence_ReadIdCode,           //!< S/N の読み出し中
    DeviceSequence_ReadColor,            //!< 色情報の読み出し中
    DeviceSequence_Pairing,              //!< ペアリング処理中
    DeviceSequence_Initialize,           //!< 各ペリフェラルの初期化処理中
    DeviceSequence_Active,               //!< 初期化処理が完了し、接続済みの状態
    DeviceSequence_Detach,               //!< 切断処理中
};

struct DeviceHandlerFlag
{
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<0> IsActivated;                  //!< アクティベートされているかどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<1> IsDeviceInfoReading;          //!< DeviceInfo の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<2> IsOnShipmentClear;            //!< Shipment の解除処理中かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<3> IsIdentificationCodeReading;  //!< S/N の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<4> IsColorReading;               //!< 色情報の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<5> IsOnPairing;                  //!< ペアリング中かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<6> IsErrorOccurred;              //!< 何かしらのエラーが起こった状態
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<7> IsRegisterationRequired;      //!< データベースへの登録処理が必要かどうか
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<8> NotifyOnConnectionComplete;   //!< 接続完了時に通知を行うためのフラグ
    typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag>::Flag<9> NeedsChangeSlotSizeRequest;   //!< スロットサイズ変更要求が必要かどうか
};

/**
 * @brief       DeviceHandler で扱うフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, DeviceHandlerFlag> DeviceHandlerFlagSet;

//!< Nx デバイスを管理/制御するためのクラスです。
class DeviceHandler final : public IEventTask, public ICommandListener
{

private:
    //!< DeviceHandler で使用するフラグの集合
    DeviceHandlerFlagSet m_Flags;

    //!< SlotSize の変更時に通知を受けるためのイベント
    nn::os::LightEventType m_SlotSizeEvent;

    //!< デバイスハンドル
    DeviceHandle m_Handle;

    //!< Hid 制御を行うためのアクセサ
    detail::HidAccessor* m_pAccessor;

    //!< OutputReport を生成するためのジェネレータ
    OutputGenerator m_Output;

    //!< HidCommand 制御を行うためのハンドラ
    CommandHandler m_Command;

    //!< Pad (ボタン, スティック, センサー) 制御
    PadInput m_PadInput;

    //!< Bluetooth ファームウェア更新
    BtFirmwareUpdater m_BtFirmwareUpdater;

    //!< 振動モーター
    VibratorBase m_VibratorBase;

    //!< Tera Firmware
    TeraBase m_TeraBase;

    //!< Transaction Monitor
    TransactionMonitor m_TransactionMonitor;

    //!< Serial Flash アクセサ
    SerialFlashAccessor m_SerialFlashAccessor;

    //!< 送受信に使用するバッファのサイズ
    static const size_t HidBufferSize = 512;

    //!< 接続状態に更新を通知するイベントオブジェクト
    nn::os::SystemEventType* m_pStatusUpdateEvent;

    //!< 新しいサンプルを受信した際に通知するイベントオブジェクト
    nn::os::SystemEventType* m_pSamplingEvent;

    //!< デバイスの状態
    DeviceSequence m_Sequence;

    //!< デバイスの情報
    DeviceInfo m_Info;

    //!< Firmware のバージョン
    FirmwareVersionImpl m_FirmwareVersion;

    //!< SerialFlash のフォーマットバージョン
    int m_FormatVersion;

    //!< デバイスの状態
    DeviceStatus m_Status;

    //!< 通信中のデータフォーマット
    PeriodicDataFormat m_DataFormat;

    //!< 変更先のデータフォーマット
    PeriodicDataFormat m_TargetDataFormat;

    //!< 本体に保存されるコントローラーの登録情報
    RegisteredDeviceInfo m_RegisteredInfo;

    //!< ペアリングマネージャー
    PairingManager*  m_pPairingManager;

    //!< インジケータ LED の点灯パターン
    uint8_t m_IndicatorLedPattern;

    //!< 拡張デバイス制御
    AttachmentBase m_AttachmentBase;

    //!< コントローラーの色
    DeviceColor m_DeviceColor;

    //!< コントローラの連続充電監視用のタイマー
    nn::os::TimerEventType m_ChargerTimer;

public:
    DeviceHandler() NN_NOEXCEPT;
    virtual ~DeviceHandler() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< DeviceHandlerを有効にする
    void Activate(DeviceHandle m_Handle, InterfaceType interfaceType, detail::HidAccessor* pAccessor) NN_NOEXCEPT;

    //!< DeviceHandlerを無効にする
    void Deactivate() NN_NOEXCEPT;

    //!< アクティベートされているかどうか
    bool isActivated() NN_NOEXCEPT
    {
        return m_Flags.Test<DeviceHandlerFlag::IsActivated>();
    };

    //!< DeviceHandleを取得する
    DeviceHandle GetDeviceHandle() NN_NOEXCEPT
    {
        return m_Handle;
    }

    //!< デバイスの情報を取得する
    void GetDeviceInfo(DeviceInfo* pInfo) NN_NOEXCEPT
    {
        *pInfo = m_Info;
    }

    //!< デバイスの状態を取得する
    void GetDeviceStatus(DeviceStatus* pStatus) NN_NOEXCEPT
    {
        *pStatus = m_Status;
    }

    //!< Firmware のバージョンを取得する
    BtFirmwareVersion GetBtFirmwareVersion() NN_NOEXCEPT;

    //!< 接続済みかどうか
    bool IsConnected() NN_NOEXCEPT;
    //!< 切断する
    void Detach() NN_NOEXCEPT;

    //!< 再起動する
    void Reboot(bool reconnect) NN_NOEXCEPT;

    //!< 再起動する isFlash==Trueの場合は即座に転送される
    void Reboot(bool reconnect, bool isFlash) NN_NOEXCEPT;

    //!< TeraMcu を起動/停止する
    void McuResume(bool isResume) NN_NOEXCEPT;

    //!< 接続状態に更新があった場合に通知するイベントオブジェクトを登録する
    void SetStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< 接続状態に更新があった場合に通知するシステムイベントを解除します
    void ClearStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< サンプリングイベントをセットする
    Result SetSamplingEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< ペアリングマネージャーを登録する
    void SetPairingManager(PairingManager* pManager) NN_NOEXCEPT;

    //!< 各下位モジュールへのインタフェース
    PadInput* GetPadInput() NN_NOEXCEPT
    {
        return &m_PadInput;
    }
    BtFirmwareUpdater* GetBtFirmwareUpdater() NN_NOEXCEPT
    {
        return &m_BtFirmwareUpdater;
    }
    VibratorBase* GetVibrator() NN_NOEXCEPT
    {
        return &m_VibratorBase;
    }
    TeraBase* GetTeraBase() NN_NOEXCEPT
    {
        return &m_TeraBase;
    }
    AttachmentBase* GetAttachmentBase() NN_NOEXCEPT
    {
        return &m_AttachmentBase;
    }
    SerialFlashAccessor* GetSerialFLashAccessor() NN_NOEXCEPT
    {
        return &m_SerialFlashAccessor;
    }

    //!< Ack の受領を受け取るための通知関数
    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;

    //!< DataFormat の変更完了を受け取るための通知関数
    virtual void NotifySetDataFormat(Result result, PeriodicDataFormat dataFormat) NN_NOEXCEPT NN_OVERRIDE;

    //!< DeviceInfoの受領を受け取るための通知関数
    virtual void NotifyDeviceInfo(const DeviceInfo& deviceInfo,
                                  FirmwareVersionUnit firmwareVersion,
                                  int serialFlashFormatVersion) NN_NOEXCEPT NN_OVERRIDE;
    //!< IdentificationCode の受領を受け取るための通知関数
    virtual void NotifyIdentificationCode(const IdentificationCode& code) NN_NOEXCEPT NN_OVERRIDE;

    //!< ペアリングの完了を受け取るための通知関数
    virtual void NotifyPairingDatabaseUpdate(const BluetoothDeviceInfo& deviceInfo) NN_NOEXCEPT NN_OVERRIDE;

    //!< ペアリングの完了を受け取るための通知関数
    virtual void NotifyPairingComplete(Result result) NN_NOEXCEPT NN_OVERRIDE;

    //!< 色情報の読み出し完了を受け取るための通知関数
    virtual void NotifyDesign(const DeviceColor& color,
                              uint8_t variation) NN_NOEXCEPT NN_OVERRIDE;

    //!< 色情報の書き込み完了を受け取るための通知関数
    virtual void NotifyUpdateControllerColor(Result result) NN_NOEXCEPT NN_OVERRIDE;

    //!< HidCommandに対して コントローラーのデザイン情報書き込みが完了した際の通知を受け取るための関数です
    virtual void NotifyUpdateDesignInfo(Result result) NN_NOEXCEPT NN_OVERRIDE;

    //!< データフォーマットを変更する
    void SetDataFormat(PeriodicDataFormat format) NN_NOEXCEPT;

    //!< 現在のデータフォーマットを取得する
    PeriodicDataFormat GetDataFormat() NN_NOEXCEPT;

    //!< データフォーマットを変更し、変更完了をポーリングで待つ
    bool SetDataFormatAndPollingTimedWait(PeriodicDataFormat format, nn::TimeSpan pollingTime, nn::TimeSpan timeOut) NN_NOEXCEPT;

    //!< インジケータ LED の点灯パターンを取得する
    Result GetIndicatorLedPattern(uint8_t* pOutPattern) const NN_NOEXCEPT;

    //!< 色情報を読み出す
    DeviceColor GetColor() NN_NOEXCEPT;

    //!< 色情報を書き込む
    void SetColor(const ::nn::util::Color4u8Type& mainColor, const ::nn::util::Color4u8Type& subColor) NN_NOEXCEPT;

    //!< デザイン情報を書き込む
    void SetDesignInfo(const DeviceColor& color, uint8_t variation) NN_NOEXCEPT;

    //!< LEDを制御する
    Result ControlLed(uint8_t pattern) NN_NOEXCEPT;

    //!< 拡張デバイスの電源を制御します。
    Result AttachmentPowerEnable(bool isEnabled) NN_NOEXCEPT;

    //!< シリアルナンバーを取得します
    IdentificationCode GetIdentificationCode() NN_NOEXCEPT;

    //!< Bluetooth の Firmware 更新を開始します
    Result StartBtFirmwareUpdate(const FirmwareImage& image, nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< Bluetooth の Firmware 更新を中断します
    Result AbortBtFirmwareUpdate() NN_NOEXCEPT;

    //!< Bluetooth 経由のFirmware更新しかサポートしていないかどうか確認します
    bool IsBtFirmwareUpdateThroughBtRequired() NN_NOEXCEPT;

    //!< パケットの受信履歴を取得します
    nn::hid::detail::RxPacketHistory GetRxPacketHistory() NN_NOEXCEPT;

    //!< ペアリング処理を行う
    void TriggerPairing() NN_NOEXCEPT;

    //!< 連続充電のタイマーをリセットします
    void ResetBatteryChargerTimer() NN_NOEXCEPT;

private:
    static void InputReportHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< 各ペリフェラルデバイスの初期化処理
    void InitializeDevice() NN_NOEXCEPT;

    //!< データフォーマット更新が可能な状況であれば更新を行う
    void TryUpdateDataFormat() NN_NOEXCEPT;

    //!< 指定されたデータフォーマットに対応するスロットサイズを取得
    int GetTargetSlotSize(PeriodicDataFormat dataFormat) NN_NOEXCEPT;

    //!< InputReport を パースする
    void ParseInputReport(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< OutputReport を送信する
    void SendOutputReport() NN_NOEXCEPT;

    //!< InputReport 内の Controller Status をパースする
    void ParseControllerStatus(const uint8_t status, const uint8_t statusForPower) NN_NOEXCEPT;

    //!< シーケンスの更新処理を行う
    void UpdateSequence() NN_NOEXCEPT;

    //!< 次のシーケンスの処理をおこなう
    void HandleNextSequence() NN_NOEXCEPT;

    //!< 各デバイスの初期化処理が完了したかどうか
    bool IsPeripheralInitializationCompleted() NN_NOEXCEPT;

    //!< InputReport を受信した際に行う処理
    void InputReportHandlerImpl(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< SlotSize の変更が行われた際に行う処理
    void OnTsiChanged() NN_NOEXCEPT;

    //!< ペアリングを開始する
    void TriggerPairingImpl() NN_NOEXCEPT;

    //!< コントローラの充電状態の確認を行う
    void CheckChargeStatus(bool isChargeStatusUpdated) NN_NOEXCEPT;

    //!< コントローラの充電をリセットする
    void ResetChargeStatus() NN_NOEXCEPT;
};

}} // namespace nn::xcd
