﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace xcd {

//!< InputReport キャラクタリスティックのデータフォーマットの定義です。

//!< フォーマットバージョン
enum BleInputReportFormatVersion : uint8_t
{
    BleInputReportFormatVersion_Unknown = 0x00,  //!< 不明なフォーマット
    BleInputReportFormatVersion_JoyRight = 0x01, //!< JoyRight に相当する入力情報を持ったフォーマット
    BleInputReportFormatVersion_JoyLeft = 0x02,  //!< JoyLeft に相当する入力情報を持ったフォーマット
    BleInputReportFormatVersion_Palma = 0x03,    //!< Palma 専用フォーマット
};

//!< JoyRight のデータフォーマットに関する定義
struct BleInputReportFormat_JoyRight
{
    static const int SampleNumber = 0;                 //!< サンプル番号
    static const int Button = SampleNumber + 1;        //!< ボタン
    static const int AnalogStick = SampleNumber + 3;   //!< アナログスティック
    static const int Accelerometer = SampleNumber + 6; //!< 加速度センサー
    static const int Gyroscope = SampleNumber + 12;    //!< ジャイロセンサー
};
//!< JoyRight フォーマットのデータサイズ
const uint8_t BleInputReportFormat_JoyRight_Size = 18;

//!< Palma のデータフォーマットに関する定義
struct BleInputReportFormat_Palma
{
    static const int SampleNumber = 0;                 //!< サンプル番号
    static const int Button = SampleNumber + 1;        //!< ボタン
    static const int AnalogStick = SampleNumber + 2;   //!< アナログスティック
    static const int Accelerometer = SampleNumber + 5; //!< 加速度センサー
    static const int Gyroscope = SampleNumber + 11;    //!< ジャイロセンサー
};
//!< Palma フォーマットのデータサイズ
const uint8_t BleInputReportFormat_Palma_Size = 17;

//!< OutputCommand キャラクタリスティックのデータフォーマットの定義です。

//!< フォーマットバージョン
enum BleOutputCommandFormatVersion : uint8_t
{
    BleOutputCommandFormatVersion_Unknown = 0x00, //!< 不明なフォーマット
    BleOutputCommandFormatVersion_Palma = 0x01,   //!< Palma 専用フォーマット
};

const uint8_t BleOutputCommand_InvalidCommandId = 0x00; //!< 不正なコマンド ID
const uint16_t BleOutputCommand_PayloadMaxLength = 500; //!< コマンドペイロードの最大サイズ

//!< OutputCommand のデスクリプタ
struct BleOutputCommandDescriptor
{
    bool           isCancelRequested; //!< 中断のリクエスト
    const uint8_t* pDataPtr;        //!< 送信データのポインタ
    size_t         maxDataSize;     //!< 送信データのサイズ
    uint8_t        _initialSequenceNumber;  //!< 最初のシーケンス番号（コマンドハンドラ内部で使用します）
    uint8_t        _sequenceNumber; //!< 分割送信する際のシーケンス番号（コマンドハンドラ内部で使用します）
    size_t         _sentByte;       //!< 送信済みのデータ量（コマンドハンドラ内部で使用します）
};

//!< CommandResponse キャラクタリスティックのデータフォーマットの定義です。

//!< フォーマットバージョン
enum BleCommandResponseFormatVersion : uint8_t
{
    BleCommandResponseFormatVersion_Unknown = 0x00, //!< 不明なフォーマット
    BleCommandResponseFormatVersion_Palma = 0x01,   //!< Palma 専用フォーマット
};

//!< コマンドの状態
enum BleCommandStatus
{
    BleCommandStatus_Empty,       //!< Commandがキューにない状態
    BleCommandStatus_Queued,      //!< Commandがキューにセットされた状態
    BleCommandStatus_Sent,        //!< Commandが送信されResponseを待っている状態
    BleCommandStatus_Completed,   //!< Responseを受信しCommandが完了した状態
};

//!< レスポンスの状態
enum BleResponseStatus
{
    BleResponseStatus_Empty,       //!< レスポンスがキューにない状態
    BleResponseStatus_Waiting,     //!< レスポンスを待っている状態
    BleResponseStatus_Completed,   //!< オペレーションが完了し、レスポンスの受信が完了した状態
};

//!< BLE デバイスのプロダクト情報
struct BleDeviceProductType
{
    uint8_t main;  //!< プロダクト種別
    uint8_t sub;   //!< サブ種別
};

//!< BLE デバイスのモデル情報の内容
struct BleDeviceModelInformation
{
    SensorState sensorHorizontalOffset; //!< 6軸センサーの Horizontal Offset
    bool isScalingRequired;             //!< X軸のスケーリング処理が必要かどうか
    uint8_t analogStickNoise;           //!< アナログスティックのノイズ
    int16_t analogStickTypicalStroke;   //!< 設計ストローク
    int16_t analogStickOriginPlay;      //!< ゼロ点の遊びの値、中央不感帯幅
    int16_t analogStickCircuitValidRatio; //!< 外側不感帯の割合 (%)
    AnalogStickState analogStickMinimumStrokePositive; //!< 正方向の保証下限ストローク
    AnalogStickState analogStickMinimumStrokeNegative; //!< 負方向の保証下限ストローク
    AnalogStickState analogStickOriginRangeMin;        //!< 原点範囲の最小値
    AnalogStickState analogStickOriginRangeMax;        //!< 原点範囲の最大値
};

}} // namespace nn::xcd
