﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Windows.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Event.h>
#include <nn/xcd/xcd_Device.h>

#include "../xcd_ReportTypes.h"
#include "xcd_IHidListener.h"
#include "xcd_HidAccessor.h"

extern "C" {
#include <hidsdi.h>
}

namespace nn { namespace xcd { namespace detail{

const int HidAccessorBufferSize = 512;
const size_t HidThreadStackSize = 1024 * 8;

//!< 汎用的なHidデバイスを扱うクラス
class Win32HidAccessor : public HidAccessor
{
    NN_DISALLOW_MOVE(Win32HidAccessor);
    NN_DISALLOW_COPY(Win32HidAccessor);

protected:
    //!< InputReportを格納するバッファ
    uint8_t m_Buffer[HidAccessorBufferSize];
    size_t  m_ReceivedSize;

    //!< InputReport を解析する関数
    InputReportParserFunc m_pInputReportParserFunc;
    void* m_pInputReportParserArg;

    //!< 終了を通知するイベント
    nn::os::EventType m_TerminateEvent;

private:
    //!< デバイスの接続状態を監視するスレッド
    nn::os::ThreadType m_Thread;

    //!< スレッドのスタック
    NN_ALIGNAS(4096) char m_ThreadStack[HidThreadStackSize];

    //!< DeviceHandle
    DeviceHandle m_DeviceHandle;
    //!< HIDデバイスへのハンドル
    HANDLE m_Handle;
    HIDP_CAPS m_Capabilities;


    //!< Activateされているかどうか
    bool m_Activated;

private:
    static void ThreadFunction(void* arg) NN_NOEXCEPT;

protected:
    virtual void ThreadFunctionImpl() NN_NOEXCEPT;

public:
    Win32HidAccessor() NN_NOEXCEPT;
    virtual ~Win32HidAccessor() NN_NOEXCEPT NN_OVERRIDE;

    //!< Accessorを有効にする (環境依存)
    virtual void Activate(DeviceHandle deviceHandle, HANDLE hidHandle) NN_NOEXCEPT;
    //!< Accessorを無効にする
    virtual void Deactivate() NN_NOEXCEPT;
    //!< Activagteされているかどうか
    bool IsActivated() NN_NOEXCEPT
    {
        return m_Activated;
    }

    //!< サンプリングを開始
    virtual void StartSampling(InputReportParserFunc func, void* pArg) NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリングを停止
    virtual void StopSampling() NN_NOEXCEPT NN_OVERRIDE;

    //!< OutputReportをセットする
    virtual void SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< InputReportを読み出す
    virtual size_t GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< SetReport を送信する
    virtual void SetReport(ReportType reportType, uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< GetReport を送信する
    virtual void GetReport(ReportType reportType, uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< 通信間隔を取得する
    virtual Result GetInterval(::nn::TimeSpan* pOutInterval) NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスを切断する
    virtual void DetachDevice() NN_NOEXCEPT NN_OVERRIDE;

    DeviceHandle GetDeviceHandle() NN_NOEXCEPT
    {
        return m_DeviceHandle;
    }
};

}}} // namespace nn::xcd::detail
