﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Windows.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_Win32HidAccessor-os.win32.h"
#include "../xcd_CommandTypes.h"

extern "C" {
#include <hidsdi.h>
#include <Dbt.h>
}

namespace nn { namespace xcd { namespace detail {

namespace
{

const ::nn::TimeSpan Win32Interval = ::nn::TimeSpan::FromMilliSeconds(15);

}

Win32HidAccessor::Win32HidAccessor() NN_NOEXCEPT
    : m_Activated(false)
    , m_pInputReportParserFunc(nullptr)
    , m_pInputReportParserArg(nullptr)
{
    // 何もしない
}

Win32HidAccessor::~Win32HidAccessor() NN_NOEXCEPT
{
    // 何もしない
}

void Win32HidAccessor::Activate(DeviceHandle deviceHandle, HANDLE hidHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    m_DeviceHandle = deviceHandle;
    m_Handle = hidHandle;
    m_Activated = true;
}

void Win32HidAccessor::Deactivate() NN_NOEXCEPT
{
    // サンプリングを中止
    StopSampling();

    CloseHandle(m_Handle);

    m_Activated = false;
}

void Win32HidAccessor::StartSampling(InputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    m_pInputReportParserFunc = func;
    m_pInputReportParserArg  = pArg;

    // 終了用のイベントを初期化
    nn::os::InitializeEvent(&m_TerminateEvent, false, nn::os::EventClearMode_AutoClear);

    // デバイスの接続状態監視用のスレッドを生成
    nn::os::CreateThread(&m_Thread,
                         Win32HidAccessor::ThreadFunction,
                         this,
                         m_ThreadStack,
                         HidThreadStackSize,
                         nn::os::DefaultThreadPriority); // TODO
    nn::os::StartThread(&m_Thread);
}

void Win32HidAccessor::StopSampling() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    nn::os::SignalEvent(&m_TerminateEvent);

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    nn::os::FinalizeEvent(&m_TerminateEvent);
}

void Win32HidAccessor::ThreadFunction(void* arg) NN_NOEXCEPT
{
    reinterpret_cast<Win32HidAccessor*>(arg)->ThreadFunctionImpl();
}

void Win32HidAccessor::ThreadFunctionImpl() NN_NOEXCEPT
{
    PHIDP_PREPARSED_DATA preparsedData;
    HidD_GetPreparsedData(
        m_Handle,
        &preparsedData);

    HidP_GetCaps(
        preparsedData,
        &m_Capabilities);

    while (NN_STATIC_CONDITION(true))
    {

        if(nn::os::TryWaitEvent(&m_TerminateEvent) == true)
        {
            break;
        }

        DWORD bytesRead;

        auto result = ReadFile(
            m_Handle,
            m_Buffer,
            HidAccessorBufferSize,
            &bytesRead,
            NULL);

        if(result == false)
        {
//            NN_SDK_LOG("Read file failed %d\n", GetLastError());
        }
        else
        {
            m_ReceivedSize = bytesRead;
            if (m_pInputReportParserFunc != nullptr)
            {
                m_pInputReportParserFunc(m_pInputReportParserArg, m_Buffer, m_ReceivedSize);
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

//!< コントローラに送るOutputReportをセットする
void Win32HidAccessor::SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);

    if (size > 0)
    {
        DWORD bytesWritten;

        if(size > m_Capabilities.OutputReportByteLength)
        {
            // TODO: エラーハンドリングを追加
            return;
        }

        auto result = WriteFile(
            m_Handle,
            pBuffer,
            m_Capabilities.OutputReportByteLength,
            &bytesWritten,
            NULL);
        if(result == false)
        {
            NN_SDK_LOG("Write file failed %d %d\n", GetLastError(), m_Capabilities.OutputReportByteLength);
        }
    }
}

size_t Win32HidAccessor::GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);
    NN_SDK_REQUIRES(size > 0);

    size_t returnSize = (m_ReceivedSize < size) ? m_ReceivedSize : size;
    std::memcpy(pOutValue, m_Buffer, returnSize);

    return returnSize;
}

void Win32HidAccessor::SetReport(ReportType reportType, uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_EQUAL(m_Activated, true);
    NN_SDK_REQUIRES(size > 0);

    NN_UNUSED(reportType);
    NN_UNUSED(pBuffer);
    NN_UNUSED(size);
    NN_UNUSED(pListener);
}

void Win32HidAccessor::GetReport(ReportType reportType, uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    NN_UNUSED(reportType);
    NN_UNUSED(reportId);
    NN_UNUSED(pListener);
}

Result Win32HidAccessor::GetInterval(::nn::TimeSpan* pOutInterval) NN_NOEXCEPT
{
    // Win 環境での通信間隔はいったん決め打ち
    *pOutInterval = Win32Interval;
    NN_RESULT_SUCCESS;
}

void Win32HidAccessor::DetachDevice() NN_NOEXCEPT
{
    uint8_t buffer[Report_CommandOut::Size] = { 0 };
    buffer[ReportByte_ReportId] = Report_CommandOut::Id;
    buffer[CommandOutReportByte_Payload] = Command_Reset.Id;
    SetOutputReport(buffer, sizeof(buffer));
}

}}} // namespace nn::xcd::detail
