﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>  // type_traits 対応判定のため、先に必要
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)
    #include <type_traits>
#endif
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "xcd_TeraNfc.h"

namespace nn { namespace xcd {

// 構造体の POD 判定
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

// 公開ヘッダの構造体
NN_STATIC_ASSERT(std::is_pod<NfcTagId>::value);
NN_STATIC_ASSERT(std::is_pod<NfcTagInfo>::value);
NN_STATIC_ASSERT(std::is_pod<NfcErrorInfo>::value);
NN_STATIC_ASSERT(std::is_pod<NtagReadAddress>::value);
NN_STATIC_ASSERT(std::is_pod<NfcNtagData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPassThruData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcInfo>::value);
NN_STATIC_ASSERT(std::is_pod<NfcDiscoveryParameter>::value);
NN_STATIC_ASSERT(std::is_pod<NtagReadParameter>::value);
NN_STATIC_ASSERT(std::is_pod<NtagWriteData>::value);
NN_STATIC_ASSERT(std::is_pod<NtagWriteParameter>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPassThruParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareEncryptedKeyValue>::value);
NN_STATIC_ASSERT(std::is_pod<MifareKeyValue>::value);
NN_STATIC_ASSERT(std::is_pod<MifareKeyWriteParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareKeyClearParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareKey>::value);
NN_STATIC_ASSERT(std::is_pod<MifareReadBlockParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareReadParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareWriteBlockParameter>::value);
NN_STATIC_ASSERT(std::is_pod<MifareWriteParameter>::value);

namespace detail
{

// 内部の構造体
NN_STATIC_ASSERT(std::is_pod<NfcPacketPayloadAttributePack>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPackedPacketHeader>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPackedCommonResponseHeader>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPackedTagInfo>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPackedStateInfPayload>::value);
NN_STATIC_ASSERT(std::is_pod<NfcDataResponseHeader>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPassThruResponseHeader>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPacketHeaderType>::value);
NN_STATIC_ASSERT(std::is_pod<NfcStartDiscoveryCommandType>::value);
NN_STATIC_ASSERT(std::is_pod<NfcCommandType>::value);
NN_STATIC_ASSERT(std::is_pod<NfcType2TagData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcPassThruResponseData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareKeyListForWrite>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareCommandAttributePack>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareBlock>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareResponseData>::value);
NN_STATIC_ASSERT(std::is_pod<NfcMifareCommandConfig>::value);
NN_STATIC_ASSERT(std::is_pod<ResponseTagData>::value);
NN_STATIC_ASSERT(std::is_pod<StatusForGetInfo>::value);

#endif  // defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

namespace
{

// 不明な値に対応する文字列
const char UnknownString[] = "Unknown";

nn::xcd::NfcTagType ConvertTagTypeA(uint8_t internalTagType) NN_NOEXCEPT
{
    switch (internalTagType)
    {
    case DiscoveryTypeId_Type1:      return nn::xcd::NfcTagType_Type1;
    case DiscoveryTypeId_Type2:      return nn::xcd::NfcTagType_Type2;
    case DiscoveryTypeId_Type3:      return nn::xcd::NfcTagType_Type3;
    case DiscoveryTypeId_Type4:      return nn::xcd::NfcTagType_Type4A;
    case DiscoveryTypeId_TypeMifare: return nn::xcd::NfcTagType_Mifare;
    default: return nn::xcd::NfcTagType_Error;
    }
}

nn::xcd::NfcTagType ConvertTagTypeB(uint8_t internalTagType) NN_NOEXCEPT
{
    switch (internalTagType)
    {
    case DiscoveryTypeId_Type1: return nn::xcd::NfcTagType_Type1;
    case DiscoveryTypeId_Type2: return nn::xcd::NfcTagType_Type2;
    case DiscoveryTypeId_Type3: return nn::xcd::NfcTagType_Type3;
    case DiscoveryTypeId_Type4: return nn::xcd::NfcTagType_Type4B;
    default: return nn::xcd::NfcTagType_Error;
    }
}

nn::xcd::NfcTagType ConvertTagTypeF(uint8_t internalTagType) NN_NOEXCEPT
{
    switch (internalTagType)
    {
    case DiscoveryTypeId_Type1: return nn::xcd::NfcTagType_Type1;
    case DiscoveryTypeId_Type2: return nn::xcd::NfcTagType_Type2;
    case DiscoveryTypeId_Type3: return nn::xcd::NfcTagType_Type3;
    default: return nn::xcd::NfcTagType_Error;
    }
}

nn::xcd::NfcTagType ConvertTagType15693(uint8_t internalTagType) NN_NOEXCEPT
{
    switch (internalTagType)
    {
    case DiscoveryTypeId_TypeIso15693: return nn::xcd::NfcTagType_Iso15693;
    default: return nn::xcd::NfcTagType_Error;
    }
}

}  // anonymous

void DumpNfcBlocks(const char* pData, int startBlock, size_t blocks) NN_NOEXCEPT
{
#if defined(NN_DETAIL_ENABLE_SDK_LOG)
    NN_SDK_REQUIRES_NOT_NULL(pData);

    NN_SDK_LOG(" Addr | ");
    for (size_t i = 0; i < 4; ++i)
    {
        NN_SDK_LOG("%02X ", i);
    }
    NN_SDK_LOG("\n--------");
    for (size_t i = 0; i < 4; ++i)
    {
        NN_SDK_LOG("---");
    }

    for (size_t i = 0; i < blocks; ++i)
    {
        const auto* pBlock = reinterpret_cast<const nn::Bit8*>(pData + i * 4);
        NN_SDK_LOG("\n 0x%02X | ", startBlock + i);
        for (int j = 0; j < 4; ++j)
        {
            NN_SDK_LOG("%02X ", pBlock[j]);
        }
    }
    NN_SDK_LOG("\n");
#else
    NN_UNUSED(pData);
    NN_UNUSED(startBlock);
    NN_UNUSED(blocks);
#endif  // if defined(NN_DETAIL_ENABLE_SDK_LOG)
}

const char* GetNfcStateString(NfcState state) NN_NOEXCEPT
{
    // 各ステートに対応した文字列定義
    static const char* StateStringTable[] =
    {
        "COMMAND_WAITING",
        "POLLING",
        "READING",
        "WRITING",
        "READ_FINISH",
        "WRITE_FINISH",
        "PASS_THRU_SENDING",
        "ERROR",
        "DEACTIVATED",
        "DETECTED",
        "FACTORY_MODE",
        "INITIALIZING",
        "PASS_THRU_FINISH",
        "RESET_REQUIRED",
        "HW_FATAL",
        "MIFARE_SENDING",
        "MIFARE_FINISH",
        "MIFARE_KEY_WRITING",
        "MIFARE_KEY_WRITE_FINISH"
    };
    static const auto StateStringTableCount =
        sizeof(StateStringTable) / sizeof(StateStringTable[0]);

    if (state >= StateStringTableCount)
    {
        return UnknownString;
    }

    return StateStringTable[state];
}

const char* GetNfcResultString(NfcResultCode result) NN_NOEXCEPT
{
    // 各 ResultCode に対応した文字列定義
    static const char* ResultStringTable[] =
    {
        "FUNCTION_ERROR",       // 60
        "RESET_REQUIRED",       // 61
        "READ_ERROR",           // 62
        "WRITE_ERROR",          // 63
        "ARGUMENT_ERROR",       // 64
        "TIMEOUT",              // 65
        "INVALID_UID",          // 66
        UnknownString,          // 67 は欠番
        "T2T_PWD_INVALID_TAG",  // 68
        "VERIFY_ERROR",         // 69
        "ACTIVATION_ERROR",     // 70
        "INVALID_TAG",          // 71
        "INVALID_FORMAT",       // 72
        "AUTHENTICATION_ERROR", // 73
        "SEQUENCE_ERROR",       // 74
        "COMMAND_TIMEOUT",      // 75
        "MIFARE_ERROR"          // 76
    };
    static const auto ResultStringTableCount =
        sizeof(ResultStringTable) / sizeof(ResultStringTable[0]);

    if (result == NfcResultCode_Ok)
    {
        return "OK";
    }
    else if (result < NfcResultCode_FunctionError)
    {
        return UnknownString;
    }

    auto resultIndex = static_cast<uint32_t>(result - NfcResultCode_FunctionError);
    if (resultIndex >= ResultStringTableCount)
    {
        return UnknownString;
    }

    return ResultStringTable[resultIndex];
}

nn::xcd::NfcTagType ConvertTagType(uint8_t discoveryType, uint8_t internalTagType) NN_NOEXCEPT
{
    // Cafe と同じ判定

#if 0
    NN_SDK_LOG(
        "%s: Protocol = %d, Type = %d\n",
        NN_CURRENT_FUNCTION_NAME,
        discoveryType,
        internalTagType);
#endif

    switch (discoveryType)
    {
    case nn::xcd::NfcProtocol_TypeA:
        return ConvertTagTypeA(internalTagType);
    case nn::xcd::NfcProtocol_TypeB:
        return ConvertTagTypeB(internalTagType);
    case nn::xcd::NfcProtocol_TypeF:
        return ConvertTagTypeF(internalTagType);
    case nn::xcd::NfcProtocol_Type15693:
        return ConvertTagType15693(internalTagType);
    default:
        return nn::xcd::NfcTagType_Error;
    }
}

void ConfigureNfcCommand(NfcCommandType* pCommand, NfcCommand commandCode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCommand);

    pCommand->header.command       = commandCode;
    pCommand->header.bluetoothMode = BluetoothMode_Nfc;
    pCommand->header.protocol      = CommandProtocol_Nfc;
    std::memset(pCommand->payload, 0, PayloadLengthMax);
    pCommand->payloadLength = 0;
}

void ConfigureNfcNopCommand(NfcCommandType* pCommand) NN_NOEXCEPT
{
    ConfigureNfcCommand(pCommand, NfcCommand_Nop);
}

bool IsExpectedNfcStateForSendFinish(NfcCommand command, NfcState state) NN_NOEXCEPT
{
    switch (command)
    {
    case NfcCommand_StartDiscovery:
        return state == NfcState_Polling ||
            state == NfcState_Detected;

    case NfcCommand_StopDiscovery:
        return state == NfcState_CommandWaiting;

    case NfcCommand_ReadStart:
        return state == NfcState_Reading ||
            state == NfcState_ReadFinish;

    default:
        return false;
    }
}

int SetMifareCommonBlockParameter(
    char* pOutParameter,
    bool isRawKey,
    const MifareKey& key,
    uint8_t blockAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParameter);
    NN_SDK_REQUIRES(key.type == MifareKeyType_A || key.type == MifareKeyType_B);

    // [0] 鍵の種類
    pOutParameter[0] = static_cast<char>(key.type);

    int dataLength = 1;

    // [1..6] 鍵の値
    if (isRawKey)
    {
        std::memcpy(pOutParameter + 1, key.raw.value, MifareKeyLength);
        dataLength += MifareKeyLength;
    }
    else
    {
        pOutParameter[1] = static_cast<char>(key.id);
        dataLength++;
    }

    // [1 + 鍵長] ブロックアドレス
    pOutParameter[dataLength] = static_cast<char>(blockAddress);
    dataLength++;

    return dataLength;
}

}}}
