﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/detail/xcd_Log.h>

//#define VERBOSE

/*
 * VERBOSE モード時のみ、引数に指定された文字列を、Tera のログとして出力します。
 */
#ifdef VERBOSE

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_TERA_LOG(...)             NN_DETAIL_XCD_INFO("[TeraMcu] " ##__VA_ARGS__)
#else
#define NN_XCD_TERA_LOG(format, ...)     NN_DETAIL_XCD_INFO("[TeraMcu] " format, ##__VA_ARGS__)
#endif

#else  // VERBOSE

#define NN_XCD_TERA_LOG(...)            static_cast<void>(0)

#endif  // VERBOSE

namespace nn { namespace xcd {

class DeviceHandler;

namespace detail {

/**
 * @brief   Tera MCU からの応答ペイロード長を指定するための Bluetooth のモードです。
 */
enum BluetoothMode : int
{
    BluetoothMode_Common   = 0,     //!< 共通コマンド用
    BluetoothMode_Nfc      = 0,     //!< NFC 用
    BluetoothMode_Ir       = 0,     //!< IR センサー用
    BluetoothMode_McuWrite = 2,     //!< MCU Write 用
    BluetoothMode_Max      = 16
};

/**
 * @brief   コマンドのプロトコルです。
 */
enum CommandProtocol : uint8_t
{
    CommandProtocol_Common = 1,     //!< Common
    CommandProtocol_Nfc    = 2,     //!< NFC
    CommandProtocol_Ir     = 3,     //!< IR
    CommandProtocol_Idle   = 4,     //!< IDLE (検査用)
};

/**
 * @brief   Common コマンドの ID です。
 */
enum CommonCommandId : uint8_t
{
    CommonCommandId_TeraControl = 0x00,     //!< TERA_CONTROL
    CommonCommandId_ExtControl  = 0x01      //!< EXT_CONTROL
};

/**
 * @brief   内部で使用する MCU のステートです。
 */
enum InternalMcuState : int
{
    InternalMcuState_Nop          = 0x00,   //!< TERA_CONTROL コマンドで、ステートを変更しないことを表す。Tera から返ってくることはない
    InternalMcuState_Idle         = 0x01,   //!< Idle
    InternalMcuState_Standby      = 0x02,   //!< Standby
    InternalMcuState_Background   = 0x03,   //!< Background
    InternalMcuState_Nfc          = 0x04,   //!< NFC
    InternalMcuState_Ir           = 0x05,   //!< IR
    InternalMcuState_Initializing = 0x06,   //!< Initializing (初期化処理中)

    InternalMcuState_SystemBoot   = 0x100,  //!< システムブート (FW アップデート用)
};

/**
 * @brief   内部で使用する MCU から返ってくるリザルトです。
 */
enum InternalMcuResult : uint8_t
{
    InternalMcuResult_Ok                        = 0x00,   //!< 成功しました。
    InternalMcuResult_Ng                        = 0x01,   //!< 失敗しました。
    InternalMcuResult_Busy                      = 0x02,   //!< 処理中です。
    InternalMcuResult_MessageTypeInvalid        = 0x03,   //!< 不正なメッセージタイプです。
    InternalMcuResult_MessageSignalInvalid      = 0x04,   //!< 不正なメッセージシグナルです。
    InternalMcuResult_MessageSignalNotSupported = 0x05,   //!< サポートされていないメッセージシグナルです。
    InternalMcuResult_PayloadFormatInvalid      = 0x06,   //!< ペイロードのフォーマットが不正です。
    InternalMcuResult_PayloadLengthOutOfLength  = 0x07,   //!< ペイロードのサイズが超過しています。
    InternalMcuResult_StateTransitionInvalid    = 0x32,   //!< ステート遷移に失敗しました。
};

/**
 * @brief   MCU から受信した Common プロトコルデータの各フィールドの開始位置です。
 */
enum McuCommonInputOffsetByte : int
{
    McuCommonInputOffsetByte_MisoProtocol = 0,  //!< MisoProtocol
    McuCommonInputOffsetByte_ResultCode   = 1,  //!< リザルトコード
    McuCommonInputOffsetByte_Command      = 2,  //!< コマンド ID
    McuCommonInputOffsetByte_MajorVersion = 3,  //!< FW メジャーバージョン
    McuCommonInputOffsetByte_MinorVersion = 5,  //!< FW マイナーバージョン
    McuCommonInputOffsetByte_State        = 7   //!< MCU ステート
};

/**
 * @brief   MCU に送信する Common プロトコルデータの各フィールドの開始位置です。
 */
enum McuCommonOutputOffsetByte : int
{
    McuCommonOutputOffsetByte_ResultCode   = 0,  //!< リザルトコード
    McuCommonOutputOffsetByte_Command      = 1,  //!< コマンド ID
    McuCommonOutputOffsetByte_State        = 2   //!< MCU ステート
};

/**
* @brief   MCU に送信する EXT_CONTROL コマンドの各フィールドの開始位置です。
*/
enum McuExtControlOutputOffsetByte : int
{
    McuExtControlOutputOffsetByte_Header     = 0,  //!< Header
    McuExtControlOutputOffsetByte_Command    = 1,  //!< コマンド ID
    McuExtControlOutputOffsetByte_Setting    = 2   //!< Setting (bit 0 が VOUT 設定。残りは reserved)
};

/**
 * @brief   IAP 動作時のリザルトコードです。
 */
enum McuIapResultCode : uint8_t
{
    McuIapResultCode_Corrupted     = 0x5A,  //!< FW が壊れている
    McuIapResultCode_Validating    = 0xC3,  //!< CRC 検証中
    McuIapResultCode_ResetRequired = 0x69   //!< MCU リセットが必要
};

/**
 * @brief   IAP 動作時の各フィールドの開始位置です。
 */
enum McuIapInputOffsetByte : int
{
    McuIapInputOffsetByte_MisoProtocol       = 0,   //!< MisoProtocol
    McuIapInputOffsetByte_ResultCode         = 1,   //!< リザルトコード
    McuIapInputOffsetByte_InvertedResultCode = 2    //!< リザルトコードの反転
};

/**
 * @brief   Tera MCU からデータを取得できなかった場合に bluetooth から返るプロトコル値です。
 */
const uint8_t NoDataProtocolValue = 0xFF;

/**
 * @brief   IAP が動作している場合に Tera から返るプロトコル値です。
 */
const uint8_t IapProtocolValue = 0xAF;

/**
 * @brief   CRC32 の初期値です。
 */
const nn::Bit32 Crc32InitialValue = 0xFFFFFFFFul;

/**
 * @brief   Bluetooth のデータ転送サイズを表す構造体です。
 */
struct BluetoothTransferSize
{
    size_t send;     //!< Console -> MCU
    size_t receive;  //!< MCU -> Console
};

/**
 * @brief   送信パケットの通信モードです。
 */
struct PacketModePackForSend
{
    typedef ::nn::util::BitPack8::Field<0, 3, uint8_t> Protocol;        //!< 通信プロトコルのバージョン
    typedef ::nn::util::BitPack8::Field<3, 1, uint8_t> Reserved;        //   予約
    typedef ::nn::util::BitPack8::Field<4, 4, uint8_t> BluetoothMode;   //!< Bluetooth 通信モード
};

/**
 * @brief   MCU から受信したデータのプロトコルです。
 */
struct ReceiveProtocolType
{
    typedef ::nn::util::BitPack8::Field<0, 3, uint8_t> Protocol;    //!< 通信プロトコル
    typedef ::nn::util::BitPack8::Field<3, 5, uint8_t> CommandId;   //!< どの ID のコマンドに対する応答か

    ::nn::util::BitPack8 _storage;  //!< ビットフィールドの内部データ。アクセス非推奨

    /**
     * @brief   プロトコルの生値を取得します。
     */
    uint8_t GetValue() NN_NOEXCEPT
    {
        return static_cast<uint8_t>(_storage.GetMaskedBits(0xFF));
    }

    /**
     * @brief   プロトコルの生値を設定します。
     *
     * @param[in]   data    設定する値
     */
    void SetValue(uint8_t data) NN_NOEXCEPT
    {
        _storage.SetMaskedBits(0xFF, data);
    }

    /**
     * @brief   どの ID のコマンドに対する応答かを取得します。
     */
    uint8_t GetCommandId() NN_NOEXCEPT
    {
        return _storage.Get<CommandId>();
    }

    /**
     * @brief   通信プロトコルを取得します。
     */
    uint8_t GetProtocolType() NN_NOEXCEPT
    {
        return _storage.Get<Protocol>();
    }
};

/**
 * @brief   TERA_CONTROL コマンドのパラメータを格納する構造体です。
 */
struct TeraControlCommandType
{
    uint8_t state;
};

/**
 * @brief   LED の輝度設定に使用するフレームの最大数です。
 */
const int LedBrightnessFrameCountMax = 16;

/**
 * @brief   EXT_CONTROL コマンドのパラメータを格納する構造体です。
 */
struct ExtControlCommandType
{
    bool isVoutEnable;
    bool isLedEnable;
    int  frameCount;
    int  cycleCount;
    int  timeUnit;
    int  startBrightness;
    int  endBrightness;
    struct
    {
        int targetBrightness;
        int slopeTime;
        int holdTime;
    } frames[LedBrightnessFrameCountMax];
};

/**
 * @brief   Common コマンドの内容を格納する構造体です。
 */
struct CommonCommandType
{
    uint8_t command;    //!< 格納しているコマンドの種類
    union
    {
        TeraControlCommandType teraControl;
        ExtControlCommandType  extControl;
    };
};

/**
 * @brief   Common コマンドのレスポンスを格納する構造体です。
 */
struct CommonResponseType
{
    uint8_t result;         //!< 前回のコマンドの実行結果
    uint8_t command;        //!< 前回のコマンド
    struct
    {
        uint16_t major;     //!< MCU FW のメジャーバージョン
        uint16_t minor;     //!< MCU FW のマイナーバージョン
    } version;
    uint8_t state;          //!< MCU のステート

    /**
     * @brief   コマンドの内容をクリアします。
     */
    void Clear() NN_NOEXCEPT
    {
        result = 0;
        command = 0;
        version.major = 0;
        version.minor = 0;
        state = InternalMcuState_Initializing;
    };
};

/**
 * @brief   Bluetooth のモード毎の送受信サイズの一覧です。
 */
extern const BluetoothTransferSize BluetoothTransferSizeList[BluetoothMode_Max];


/**
 * @brief   バイナリを表示
 */
void DumpBinary(const void* pData, size_t dataSize) NN_NOEXCEPT;

/**
 * @brief   バイナリをアドレス情報つきで表示
 */
void DumpBinaryFormatted(const char* pData, size_t dataSize) NN_NOEXCEPT;

/**
 * @brief   VERBOSE モード時のみバイナリを表示
 */
NN_FORCEINLINE
void DumpBinaryVerbose(const void* pData, size_t dataSize) NN_NOEXCEPT
{
#ifdef VERBOSE
    DumpBinary(pData, dataSize);
#else
    NN_UNUSED(pData);
    NN_UNUSED(dataSize);
#endif
}

/**
 * @brief   システムブートローダで使用するチェックサムを計算
 */
char CalcCheckSum(const void* pData, size_t dataSize) NN_NOEXCEPT;

/**
 * @brief   CRC32 を計算します。
 *
 * @param[in]   pData           CRC 計算対象のデータ
 * @param[in]   dataSize        計算対象データのバイト数
 * @param[in]   initialValue    CRC 初期値
 *
 * @return  CRC32 の計算結果
 *
 * @details
 *  CRC 32 の計算仕様は下記の通りです。
 *   - 生成多項式       : 0x4C11DB7
 *   - 出力 XOR         : なし
 *   - ビットシフト方向 : 左送り
 *
 * @pre
 *  - @a pData が有効なアドレスを指している必要があります。
 *  - @a dataSize は 4 の倍数である必要があります。
 */
nn::Bit32 CalcCrc32(void* pData, size_t dataSize, nn::Bit32 initialValue) NN_NOEXCEPT;

/**
 * @brief   初期値を 0xFFFFFFFF に固定して CRC32 を計算します。
 *
 * @param[in]   pData       CRC 計算対象のデータ
 * @param[in]   dataSize    計算対象データのバイト数
 *
 * @return  CRC32 の計算結果
 *
 * @pre
 *  - @a pData が有効なアドレスを指している必要があります。
 *  - @a dataSize は 4 の倍数である必要があります。
 */
NN_FORCEINLINE
nn::Bit32 CalcCrc32(void* pData, size_t dataSize) NN_NOEXCEPT
{
    return CalcCrc32(pData, dataSize, Crc32InitialValue);
}

/**
 * @brief   CRC8 を計算します。
 *
 * @param[in]   pData       CRC 計算対象のデータ
 * @param[in]   dataSize    計算対象データのバイト数
 *
 * @return  CRC8 の計算結果
 *
 * @pre
 *  - @a pData が有効なアドレスを指している必要があります。
 */
nn::Bit8 CalcCrc8(const void* pData, size_t dataBytes) NN_NOEXCEPT;

/**
 * @brief   MCU 送信パケットの通信モード指定バイトを作成します。
 *
 * @param[in]   bluetoothMode   Bluetooth Mode
 * @param[in]   protocol        コマンドプロトコル
 *
 * @return  通信モードに指定するバイト
 */
uint8_t CreateMcuPacketModeByte(BluetoothMode bluetoothMode, CommandProtocol protocol) NN_NOEXCEPT;

/**
 * @brief   TERA_CONTROL NOP コマンドのペイロードを作成します。
 *
 * @param[out]  pOutCommand     コマンドの格納先
 * @param[in]   outSize         コマンド格納先のサイズ
 * @param[in]   bluetoothMode   Bluetooth Mode
 *
 * @return  ペイロードのバイト数
 */
size_t CreateTeraControlNopCommand(uint8_t* pOutCommand, size_t outSize, BluetoothMode mode) NN_NOEXCEPT;

/**
* @brief   EXT_CONTROL ON / OFF コマンドのペイロードを作成します。
*
* @param[out]  pOutCommand     コマンドの格納先
* @param[in]   outSize         コマンド格納先のサイズ
* @param[in]   bluetoothMode   Bluetooth Mode
* @param[in]   isOn            ON / OFF
*
* @return  ペイロードのバイト数
*/
size_t CreateTeraExtControlCommand(uint8_t* pOutCommand, size_t outSize, BluetoothMode mode, bool isOn) NN_NOEXCEPT;

/**
 * @brief   内部表現の @ref InternalMcuState を外部仕様の @ref McuState に変換します。
 *
 * @param[in]   mcuState    内部表現の @ref InternalMcuState
 *
 * @return  変換結果の @ref McuState の値
 */
McuState ConvertInternalMcuStateToExternal(InternalMcuState mcuState) NN_NOEXCEPT;

/**
 * @brief   外部仕様の @ref McuState を内部表現に変換します。
 *
 * @param[in]   mcuState    外部仕様の @ref McuState
 *
 * @return  変換結果の @ref InternalMcuState の値
 */
InternalMcuState ConvertMcuStateToInternal(nn::xcd::McuState mcuState) NN_NOEXCEPT;

/**
 * @brief   データがすべて 0 であるか確認します。
 *
 * @param[in]   pData       確認対象のデータ
 * @param[in]   dataSize    対象データのバイト数
 *
 * @return  すべて 0 であれば true
 *
 * @pre
 *  - @a pData が有効なアドレスを指している必要があります。
 */
bool IsAllZero(const void* pData, size_t dataSize) NN_NOEXCEPT;

/**
 * @brief   パケットの CRC8 が正しいか確認します。
 *
 * @param[in]   pData       確認対象のデータ
 * @param[in]   dataSize    確認対象のデータサイズ
 * @param[in]   mode        Bluetooth Mode
 *
 * @return  CRC8 が正しければ true
 *
 * @pre
 *  - @a pData が有効なアドレスを指している必要があります。
 */
bool IsCrc8Match(const uint8_t* pData, size_t dataSize, detail::BluetoothMode mode) NN_NOEXCEPT;

/**
 * @brief   デバイスが NFC を使用できる電源状態であるか確認します。
 *
 * @param[in]   pHandler    デバイスを管理するクラス
 *
 * @return  確認結果 (@ref nn::xcd::CheckNfcDevicePower と同じ)
 */
nn::Result CheckNfcDevicePowerImpl(nn::xcd::DeviceHandler* pHandler) NN_NOEXCEPT;

/**
 * @brief   デバイスが IR センサーを使用できる電源状態であるか確認します。
 *
 * @param[in]   pHandler    デバイスを管理するクラス
 *
 * @return  確認結果 (@ref nn::xcd::CheckIrDevicePower と同じ)
 */
nn::Result CheckIrDevicePowerImpl(nn::xcd::DeviceHandler* pHandler) NN_NOEXCEPT;

}}}  // nn::xcd::detail
