﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/gpio/gpio.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include "../xcd_ActivationCount.h"
#include "../xcd_IEventTask.h"
#include "../xcd_TaskManager.h"
#include "xcd_NwcpUartDriver-os.horizon.h"
#include "xcd_PowerSupplyManager-os.horizon.h"

namespace nn { namespace xcd { namespace detail{

class NwcpCharger final : public IEventTask
{
private:
    //!< 電源供給状態の変化を待ち受けるためのイベント
    nn::os::LightEventType m_PowerSupplyEvent;

    //!< 電池残量に変化があった場合に通知をうけるためのイベント
    nn::os::LightEventType m_BatteryLevelEvent;

    //!< 監視する GPIO
    nn::gpio::GpioPadName m_PadName;

    //!< PowerSupplyMangaer
    PowerSupplyManager* m_pPowerSupplyManager;

    //!< NwcpUartDriver
    NwcpUartDriver* m_pUartDriver;

    //!< NwcpCharger がアクティベートされた回数
    ActivationCount m_ActivationCount;

    //!< 充電制御状態
    bool m_IsCharging;

    //!< 電源供給の有効/無効の制御状態
    bool m_IsPowerSupplyEnabled;

    //!< 充電制御処理を行うかどうか
    bool m_IsChargerManagementEnabled;

    //!< 対象のハードウェアが SDEV かどうか
    bool m_IsSdev;

public:
    explicit NwcpCharger(nn::gpio::GpioPadName padName) NN_NOEXCEPT;
    virtual ~NwcpCharger() NN_NOEXCEPT NN_OVERRIDE;

    //!< PowerSupplyManager をセットする
    void SetPowerSupplyManager(PowerSupplyManager* pManager) NN_NOEXCEPT;

    //!< NwcpUartDriver をセットする
    void SetNwcpUartDriver(NwcpUartDriver* pDriver) NN_NOEXCEPT;

    Result Activate() NN_NOEXCEPT;
    Result Deactivate() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< 強制的に充電/給電を開始します
    void StartPowerSupply() NN_NOEXCEPT;

    //!< 強制的に充電/給電を終了します
    void StopPowerSupply() NN_NOEXCEPT;

    //!< 充電制御を開始します。
    void StartChargeStateManagement() NN_NOEXCEPT;

    //!< 充電制御を終了します。
    void StopChargeStateManagement() NN_NOEXCEPT;

    //!< NwcpCharger の処理を Resume します。
    void Resume() NN_NOEXCEPT;

    //!< NwcpCharger の処理を Suspend します。
    void Suspend() NN_NOEXCEPT;

    //!< Suspend の最終段の処理。
    void SuspendLow() NN_NOEXCEPT;

    //!< 本体非給電状態に充電制御のために、内部の充電状態をリセットします
    void ResetChargeState() NN_NOEXCEPT;

private:
    //!< 充電制御を更新します。
    void UpdateChargeState() NN_NOEXCEPT;

    //!< 電源の制御を行います
    void ControlPowerSupply(bool supplyEnabled) NN_NOEXCEPT;

    //!< 充電状態を制御します
    void ControlChargeState(bool chargeEnabled) NN_NOEXCEPT;

    //!< 電源供給スイッチの制御を行います
    void ControlPowerSwitch(bool supplyEnabled) NN_NOEXCEPT;

    //!< 本体非給電状態において充電をおこなうかどうか判定する
    bool IsPowerSupplyRequiredOnNonPoweredState(BatteryLevel batteryLevel) NN_NOEXCEPT;

};

}}} // namespace nn::xcd::detail
