﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_TeraNfc.h"
#include "xcd_NfcParser.h"

//#define VERBOSE

/*
 * VERBOSE モード時のみ、引数に指定された文字列を、Tera のログとして出力します。
 */
#ifdef VERBOSE

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_DETAIL_PARSER_LOG(...)           NN_DETAIL_XCD_INFO("[NfcParser] " ##__VA_ARGS__)
#else
#define NN_XCD_DETAIL_PARSER_LOG(format, ...)   NN_DETAIL_XCD_INFO("[NfcParser] " format, ##__VA_ARGS__)
#endif

#else  // VERBOSE

#define NN_XCD_DETAIL_PARSER_LOG(...)           static_cast<void>(0)

#endif  // VERBOSE

namespace nn { namespace xcd { namespace detail {

namespace
{

/**
 * @brief   タグ情報の生データを上位に渡す用の形式に変換
 *
 * @param[out]  pOutTagInfo     生成したパケットのサイズ
 * @param[in]   packedTagInfo   生成したパケットデータの格納先
 *
 * @pre
 *  - pOutTagInfo != nullptr
 *
 * @details
 *  内部表現でパックされた TagInfo を、上位レイヤーが参照するための構造体に格納する。
 */
void ConvertTagInfoForOutput(
    NfcTagInfo* pOutTagInfo,
    const NfcPackedTagInfo& packedTagInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);

    pOutTagInfo->protocol     = static_cast<nn::xcd::NfcProtocol>(packedTagInfo.nfcDiscoveryType);
    pOutTagInfo->tagType      = ConvertTagType(packedTagInfo.nfcDiscoveryType, packedTagInfo.protocolType);
    pOutTagInfo->tagId.length = std::min<int>(packedTagInfo.uidLength, nn::xcd::NfcUidLengthMax);
    std::memcpy(pOutTagInfo->tagId.uid, packedTagInfo.uid, pOutTagInfo->tagId.length);
}

/**
 * @brief   タグ検知イベントの情報を取得
 */
nn::Result GetNfcInfoForTagDetect(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);

    pOutInfo->reason = nn::xcd::NfcEventReason_Detected;

    const auto& stateInfo = status.stateInfo;
    if (stateInfo.detectedTagNumber <= 0)
    {
        // サイズ 0 で返すと、上位で 0 埋めされた TagInfo 扱いになる。
        // 0 埋めされると NfcTagType が Error になり、不正なタグを検出したような動作になる。
        // XXX: Fail すべき？
        *pOutSize = 0;
        NN_XCD_DETAIL_PARSER_LOG("WARNING: DetectedTagNumber is 0. Unexpected.\n");
        NN_RESULT_SUCCESS;
    }

    // 今はタグ 1 枚固定
    const auto& tag = stateInfo.tags[0];

    // 検出したタグの情報を取得
    auto& tagInfo = pOutInfo->ntagData.tagInfo;
    ConvertTagInfoForOutput(&tagInfo, tag);
    *pOutSize = sizeof(NfcTagInfo);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   タグ喪失イベントの情報を取得
 */
nn::Result GetNfcInfoForTagLost(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);

    // 特に付加情報なし
    // (喪失したタグに関する情報を取得しても良いが、やや実装が複雑化する)
    pOutInfo->reason = nn::xcd::NfcEventReason_Deactivated;
    *pOutSize = 0;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   特に取得すべき情報がない場合に使用するパーサー
 */
nn::Result GetNfcInfoForNone(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);
    NN_UNUSED(tagData);

    // 特に付加情報なし
    pOutInfo->reason = nn::xcd::NfcEventReason_None;
    *pOutSize = 0;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   発生したエラーの情報を取得
 */
nn::Result GetNfcInfoForError(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(tagData);

    // 発生したエラーコードを取得
    pOutInfo->reason = nn::xcd::NfcEventReason_Error;
    pOutInfo->errorInfo.resultCode = status.resultCode;
    *pOutSize = sizeof(NfcErrorInfo);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   受信したタグデータの内容を取得
 */
nn::Result GetNfcInfoForDataReceived(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);

    pOutInfo->reason = nn::xcd::NfcEventReason_ReadFinish;

    // 読み込んだタグの情報を取得
    const auto& detectedTag = tagData.type2Read.header.tagInfo;
    auto& outTagData = pOutInfo->ntagData;
    ConvertTagInfoForOutput(&outTagData.tagInfo, detectedTag);
    outTagData.type2TagVersion = tagData.type2Read.header.t2tVersion;
    std::memcpy(
        outTagData.signature,
        tagData.type2Read.header.signature,
        detail::SignatureBytes);

    auto* pPayload = tagData.type2Read.payload;

    // 異常なブロック数が来た場合は、正常に読める範囲で扱う
    outTagData.blockCount = std::min<int>(
        std::max<int>(pPayload[0], 0),
        NtagReadBlockCountMax);
    if (outTagData.blockCount != pPayload[0])
    {
        NN_XCD_DETAIL_PARSER_LOG(
            "WARNING: Received illegal block count (%d). Rounded to %d.\n",
            pPayload[0],
            outTagData.blockCount);
    }

    // 各ブロックのパース
    size_t dataOffset = 1 + 2 * NtagReadBlockCountMax;
    for (int i = 0; i < NtagReadBlockCountMax; ++i)
    {
        auto& block = outTagData.readDataBlocks[i];
        std::memset(block.data, 0, sizeof(block.data));

        if (i >= outTagData.blockCount)
        {
            // 読み込んでいないブロックは無視
            continue;
        }

        int addressOffset = 1 + 2 * i;
        block.address.startPage = pPayload[addressOffset];
        block.address.endPage   = pPayload[addressOffset + 1];

        if (block.address.startPage > block.address.endPage)
        {
            // ページ範囲が異常なら無視
            NN_XCD_DETAIL_PARSER_LOG(
                "WARNING: Received invalid page address (%d-%d). Ignored.\n",
                block.address.startPage,
                block.address.endPage);
            continue;
        }

        // 異常なアドレスによってバッファをオーバーランしないように範囲を制限
        size_t dataSize = Type2TagPageSize * std::min<size_t>(
            (block.address.endPage - block.address.startPage + 1),
            NtagReadBlockPageCountMax);
        std::memcpy(block.data, &pPayload[dataOffset], dataSize);
        dataOffset += dataSize;
    }

    *pOutSize = sizeof(NfcNtagData);

#if 0
    NN_SDK_LOG("[xcd] Read raw data:\n");
    DumpBinaryFormatted(tagData.rawData, 128);
    NN_SDK_LOG("\n");
#endif

    NN_RESULT_SUCCESS;
}

/**
 * @brief   書き込み完了イベントの情報を取得
 */
nn::Result GetNfcInfoForDataWritten(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);
    NN_UNUSED(tagData);

    // 特に付加情報なし
    pOutInfo->reason = nn::xcd::NfcEventReason_WriteFinish;
    *pOutSize = 0;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   パススルーコマンドの結果を取得
 */
nn::Result GetNfcInfoForPassThru(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);

    pOutInfo->reason = nn::xcd::NfcEventReason_PassThruResult;

    // PassThru の応答を取得
    const auto& detectedTag = tagData.passThru.header.tagInfo;
    auto& passThruData = pOutInfo->passThruData;
    ConvertTagInfoForOutput(&passThruData.tagInfo, detectedTag);

    auto dataSize = std::min<size_t>(tagData.passThru.dataSize, NfcPassThruDataSizeMax);
    passThruData.responseSize = static_cast<uint32_t>(dataSize);

    auto* pData = tagData.passThru.data;
    std::memcpy(passThruData.responseData, pData, dataSize);

    // 空き領域は all 0
    size_t remainSize = NfcPassThruDataSizeMax - dataSize;
    if (remainSize > 0)
    {
        std::memset(&passThruData.responseData[dataSize], 0, remainSize);
    }

    *pOutSize = sizeof(NfcPassThruData);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   MIFARE コマンドの結果を取得
 */
nn::Result GetNfcInfoForMifare(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);

    // 最大ブロック数を超える応答が来た場合はログを残す
    auto& receivedData = tagData.mifare;
    if (tagData.mifare.blockCount > MifareReadBlockCountMax)
    {
        NN_XCD_DETAIL_PARSER_LOG(
            "Too many mifare blocks are read: %d\n",
            tagData.mifare.blockCount);
    }

    pOutInfo->reason = nn::xcd::NfcEventReason_MifareResult;

    // Mifare の応答を取得
    auto& mifareData = pOutInfo->mifareData;
    std::memset(&mifareData, 0, sizeof(NfcMifareData));

    mifareData.dataCount = std::min<int>(tagData.mifare.blockCount, MifareReadBlockCountMax);
    for (int i = 0; i < mifareData.dataCount; i++)
    {
        auto& outputBlock   = mifareData.readDataBlocks[i];
        auto& receivedBlock = receivedData.blocks[i];
        outputBlock.blockAddress = receivedBlock.address;
        std::memcpy(outputBlock.data, receivedBlock.data, MifareBlockBytes);
    }

    *pOutSize = sizeof(NfcMifareData);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   MIFARE 鍵書き込み完了イベントの情報を取得
 */
nn::Result GetNfcInfoForMifareKeyWritten(
    size_t* pOutSize,
    NfcInfo* pOutInfo,
    const StatusForGetInfo& status,
    const ResponseTagData& tagData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_UNUSED(status);
    NN_UNUSED(tagData);

    // 特に付加情報なし
    pOutInfo->reason = nn::xcd::NfcEventReason_MifareKeyWriteFinish;
    *pOutSize = 0;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   NFC のイベントに応じたパーサーの一覧
 */
const struct
{
    InternalNfcEventType                    eventType;  //!< NFC のイベントの種類
    NfcParser::ParserFunctionTypeForGeneral parser;     //!< イベントに紐付くパーサー
} ParserList[] =
{
    { InternalNfcEventType::Error,            GetNfcInfoForError },
    { InternalNfcEventType::DataReceived,     GetNfcInfoForDataReceived },
    { InternalNfcEventType::DataWritten,      GetNfcInfoForDataWritten },
    { InternalNfcEventType::PassThru,         GetNfcInfoForPassThru },
    { InternalNfcEventType::Mifare,           GetNfcInfoForMifare },
    { InternalNfcEventType::MifareKeyWritten, GetNfcInfoForMifareKeyWritten },
    { InternalNfcEventType::ModuleCheck,      nullptr },    // 非対応
    { InternalNfcEventType::ModuleSpi,        nullptr }     // 非対応
};

}  // anonymous

bool NfcParser::IsValidReceivedPacket(
    const uint8_t* pPacket,
    size_t packetLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPacket);

    const auto& header = *reinterpret_cast<const NfcPackedPacketHeader*>(pPacket);
    const size_t payloadLengthMax = packetLength - sizeof(NfcPackedPacketHeader) - 1;

    // 異常なペイロード長の場合は不正パケット
    if (header.GetPayloadLength() > payloadLengthMax)
    {
        return false;
    }

    // コマンドコードが異常値の場合は不正パケット
    if (header.command == 0)
    {
        return false;
    }

    return true;
}

NfcParser::ParserFunctionTypeForDetect NfcParser::GetParserForDetectEvent(
    InternalNfcEventType eventType) NN_NOEXCEPT
{
    switch (eventType)
    {
    case InternalNfcEventType::TagDetect:
        return GetNfcInfoForTagDetect;
    case InternalNfcEventType::TagLost:
        return GetNfcInfoForTagLost;
    default:
        return nullptr;
    }
}

NfcParser::ParserFunctionTypeForGeneral NfcParser::GetParserForGeneralEvent(
    InternalNfcEventType eventType) NN_NOEXCEPT
{
    // 渡されたイベントに対応するパーサーを返す
    for (const auto item : ParserList)
    {
        if (item.eventType != eventType)
        {
            continue;
        }

        // 非対応イベントはログを残しつつ何もしない
        if (item.parser == nullptr)
        {
            NN_XCD_DETAIL_PARSER_LOG(
                "Unsupported event: %d\n",
                static_cast<int>(eventType));
            break;
        }

        return item.parser;
    }

    // パーサーが定義されていない場合は何もしない
    return GetNfcInfoForNone;
}

}}}  // nn::xcd::detail
