﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_NfcTypes.h>
#include "xcd_TeraCommon.h"
#include "xcd_TeraNfc.h"
#include "xcd_TimeCounter.h"
#include "xcd_TypedQueue.h"

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   NFC のコマンド生成、送受信のシーケンス管理を行うクラスです。
 */
class NfcCommandHandler
{
    NN_DISALLOW_COPY(NfcCommandHandler);
    NN_DISALLOW_MOVE(NfcCommandHandler);

public:
    /**
     * @brief   送受信するデータのサイズを取得
     */
    static const BluetoothTransferSize& GetTransferSize() NN_NOEXCEPT
    {
        return BluetoothTransferSizeList[BluetoothMode_Nfc];
    }

public:
    NfcCommandHandler() NN_NOEXCEPT :
        m_Mutex(true),
        m_SendTimeoutCounter(),
        m_CurrentCommand(),
        m_SequenceNumber(0),
        m_AckNumber(0),
        m_ProcessingState(CommandProcessingState::Idle),
        m_ExpectedState(NfcState_CommandWaiting),
        m_CommandQueue()
    {}

    /**
     * @brief   状態のクリア
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   予約済みのコマンドの発行をやめる
     */
    void CancelAllCommand() NN_NOEXCEPT;

    /**
     * @brief   受信完了待ちを開始
     */
    void StartWaitingReceive(NfcState expectedState) NN_NOEXCEPT;

    /**
     * @brief   受信完了の確認
     */
    bool CheckReceiveFinished(InternalNfcEventType* pOutEventType, NfcState state) NN_NOEXCEPT;

    /**
     * @brief   パケット送受信の成否判定と再送処理
     *
     * @param[in]   packetHeader    受信したパケットのヘッダ
     * @param[in]   commonHeader    受信したパケットの共通応答ヘッダ
     *
     * @retval true     送受信成功
     * @retval false    受信失敗
     */
    bool CheckPacketTransferSuccess(
        const NfcPackedPacketHeader& packetHeader,
        const NfcPackedCommonResponseHeader& commonHeader) NN_NOEXCEPT;

    /**
     * @brief   次に送信するコマンドの準備
     *
     * @param[out]  pOutNeedsClearTagData   受信済みのタグデータを消去する必要があるか
     * @param[in]   currentState            現在の NFC ステート
     */
    void PrepareNextCommand(bool* pOutNeedsClearTagData, NfcState currentState) NN_NOEXCEPT;

    /**
     * @brief   次に送信するコマンドのパケットを生成
     *
     * @param[out]  pOutPacketSize  生成したパケットのサイズ
     * @param[out]  pOutPacketData  生成したパケットデータの格納先
     * @param[in]   maxSize         パケットサイズの上限
     *
     * @pre
     *  - pOutPacketSize != nullptr
     *  - pOutPacketData != nullptr
     *
     * @details
     *  送信バッファ内のデータから、次に送信するコマンドパケットを生成する。
     */
    void CreateNextCommandPacket(size_t* pOutPacketSize, uint8_t* pOutPacketData, size_t maxSize) NN_NOEXCEPT;

    /**
     * @brief   タグの検出開始
     *
     * @param[in]   nfcDiscoveryParam   NFC_START_DISCOVERY のパラメータ
     */
    nn::Result StartDiscovery(const NfcDiscoveryParameter& nfcDiscoveryParam) NN_NOEXCEPT;

    /**
     * @brief   タグの検出停止
     */
    nn::Result StopDiscovery() NN_NOEXCEPT;

    /**
     * @brief   タグの読み取り開始
     *
     * @param[in]   ntagReadParam       NFC_READ_START のパラメータ
     */
    nn::Result StartNtagRead(const NtagReadParameter& ntagReadParam) NN_NOEXCEPT;

    /**
     * @brief   タグへの書き込み開始
     *
     * @param[in]   ntagWriteParam      NFC_WRITE_START のパラメータ
     */
    nn::Result StartNtagWrite(const NtagWriteParameter& ntagWriteParam) NN_NOEXCEPT;

    /**
     * @brief   パススルーコマンドの送信
     *
     * @param[in]   passThruParam       NFC_PASS_THRU_REQ のパラメータ
     */
    nn::Result SendRawData(const NfcPassThruParameter& passThruParam) NN_NOEXCEPT;

    /**
     * @brief   MIFARE 鍵の登録
     *
     * @param[in]   keyWriteParameter   鍵の書き込みパラメータ
     */
    nn::Result RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE 鍵の消去
     *
     * @param[in]   keyClearParameter   鍵の消去パラメータ
     */
    nn::Result ClearMifareKey(const MifareKeyClearParameter& keyClearParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグの読み取り開始
     *
     * @param[in]   readParameter   読み取りパラメータ
     */
    nn::Result StartMifareRead(const MifareReadParameter& readParameter) NN_NOEXCEPT;

    /**
     * @brief   MIFARE タグへの書き込み開始
     *
     * @param[in]   writeParameter  書き込みパラメータ
     */
    nn::Result StartMifareWrite(const MifareWriteParameter& writeParameter) NN_NOEXCEPT;

private:
    /**
     * @brief   コマンドの送受信状況。この状態に応じて、次のコマンド発行可否を判定する。
     */
    enum class CommandProcessingState
    {
        Idle,                   //!< NOP 送信中。次のコマンドを発行可能
        NeedWaitSend,           //!< 送信待ちが必要。他のコマンドは発行しない
        NeedWaitReceive,        //!< 受信待ちが必要。他のコマンドは発行しない
        WaitingSendFinish,      //!< 送信完了待機中。他のコマンドは発行しない
        WaitingReceiveFinish,   //!< 受信完了待機中。他のコマンドは発行しない
        SendingMultiPacket      //!< 分割パケットコマンドを送信中。他のコマンドは発行しない
    };

    static const size_t MaxPacketSize    = 1024;    //!< 1 パケットの最大サイズ
    static const int    CommandQueueSize = 2;       //!< コマンドキューのサイズ

private:
    mutable nn::os::Mutex   m_Mutex;                //!< メンバへのアクセスを排他するための Mutex

    TimeCounter             m_SendTimeoutCounter;   //!< コマンド送信のタイムアウト判定用カウンタ
    NfcCommandType          m_CurrentCommand;       //!< 現在送信中のコマンド
    int                     m_SequenceNumber;       //!< マルチパケットコマンド用シーケンス番号
    int                     m_AckNumber;            //!< パケット落ち検出のための ACK 番号
    CommandProcessingState  m_ProcessingState;      //!< コマンドの送受信状態
    NfcState                m_ExpectedState;        //!< 期待する遷移先ステート

    TypedQueue<NfcCommandType, CommandQueueSize> m_CommandQueue;    //!< コマンドを予約しておくキュー

private:
    /**
     * @brief   現在の SQN を取得
     */
    uint8_t GetSequenceNumber() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        return static_cast<uint8_t>(m_SequenceNumber);
    }

    /**
     * @brief   前回の SQN を取得
     */
    uint8_t GetLastSequenceNumber() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        return static_cast<uint8_t>(std::max(m_SequenceNumber - 1, 0));
    }

    /**
     * @brief   SQN を進める
     */
    void UpdateSequenceNumber() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_SequenceNumber = (m_SequenceNumber + 1) % 255;
    }

    /**
     * @brief   SQN を初期化
     */
    void ClearSequenceNumber() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_SequenceNumber = 0;
    }

    /**
     * @brief   現在のコマンドのパケット分割数を取得
     */
    int GetCurrentCommandPacketCount() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        const size_t PayloadLength = GetTransferSize().send - sizeof(NfcPackedPacketHeader) - 1;
        return std::max<int>(
            static_cast<int>((m_CurrentCommand.payloadLength + PayloadLength - 1) / PayloadLength), 1);
    }

    /**
     * @brief   送信成功判定
     *
     * @param[in]   packetHeader    受信したパケットのヘッダ
     * @param[in]   commonHeader    受信したパケットの共通応答ヘッダ
     */
    void CheckSendSuccess(
        const NfcPackedPacketHeader& packetHeader,
        const NfcPackedCommonResponseHeader& commonHeader) NN_NOEXCEPT;

    /**
     * @brief   受信成功判定
     */
    bool CheckReceiveSuccess(
        const NfcPackedPacketHeader& packetHeader) NN_NOEXCEPT;

    /**
     * @brief   パケット送信完了処理
     */
    void FinishSendingPacket() NN_NOEXCEPT;

    /**
     * @brief   複数分割パケットの送信成功判定
     *
     * @param[in]   mcuAckNumber    MCU から受信した ACKN
     *
     * @return  成否
     * @retval  true    送信成功
     * @retval  false   送信失敗
     *
     * @details
     *  自身の SQN と MCU から受信した ACKN から、コマンドの送信成否を判定する。
     */
    bool IsMultiPacketSendSuccess(int mcuAckNumber) const NN_NOEXCEPT;

    /**
     * @brief   受信成功判定
     *
     * @param[in]   mcuSequenceNumber   MCU から受信した SQN
     * @param[in]   isLastPacket        最終パケットか
     *
     * @return  成否
     * @retval  true    受信成功
     * @retval  false   受信失敗
     *
     * @details
     *  自身の ACKN と MCU から受信した SQN から、コマンドの受信成否を判定する。
     */
    bool IsReceiveSuccess(int mcuSequenceNumber, bool isLastPacket) const NN_NOEXCEPT;

    /**
     * @brief   次のコマンド送信が可能な状態か判定
     *
     * @param[in]   currentState        現在の NFC ステート
     *
     * @retval  true    送信可能
     * @retval  false   送信不可
     *
     * @details
     *  次のコマンドを送信可能か判定する。送信不可の場合は NFC_NOP を発行する必要がある。
     */
    bool IsNextCommandReady(NfcState currentState) const NN_NOEXCEPT;

    /**
     * @brief   次に送信するコマンドの準備
     *
     * @param[in]   command     次に送信するコマンド
     *
     * @details
     *  送信バッファにコマンドを登録する。
     */
    void SetupCommandForSend(
        bool* pOutNeedsClearTagData,
        const NfcCommandType& command) NN_NOEXCEPT;

    /**
     * @brief   コマンドパケットのペイロードを生成
     *
     * @param[out]  pOutPacketData  生成したパケットのデータ
     * @param[in]   maxSize         パケットサイズの上限
     * @param[in]   commandToSend   送信するコマンド
     *
     * @return  パケットのサイズ
     *
     * @pre
     *  - pOutPacketData != nullptr
     */
    size_t CreateCommandPacketPayload(
        uint8_t* pOutPacketData,
        size_t maxSize,
        const NfcCommandType& command) NN_NOEXCEPT;

    /**
     * @brief   コマンドの送信予約
     *
     * @param[in]   command     送信するコマンド
     *
     * @retresult
     *      @handleresult{nn::ResultSuccess,        処理に成功しました。}
     *      @handleresult{nn::xcd::ResultMcuBusy,   キューが埋まっているため、送信予約できません。}
     * @endretresult
     */
    nn::Result TryRegisterNextCommand(NfcCommandType& command) NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
