﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_BluetoothSettings.h>
#include <nn/xcd/xcd_Usb.h>

#include "xcd_BleCentralTask-hardware.nx.h"
#include "xcd_BluetoothHidTask-hardware.nx.h"
#include "xcd_HidAccessor.h"
#include "xcd_ILinkMonitorImpl.h"
#include "xcd_LinkMonitorTypes.h"
#include "xcd_NwcpManager-os.horizon.h"
#include "xcd_UsbHidManager-os.horizon.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの実装クラス
class LinkMonitorImpl final : public IEventTask, public ILinkMonitorImpl
{
    NN_DISALLOW_MOVE(LinkMonitorImpl);
    NN_DISALLOW_COPY(LinkMonitorImpl);

private:
    //!< BLE の接続状態を待ち受けるイベント
    ::nn::os::LightEventType m_BleEvent;

    //!< Bluetooth の接続状態を待ち受けるイベント
    ::nn::os::LightEventType m_BluetoothEvent;

    //!< Uart の接続状態を待ち受けるイベント
    ::nn::os::LightEventType m_UartEvent;

    //!< UsbHid の接続状態を待ち受けるイベント
    ::nn::os::LightEventType m_UsbHidEvent;

    //!< Suspend 開始を待ち受けるイベント
    ::nn::os::LightEventType m_SuspendEvent;

    //!< Nwcp の Suspend 処理の完了を待つイベント
    ::nn::os::LightEventType m_NwcpSuspendCompleteEvent;

    //!< UsbHid の Suspend 処理の完了を待つイベント
    ::nn::os::LightEventType m_UsbHidSuspendCompleteEvent;

    //!< Resume 処理の完了を待つイベント
    ::nn::os::LightEventType m_ResumeCompleteEvent;

    //!< デバイスリストに更新があった際に通知するためのイベント
    ::nn::os::LightEventType* m_pUpdateEvent;

    //!< BLE Central を制御するタスク
    BleCentralTask m_BleCentralTask;

    //!< Bluetooth を制御するタスク
    BluetoothHidTask m_BluetoothTask;

    //!< Nwcp を制御するタスク
    NwcpManager m_NwcpManager;

    //!< UsbHid を制御するタスク
    UsbHidManager m_UsbHidManager;

    //!< LinkMonitor がアクティベートされているかどうか
    bool m_Activated;

    //!< サスペンド処理の状態を表す enum
    enum SuspendState
    {
        SuspendState_Resumed,                //!< レジューム状態
        SuspendState_ToSuspend,              //!< サスペンドに移行する
        SuspendState_Suspended,              //!< サスペンド状態
        SuspendState_ToResume,               //!< レジュームに移行する
    };

    //!< サスペンド状態
    SuspendState m_SuspendState;

    //!< Ble の無効化フラグ
    bool m_IsBleDisabled;

public:
    LinkMonitorImpl() NN_NOEXCEPT;

    virtual ~LinkMonitorImpl() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態の監視を開始する
    virtual void StartMonitoring(nn::os::LightEventType* pUpdateEvent) NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態の監視を停止する
    virtual void StopMonitoring() NN_NOEXCEPT NN_OVERRIDE;

    //!< Suspend します。
    virtual void Suspend() NN_NOEXCEPT NN_OVERRIDE;

    //!< Resume します。
    virtual void Resume() NN_NOEXCEPT NN_OVERRIDE;

    //!< 接続済みデバイスのハンドルを取得する
    virtual size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT NN_OVERRIDE;

    //!< 接続済みデバイスのハンドルを取得する
    virtual size_t GetBleDevices(uint32_t *pOutValue, size_t deviceCount) NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスハンドルからHIDAccessorを取得する
    virtual HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT NN_OVERRIDE;

    //!< ハンドルから BleHidAccessor を取得する
    virtual BleHidAccessor* GetBleHidAccessor(uint32_t connectionHandle) NN_NOEXCEPT NN_OVERRIDE;

    //!< SlotSize の変更が起きた際に通知されるイベントを登録します
    virtual void SetSlotSizeUpdateEvent(DeviceHandle deviceHandle, ::nn::os::LightEventType* pEvent) NN_NOEXCEPT NN_OVERRIDE;

    //!< Bluetooth で使用するスロットサイズを設定します。
    virtual Result SetSlotSize(DeviceHandle deviceHandle, int slotsize) NN_NOEXCEPT NN_OVERRIDE;

    //!< Bluetooth で使用されている現在のスロットサイズを取得します。スロットサイズ変更に対応しないプラットフォームでは 0 を返します
    virtual int GetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT NN_OVERRIDE;

    //!< Bluetooth で使用する変更先のスロットサイズの値を取得します。スロットサイズ変更に対応しないプラットフォームでは 0 を返します
    virtual int GetTargetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT NN_OVERRIDE;

    //!< Bluetooth で使用するスロットサイズを変更中かどうか返します
    virtual bool IsChangingSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT NN_OVERRIDE;

    //!< 右レールへのデバイスの接続状態を取得する
    virtual bool IsRightNwcpAttached() NN_NOEXCEPT NN_OVERRIDE;

    //!< 左レールへのデバイスの接続状態を取得する
    virtual bool IsLeftNwcpAttached() NN_NOEXCEPT NN_OVERRIDE;

    //!< Nwcp プロトコルを有効にする
    virtual void SetNwcpEnabled(bool enabled) NN_NOEXCEPT NN_OVERRIDE;

    //!< 本体の給電状態を取得します。
    virtual bool IsConsolePowered() NN_NOEXCEPT NN_OVERRIDE;

    //!< ジョイコンのレール部の接続状態に変化があった場合に通知するイベントを登録します
    virtual void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT NN_OVERRIDE;

    //!< ジョイコンのレール部の接続状態に変化について、変化の内容を取得します
    virtual void GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddressRailUpdateEventType) NN_NOEXCEPT NN_OVERRIDE;

    //!< 左レール部のスリープ復帰要因を取得します
    virtual AwakeTriggerReason GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT NN_OVERRIDE;

    //!< 右レール部のスリープ復帰要因を取得します
    virtual AwakeTriggerReason GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT NN_OVERRIDE;

    //!< 指定された Usb デバイスをサポートしているか評価します
    virtual bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT NN_OVERRIDE;

    //!< USB デバイスを追加します
    virtual Result AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT NN_OVERRIDE;

    //!< USB デバイスを削除します
    virtual Result RemoveUsbHidDevice(int index) NN_NOEXCEPT NN_OVERRIDE;

    //!< Input Report を処理します
    virtual Result SetUsbHidInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT NN_OVERRIDE;

    //!< Output Report を取得します
    virtual size_t GetUsbHidOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT NN_OVERRIDE;

    //!< USB FullKeyController を有効にする
    virtual void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの USB 接続状態を取得する
    virtual bool IsUsbConnected(DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE;

    //!< USB デバイスの接続状態を更新します
    virtual void UpdateUsbHidDeviceLists() NN_NOEXCEPT NN_OVERRIDE;

    //!< Hid で待機状態のデバイスを再認識する
    virtual void ReInitializePendingUsbDevices() NN_NOEXCEPT NN_OVERRIDE;

    //!< Bluetooth の接続可能最大数を取得します
    virtual int GetMaxBluetoothLinks() NN_NOEXCEPT NN_OVERRIDE;

    //!< ファームウェア更新中のデバイスを指定します
    virtual Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE;

    //!< ファームウェア更新中のデバイス指定を解除します
    virtual void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT NN_OVERRIDE;

    //!< Ble を無効化します
    virtual void DisableBle() NN_NOEXCEPT NN_OVERRIDE;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    virtual void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT NN_OVERRIDE;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    virtual void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT NN_OVERRIDE;

    //!< 指定されたデバイスとペアリングする
    virtual Result PairPalma(const ::nn::bluetooth::Address& address) NN_NOEXCEPT NN_OVERRIDE;

    //!< Fifty が接続されているかどうかを取得します
    virtual bool IsFiftyConnected() NN_NOEXCEPT NN_OVERRIDE;

    //!< Fifty の入力状態を取得します
    virtual PadState GetFiftyPadState() NN_NOEXCEPT NN_OVERRIDE;

    //!< Kuina の Fw バージョンを取得します
    virtual Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT NN_OVERRIDE;

    //!< Kuina の Fw バージョンを要求します
    virtual Result RequestKuinaVersion(int index) NN_NOEXCEPT NN_OVERRIDE;

    //!< Kuina の Fw 更新モードに遷移させます
    virtual Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT NN_OVERRIDE;
};

}}} // namespace nn::xcd::detail
