﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/xcd/xcd_DeviceState.h>

namespace nn { namespace xcd { namespace detail{

struct BleNbatFlag
{
    typedef ::nn::util::BitFlagSet<32, BleNbatFlag>::Flag<0> IsActivated;                  //!< アクティベートされているかどうか
    typedef ::nn::util::BitFlagSet<32, BleNbatFlag>::Flag<1> IsBatteryLevelReadDone;       //!< 電池残量の読み込まれている
    typedef ::nn::util::BitFlagSet<32, BleNbatFlag>::Flag<2> IsBatteryStateReadDone;       //!< 充電/電源状態が読み込まれている
    typedef ::nn::util::BitFlagSet<32, BleNbatFlag>::Flag<3> IsPowered;                    //!< 給電中
    typedef ::nn::util::BitFlagSet<32, BleNbatFlag>::Flag<4> IsCharging;                   //!< 充電中
};

/**
 * @brief       BleNbatClient で扱うフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, BleNbatFlag> BleNbatFlagSet;

//!< Nintendo Hid over Gatt サービスのクライアント
class IBleNbatClient
{
protected:
    BatteryLevel m_BatteryLevel;
    BleNbatFlagSet m_Flags;

public:
    IBleNbatClient() NN_NOEXCEPT :
        m_BatteryLevel(BatteryLevel_None)
    {
        m_Flags.Reset();
    };
    virtual ~IBleNbatClient() NN_NOEXCEPT { /* 何もしない */ };

    //!< 最新の電池の状態を取得します。
    bool GetBatteryStatus(BatteryLevel* pOutLevel, bool* pOutIsPowered, bool* pOutIsCharging) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutLevel);
        NN_SDK_REQUIRES_NOT_NULL(pOutIsPowered);
        NN_SDK_REQUIRES_NOT_NULL(pOutIsCharging);
        if (m_Flags.Test<BleNbatFlag::IsBatteryLevelReadDone>() == false ||
            m_Flags.Test<BleNbatFlag::IsBatteryStateReadDone>() == false)
        {
            return false;
        }
        *pOutLevel = m_BatteryLevel;
        *pOutIsPowered = m_Flags.Test<BleNbatFlag::IsPowered>();
        *pOutIsCharging = m_Flags.Test<BleNbatFlag::IsCharging>();
        return true;
    };

    //!< 電池の状態を再取得します
    virtual void UpdateBatteryStatus() NN_NOEXCEPT
    {
        // 何もしない
    }
};

}}} // namespace nn::xcd::detail
