﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace xcd { namespace detail{

/**
 * @brief       BTM のリトライフラグの定義です。
 */
struct BtmRetryFlag
{
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<0> Retrying;          //!< リトライ中かどうか
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<1> GeneralScan;       //!< GeneralScan をリトライする
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<2> AutoConnect;       //!< AutoConnect をリトライする
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<3> PairGattServer;    //!< PairGattServer をリトライする
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<4> OverrideConnection; //!< OverrideConnection をリトライする
    typedef ::nn::util::BitFlagSet<32, BtmRetryFlag>::Flag<5> ConfigureBleMtu;   //!< ConfigureBleMtu をリトライする
};

/**
 * @brief       BTM のリトライで扱うフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, BtmRetryFlag> BtmRetryFlagSet;

/**
 * @brief       BTM のリトライをするまでの待ち時間です
 */
const ::nn::TimeSpan BluetoothRetryTimeSpan = ::nn::TimeSpan::FromMilliSeconds(100);

//!< リトライタイマーをセットする
template<int N, typename Tag>
void SetRetryTimer(::nn::os::TimerEventType* pEvent, ::nn::util::BitFlagSet<N, Tag>& flags, int index) NN_NOEXCEPT
{
    if (!flags.Test(Tag::Retrying::Index))
    {
        flags.Set(Tag::Retrying::Index);
        ::nn::os::StartOneShotTimerEvent(pEvent, BluetoothRetryTimeSpan);
    }
    flags.Set(index, true);
}

//!< Result Failure の場合に、リトライタイマーをセットする
template<int N, typename Tag>
void SetRetryTimerIfFail(const ::nn::Result& result, ::nn::os::TimerEventType* pEvent, ::nn::util::BitFlagSet<N, Tag>& flags, int index) NN_NOEXCEPT
{
    if (result.IsFailure())
    {
        SetRetryTimer(pEvent, flags, index);
    }
}

}}} // namespace nn::xcd::detail
