﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/btm/btm_Types.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_BluetoothSettings.h>
#include <nn/bluetooth/bluetooth_Types.h>
#include "xcd_BluetoothHidAccessor-hardware.nx.h"
#include "xcd_LinkMonitorTypes.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

const int HidDeviceCountMax = 8;

class BluetoothHidTask final : public IEventTask
{
    NN_DISALLOW_MOVE(BluetoothHidTask);
    NN_DISALLOW_COPY(BluetoothHidTask);

private:
    //!< Bluetooth Hid の通知を待ち受ける多重待ちイベント
    MultiWaitSystemEvent m_HidEvent;

    //!< BTM からの通知を待ち受ける多重待ちイベント
    nn::os::SystemEventType m_BtmEvent;

    //!< BTM が Busy の際にリトライするためのタイマーイベント
    nn::os::TimerEventType m_BtmRetryEvent;

    //!< デバイス接続監視の終了を通知するイベント
    nn::os::LightEventType* m_pUpdatedEvent;

    //!< HIDデバイスのリスト
    struct HidDeviceDescryptor
    {
        BluetoothHidAccessor accessor;
        nn::bluetooth::BluetoothAddress address;
        HidDeviceInfo deviceInfo;
        nn::btm::SlotMode currentSlotMode;
        nn::btm::SlotMode targetSlotMode;
        ::nn::os::LightEventType* pEvent;
        bool isChangingSlotMode;
    };

    HidDeviceDescryptor m_Devices[HidDeviceCountMax];

    // TSI 変更時に通知するイベント
    nn::os::SystemEventType *m_pBluetoothSettingsEvent;

    //!< 現在のBluetooth の通信モード
    BluetoothSettings m_CurrentBluetoothSettings;

    //!< サスペンド中かどうか
    bool m_IsSuspended;

    //!< 現在の BluetoothMode
    ::nn::btm::BluetoothMode m_BluetoothMode;

    //!< Bluetooth のユースケース変更処理中
    bool isOnBluetoothModeChange;

    //!< Bluetooth で許容されている最大接続台数
    int m_LinkCountMax;

public:
    BluetoothHidTask() NN_NOEXCEPT;
    virtual ~BluetoothHidTask() NN_NOEXCEPT NN_OVERRIDE;

    nn::Result Activate(nn::os::LightEventType* pUpdatedEvent) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< 接続済みデバイスのハンドルを取得する
    size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT;

    //!< デバイスハンドルからHIDAccessorを取得する
    HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< 指定したデバイスを切断する
    void DetachDevice(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< Bluetooth のモードを変更します。
    Result SetBluetoothSettings(BluetoothSettings bluetoothSettings, nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< 現在の Bluetooth のモードを取得します。
    BluetoothSettings GetBluetoothSettings() NN_NOEXCEPT;

    //!< SlotSize が変更になった際に通知するイベントを登録します
    void SetSlotSizeEvent(DeviceHandle deviceHandle, ::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< Bluetooth で使用されている現在のスロットサイズを取得します。
    int GetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< Bluetooth で使用する変更先のスロットサイズを取得します。
    int GetTargetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< Bluetooth で使用するスロットサイズを設定します。
    Result SetSlotSize(DeviceHandle deviceHandle, int slotsize) NN_NOEXCEPT;

    //!< Bluetooth で使用するスロットサイズを変更中かどうか返します
    bool IsChangingSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< Bluetooth の処理を Suspend します。
    void Suspend() NN_NOEXCEPT;

    //!< Bluetooth の処理を Resume します。
    void Resume() NN_NOEXCEPT;

    //!< Bluetooth の接続可能最大数を取得します
    int GetMaxLinks() NN_NOEXCEPT;

private:
    //!< bluetooth からのイベントをハンドリングする
    void HandleBluetoothEvent() NN_NOEXCEPT;

    //!< btm からのイベントをハンドリングする
    void HandleBtmEvent() NN_NOEXCEPT;

    //!< SetSlotMode の Retry をハンドリングする
    void HandleSetSlotModeRetry() NN_NOEXCEPT;

    //!< 切断されたデバイスを探索して、切断されていれば Remove する
    void HandleDetachedDevices(nn::btm::DeviceConditionList& list) NN_NOEXCEPT;

    //!< 接続されたデバイスがないか探索して、接続されていれば Register する
    void HandleAttachedDevices(nn::btm::DeviceConditionList& list) NN_NOEXCEPT;

    //!< SlotMode を更新する
    void HandleSlotModeUpdate(nn::btm::DeviceConditionList& list) NN_NOEXCEPT;

    //!< 接続されたデバイスを登録します
    bool RegisterDevice(nn::bluetooth::BluetoothAddress address, DeviceType deviceType) NN_NOEXCEPT;

    //!< 切断されたデバイスを削除します
    bool RemoveDevice(nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT;

    //!< SlotMode を変更します
    void ChangeSlotMode(nn::bluetooth::BluetoothAddress address, nn::btm::SlotMode slotMode) NN_NOEXCEPT;

    //!< Btm に対して SlotMode 変更コマンドを送信します
    void SetSlotModeToBtm(bool forceUpdate) NN_NOEXCEPT;

    //!< SlotMode の変更を中止
    void CancelChangingSlotMode() NN_NOEXCEPT;

    //!< Bluetooth アドレスから HidDeviceDescryptor を取得します
    HidDeviceDescryptor* GetDescryptorFromAddress(const nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
