﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Event.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/bluetooth/bluetooth_Types.h>

#include "../xcd_ReportTypes.h"
#include "xcd_HidAccessor.h"
#include "xcd_IHidListener.h"

namespace nn { namespace xcd { namespace detail{

const int HidAccessorBufferSize = 512;

//!< 汎用的なHidデバイスを扱うクラス
class BluetoothHidAccessor final : public HidAccessor
{
    NN_DISALLOW_MOVE(BluetoothHidAccessor);
    NN_DISALLOW_COPY(BluetoothHidAccessor);

private:
    //!< DeviceHandle
    DeviceHandle m_DeviceHandle;
    //!< HIDデバイスへのハンドル
    nn::bluetooth::BluetoothAddress m_Address;

    //!< InputReportを格納するバッファ
    uint8_t m_Buffer[HidAccessorBufferSize];
    size_t  m_ReceivedSize;

    //!< InputReport を解析する関数
    InputReportParserFunc m_pInputReportParserFunc;
    void* m_pInputReportParserArg;

    //!< GetReport が完了したときに通知するためのリスナー
    IHidListener* m_pGetReportListener;

    //!< SetReport が完了したときに通知するためのリスナー
    IHidListener* m_pSetReportListener;

    //!< Sniff の通信を行っているかどうか
    bool m_IsInSniff;

    //!< Sniff 中の通信周期
    ::nn::TimeSpan m_Interval;

    //!< Activateされているかどうか
    bool m_Activated;

public:
    BluetoothHidAccessor() NN_NOEXCEPT;
    virtual ~BluetoothHidAccessor() NN_NOEXCEPT NN_OVERRIDE;

    //!< Accessorを有効にする (環境依存)
    void Activate(DeviceHandle deviceHandle, nn::bluetooth::BluetoothAddress address) NN_NOEXCEPT;
    //!< Accessorを無効にする
    void Deactivate() NN_NOEXCEPT;
    //!< Activagteされているかどうか
    bool IsActivated() NN_NOEXCEPT
    {
        return m_Activated;
    }

    //!< 入力データを解析する
    void ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< GetReport が完了したときに呼ばれる関数
    void SetGetReportData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    //!< GetReport が失敗したときに呼ばれる関数
    void NotifyGetReportFailed(nn::bluetooth::BluetoothHhStatus status) NN_NOEXCEPT;

    //!< SetReport が完了したときに呼ばれる関数
    void SetSetReportResult(Result result) NN_NOEXCEPT;

    //!< サンプリングを開始
    virtual void StartSampling(InputReportParserFunc func, void* pArg) NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリングを停止
    virtual void StopSampling() NN_NOEXCEPT NN_OVERRIDE;

    //!< OutputReportをセットする
    virtual void SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< InputReportを読み出す
    virtual size_t GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    //!< SetReport を送信する
    virtual void SetReport(ReportType reportType, uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< GetReport を送信する
    virtual void GetReport(ReportType reportType, uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< 通信周期を取得する
    virtual Result GetInterval(::nn::TimeSpan* pOutInterval) NN_NOEXCEPT NN_OVERRIDE;

    //!< 通信周期をセットする
    void SetInterval(bool isPeriodical, ::nn::TimeSpan interval) NN_NOEXCEPT;

    //!< デバイスを切断する
    virtual void DetachDevice() NN_NOEXCEPT NN_OVERRIDE;

    DeviceHandle GetDeviceHandle() NN_NOEXCEPT
    {
        return m_DeviceHandle;
    }
};

}}} // namespace nn::xcd::detail
