﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_BluetoothSettings.h>
#include <nn/xcd/xcd_Pairing.h>

#include "../xcd_BluetoothTypes.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

class BluetoothDriver final : public IEventTask
{
    NN_DISALLOW_MOVE(BluetoothDriver);
    NN_DISALLOW_COPY(BluetoothDriver);

private:
    //!< ペアリングのタイムアウトをハンドリングするためのイベント
    nn::os::TimerEventType m_Event;

    //!< LLR 通知処理の状態変化をハンドリングするためのイベント
    nn::os::SystemEventType m_LlrNotifyEvent;

    //!< LLR 通知処理が失敗終了したことを上位に通知するためのイベント
    nn::os::EventType* m_pLlrCompleteEvent;

    //!< ペアリング中かどうかを示すフラグ
    bool m_OnPairing;

    //!< ペアリング制御を排他するための Mutex
    nn::os::Mutex m_Mutex;

    //!< LLR 通知を発行している対象の Bluetooth Address
    nn::bluetooth::Address m_LlrDeviceAddress;

    //!< Bluetoothがアクティブかどうか
    bool m_IsActivated;

public:
    BluetoothDriver() NN_NOEXCEPT;
    ~BluetoothDriver() NN_NOEXCEPT;

    //!< アクティブ化する
    void Activate() NN_NOEXCEPT;

    //!< 非アクティブ化する
    void Deactivate() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< ペアリング情報が更新された際に通知されるイベントを登録する
    void SetPairingUpdateEvent(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< ペアリングを開始する
    void StartButtonPairing() NN_NOEXCEPT;

    //!< 本体情報を取得する
    BluetoothDeviceInfo GetDeviceInfo() NN_NOEXCEPT;

    //!< 指定されたデバイスがペアリング済みかどうかをチェックする
    bool IsRegisteredDevice(const ::nn::bluetooth::Address address) NN_NOEXCEPT;

    //!< 登録されているデバイスのリストを取得する
    int GetRegisteredDevices(BluetoothDeviceInfo pOutValues[], int length) NN_NOEXCEPT;

    //!< ペアリング情報を登録する
    void RegisterBluetoothDevice(const BluetoothDeviceInfo& deviceInfo) NN_NOEXCEPT;

    //!< 指定したデバイスに接続する
    ::nn::Result SendConnectionTrigger(const ::nn::bluetooth::Address address, nn::os::EventType* pEvent) NN_NOEXCEPT;

    //!< ペアリング情報のリストの中での優先順位を変更する
    void IncreaseDeviceInfoOrder(const ::nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< ペアリング情報から削除されないように保護する
    void ProtectDeviceInfo(const ::nn::bluetooth::Address& address, bool protect) NN_NOEXCEPT;

    //!< サスペンドする
    void Suspend() NN_NOEXCEPT;

    //!< レジュームする
    void Resume() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
