﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Pairing.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BluetoothDriver.h"
#include "xcd_BluetoothUtil-hardware.nx.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail {

namespace {

int PairingDuration = 10; // ペアリングを継続する時間

}

BluetoothDriver::BluetoothDriver() NN_NOEXCEPT :
    m_OnPairing(false),
    m_Mutex(false),
    m_IsActivated(false)
{
    // 何もしない
}

BluetoothDriver::~BluetoothDriver() NN_NOEXCEPT
{
    // 何もしない
}

void BluetoothDriver::Activate() NN_NOEXCEPT
{
    m_IsActivated = true;

    nn::os::InitializeTimerEvent(&m_Event, nn::os::EventClearMode_ManualClear);
    nn::btm::AcquireLlrStateEvent(&m_LlrNotifyEvent);

    GetTaskManager().RegisterPeriodicTask(this);
}

void BluetoothDriver::Deactivate() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);
    nn::os::FinalizeTimerEvent(&m_Event);

    m_IsActivated = false;
}

void BluetoothDriver::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void BluetoothDriver::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitTimerEvent(&m_Event))
    {
        nn::os::ClearTimerEvent(&m_Event);

        ::std::lock_guard<decltype(m_Mutex)
        > locker(m_Mutex);

        if (m_OnPairing == true)
        {
            nn::btm::system::CancelGamepadPairing();
            m_OnPairing = false;
        }
    }

    if (nn::os::TryWaitSystemEvent(&m_LlrNotifyEvent))
    {
        nn::os::ClearSystemEvent(&m_LlrNotifyEvent);
        if (nn::btm::IsLlrStarted() == false)
        {
            nn::btm::DeviceConditionList deviceList;
            nn::btm::GetConnectedDeviceCondition(&deviceList);

            for (int i = 0; i < deviceList.deviceCount; i++)
            {
                if (ConvertToCommonBluetoothAddress<::nn::btm::BdAddress>(deviceList.device[i].bdAddress) == m_LlrDeviceAddress)
                {
                    // LLR が成功した
                    return;
                }
            }
            // LLR が失敗終了した
            if (m_pLlrCompleteEvent != nullptr)
            {
                nn::os::SignalEvent(m_pLlrCompleteEvent);
                m_pLlrCompleteEvent = nullptr;
            }
        }
    }
}

void BluetoothDriver::SetPairingUpdateEvent(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    ::nn::btm::RegisterSystemEventForRegisteredDeviceInfo(pEvent);
}

void BluetoothDriver::StartButtonPairing() NN_NOEXCEPT
{
    if (m_IsActivated == false)
    {
        return;
    }

    if (m_OnPairing == false)
    {
        ::std::lock_guard<decltype(m_Mutex)
                          > locker(m_Mutex);
        nn::btm::system::StartGamepadPairing();
        nn::os::StartOneShotTimerEvent(&m_Event, nn::TimeSpan::FromSeconds(PairingDuration));
        m_OnPairing = true;
    }
}

BluetoothDeviceInfo BluetoothDriver::GetDeviceInfo() NN_NOEXCEPT
{
    BluetoothDeviceInfo deviceInfo;

    ::nn::btm::HostDeviceProperty hostDeviceProperty;
    ::nn::btm::GetHostDeviceProperty(&hostDeviceProperty);

    deviceInfo.address = ConvertToCommonBluetoothAddress(hostDeviceProperty.bdAddress);
    std::memcpy(deviceInfo.bdName.name, hostDeviceProperty.bdName.name, sizeof(deviceInfo.bdName.name));
    std::memcpy(deviceInfo.classOfDevice.cod, &hostDeviceProperty.classOfDevice.cod, sizeof(hostDeviceProperty.classOfDevice.cod));
    deviceInfo.featureSet = hostDeviceProperty.featureSet;

    return deviceInfo;
}

bool BluetoothDriver::IsRegisteredDevice(const ::nn::bluetooth::Address address) NN_NOEXCEPT
{
    ::nn::btm::DeviceInfoList list;
    ::nn::btm::GetDeviceInfo(&list);

    for (int deviceIndex = 0; deviceIndex < list.deviceCount; deviceIndex++)
    {
        if (IsSameBluetoothAddress(ConvertToCommonBluetoothAddress(list.device[deviceIndex].bdAddress), address))
        {
            return true;
        }
    }

    return false;
}

int BluetoothDriver::GetRegisteredDevices(BluetoothDeviceInfo pOutValues[], int length) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(::nn::btm::DeviceInfo::bdAddress.address) >= sizeof(BluetoothDeviceInfo::address.address));
    NN_STATIC_ASSERT(sizeof(::nn::btm::DeviceInfo::bdName.name)       >= sizeof(BluetoothDeviceInfo::bdName.name));
    NN_STATIC_ASSERT(sizeof(::nn::btm::DeviceInfo::classOfDevice.cod) >= sizeof(BluetoothDeviceInfo::classOfDevice.cod));
    NN_STATIC_ASSERT(sizeof(::nn::btm::DeviceInfo::linkKey.key)       >= sizeof(BluetoothDeviceInfo::linkKey.key));

    ::nn::btm::DeviceInfoList list;
    ::nn::btm::GetDeviceInfo(&list);

    int returnCount = std::min<int>(list.deviceCount, length);

    for (int deviceIndex = 0; deviceIndex < returnCount; deviceIndex++)
    {
        pOutValues[deviceIndex].address = ConvertToCommonBluetoothAddress(list.device[deviceIndex].bdAddress);
        std::memcpy(pOutValues[deviceIndex].bdName.name, list.device[deviceIndex].bdName.name, sizeof(pOutValues[deviceIndex].bdName.name));
        std::memcpy(pOutValues[deviceIndex].classOfDevice.cod, list.device[deviceIndex].classOfDevice.cod, sizeof(pOutValues[deviceIndex].classOfDevice.cod));
        std::memcpy(pOutValues[deviceIndex].linkKey.key, list.device[deviceIndex].linkKey.key, sizeof(pOutValues[deviceIndex].linkKey.key));
    }

    return returnCount;
}

void BluetoothDriver::RegisterBluetoothDevice(const BluetoothDeviceInfo& deviceInfo) NN_NOEXCEPT
{
    ::nn::btm::DeviceInfo btmDeviceInfo;

    btmDeviceInfo.bdAddress = ConvertFromCommonBluetoothAddress<::nn::btm::BdAddress>(deviceInfo.address);
    std::memcpy(btmDeviceInfo.bdName.name, deviceInfo.bdName.name, sizeof(deviceInfo.bdName.name));
    std::memcpy(btmDeviceInfo.classOfDevice.cod, deviceInfo.classOfDevice.cod, sizeof(deviceInfo.classOfDevice.cod));
    std::memcpy(btmDeviceInfo.linkKey.key, deviceInfo.linkKey.key, sizeof(deviceInfo.linkKey.key));
    ::nn::btm::AddDeviceInfo(&btmDeviceInfo);
}

::nn::Result BluetoothDriver::SendConnectionTrigger(const ::nn::bluetooth::Address address, nn::os::EventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    if (IsRegisteredDevice(address) == false)
    {
        NN_RESULT_THROW(ResultBluetoothDeviceNotRegistered());
    }

    m_pLlrCompleteEvent = pEvent;

    auto btmAddress = ConvertFromCommonBluetoothAddress<::nn::btm::BdAddress>(address);
    m_LlrDeviceAddress = address;
    auto result = ::nn::btm::LlrNotify(&btmAddress);

    if (result.IsFailure())
    {
        if (::nn::btm::ResultBusy().Includes(result))
        {
            NN_RESULT_THROW(ResultBusy());
        }
        else
        {
            // 上記以外のリザルトは接続済み状態とみなす
            NN_RESULT_THROW(ResultBluetoothAlreadyConnected());
        }
    }

    NN_RESULT_SUCCESS;
}

void BluetoothDriver::IncreaseDeviceInfoOrder(const ::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    auto btmAddress = ConvertFromCommonBluetoothAddress<::nn::btm::BdAddress>(address);
    ::nn::btm::IncreaseDeviceInfoOrder(&btmAddress);
}

void BluetoothDriver::ProtectDeviceInfo(const ::nn::bluetooth::Address& address, bool protect) NN_NOEXCEPT
{
    auto btmAddress = ConvertFromCommonBluetoothAddress<::nn::btm::BdAddress>(address);
    ::nn::btm::ProtectDeviceInfo(&btmAddress, protect);
}

void BluetoothDriver::Suspend() NN_NOEXCEPT
{
    if (m_OnPairing)
    {
        nn::btm::system::CancelGamepadPairing();
        m_OnPairing = false;
    }
}

void BluetoothDriver::Resume() NN_NOEXCEPT
{
    // 何もしない
}

}}} // namespace nn::xcd::detail
