﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/btm/user/btm_UserTypes.h>

#include "xcd_BleGattUuids.h"
#include "../xcd_IBleCommandListener.h"
#include "xcd_IBleNhogClient.h"
#include "xcd_IBleGattClient-hardware.nx.h"
#include "xcd_BleGattClientCommon-hardware.nx.h"

namespace nn { namespace xcd { namespace detail{

//!< Nintendo Hid over Gatt サービスのクライアント
class BleNhogClient final : IBleGattClient, public IBleNhogClient
{
    NN_DISALLOW_MOVE(BleNhogClient);
    NN_DISALLOW_COPY(BleNhogClient);

private:
    //!< GATT クライアントで取り扱う Characteristic のリスト
    enum CharacteristicIndex
    {
        CharacteristicIndex_InputReport,
        CharacteristicIndex_OutputCommand,
        CharacteristicIndex_CommandResponse,

        CharacteristicIndex_Max
    };
    //!< Characteristic にアクセスするための情報
    GattClientAttributeParameter m_CharacteristicParameterList[CharacteristicIndex_Max];

    //!< GATT クライアントで取り扱う Descriptor のリスト
    enum DescriptorIndex
    {
        DescriptorIndex_InputReport_ClientConfig,
        DescriptorIndex_InputReport_ReportReference,
        DescriptorIndex_OutputCommand_ReportReference,
        DescriptorIndex_CommandResponse_ClientConfig,
        DescriptorIndex_CommandResponse_ReportReference,

        DescriptorIndex_Max
    };
    //!< Descriptor にアクセスするための情報
    GattClientAttributeParameter m_DescriptorParameterList[DescriptorIndex_Max];

    //!< Activate されているかどうか
    bool m_Activated;

    //!< BLE デバイスのコネクションハンドル
    uint32_t m_ConnectionHandle;
    //!< GATT サービスのハンドル
    uint16_t m_ServiceHandle;

    //!< GattOperation のキュー
    GattOperationConfig m_OperationQueue[MaxOperationCount];
    //!< キューにある GattOperation の数
    int m_OperationCount;
    //!< キューの先頭
    int m_OperationQueueHead;
    //!< キューの最後尾
    int m_OperationQueueTail;

    uint8_t m_InputReportFormatId;

    uint8_t m_OutputCommandFormatId;

    uint8_t m_CommandResponseFormatId;
public:
    BleNhogClient() NN_NOEXCEPT;
    virtual ~BleNhogClient() NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT サーバからの応答のハンドラ
    virtual bool GattOperationCompletedHandler(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントを有効にする
    virtual void Activate(uint32_t connectionHandle, const nn::btm::user::GattService& service) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントを無効にする
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    //!< Activate されているかどうか
    virtual bool IsActivated() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Activated;
    }

    //!< サービスの UUID を返す
    virtual const nn::bluetooth::GattAttributeUuid* GetServiceUuid() NN_NOEXCEPT NN_OVERRIDE
    {
        return &NhogService.Uuid;
    }

    //!< サービスのハンドルを返す
    virtual uint16_t GetServiceHandle() NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントの処理をポーリングする
    Result Proceed() NN_NOEXCEPT;

    //!< GATT サーバに OutputCommand を書き込む。IBleCommandListener::CharacteristicWriteComplete で書き込みの完了を通知します
    virtual void WriteOutputCommandCharacteristicAsync(const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT サーバから InputReport の ReportReference を読み込む。IBleCommandListener::DescriptorReadComplete で読み込みの完了を通知します
    virtual void ReadInputReportReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT サーバから OutputCommand の ReportReference を読み込む。IBleCommandListener::DescriptorReadComplete で読み込みの完了を通知します
    virtual void ReadOutputCommandReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT サーバから CommandResponse の ReportReference を読み込む。IBleCommandListener::DescriptorReadComplete で読み込みの完了を通知します
    virtual void ReadCommandResponseReportReferenceAsync(IBleCommandListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< InputReport の Notification の有効/無効を設定する。IBleCommandListener::DescriptorWriteComplete で完了を通知します
    virtual void EnableInputReportNotification(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandResponse の Notification の有効/無効を設定する。IBleCommandListener::DescriptorWriteComplete で完了を通知します
    virtual void EnableCommandResponseNotification(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandResponse の Indication の有効/無効を設定する。IBleCommandListener::DescriptorWriteComplete で完了を通知します
    virtual void EnableCommandResponseIndication(IBleCommandListener* pListener, bool isEnabled) NN_NOEXCEPT NN_OVERRIDE;

    //!< InputReport の FormatId を取得する
    virtual BleInputReportFormatVersion GetInputReportFormatVersion() NN_NOEXCEPT NN_OVERRIDE;

    //!< OutputCommand の FormatId を取得する
    virtual BleOutputCommandFormatVersion GetOutputCommandFormatVersion() NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandResponse の FormatId を取得する
    virtual BleCommandResponseFormatVersion GetCommandResponseFormatVersion() NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< GattClientAttributeParameter の値をセットする
    void SetGattClientAttributeParameterValue(bool isValid, CharacteristicIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic) NN_NOEXCEPT;

    void SetGattClientAttributeParameterValue(bool isValid, DescriptorIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic, const nn::btm::user::GattDescriptor& descriptor) NN_NOEXCEPT;

    //!< GattOperation キューにタスクを積む
    Result Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& status, IBleCommandListener* pListener) NN_NOEXCEPT;

    Result Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& status, const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT;

    //!< キューの先頭から要素を削除する
    void RemoveHead() NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_READ をハンドルする
    bool HandleReadCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_WRITE をハンドルする
    bool HandleWriteCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_NOTIFY をハンドルする
    bool HandleNotifyCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    const char* GetName(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
