﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/xcd/detail/xcd_Log.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace xcd { namespace detail{

struct GattClientAttributeParameter;

//!< GATT サーバに Characteristic のデータを要求する
inline Result ReadCharacteristic(uint32_t connectionHandle, const GattClientAttributeParameter& param, uint8_t authType) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bluetooth::LeClientReadCharacteristic(connectionHandle, param.service, param.isPrimary, param.characteristic, authType));
    NN_RESULT_SUCCESS;
}

//!< GATT サーバに Characteristic のデータを書き込む
inline Result WriteCharacteristic(uint32_t connectionHandle, const GattClientAttributeParameter& param, uint8_t authType, bool withResponse, const GattOperationPayload& payload) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(payload.dataptr);
    NN_RESULT_DO(
        nn::bluetooth::LeClientWriteCharacteristic(
            connectionHandle,
            param.service,
            param.isPrimary,
            param.characteristic,
            reinterpret_cast<const uint8_t*>(payload.dataptr),
            static_cast<uint16_t>(payload.length),
            authType,
            withResponse
        )
    );
    NN_RESULT_SUCCESS;
}

//!< GATT サーバの Notification の有効/無効を設定する
inline Result SetNotification(uint32_t connectionHandle, const GattClientAttributeParameter& param, bool enable) NN_NOEXCEPT
{
    if (enable == true)
    {
        NN_RESULT_DO(nn::bluetooth::LeClientRegisterNotification(connectionHandle, param.service, param.isPrimary, param.characteristic));
    }
    else
    {
        NN_RESULT_DO(nn::bluetooth::LeClientDeregisterNotification(connectionHandle, param.service, param.isPrimary, param.characteristic));
    }
    NN_RESULT_SUCCESS;
}

//!< GATT サーバの Indication の有効/無効を設定する
inline Result SetIndication(nn::btm::user::GattCharacteristic& characteristic, bool enable) NN_NOEXCEPT
{
    NN_UNUSED(characteristic);
    NN_UNUSED(enable);

    NN_RESULT_SUCCESS;
}

//!< GATT サーバに Descriptor のデータを要求する
inline Result ReadDescriptor(uint32_t connectionHandle, const GattClientAttributeParameter& param, uint8_t authType) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bluetooth::LeClientReadDescriptor(connectionHandle, param.service, param.isPrimary, param.characteristic, param.descriptor, authType));
    NN_RESULT_SUCCESS;
}

//!< GATT サーバに Descriptor のデータを書き込む
inline Result WriteDescriptor(uint32_t connectionHandle, const GattClientAttributeParameter& param, uint8_t authType, const GattOperationPayload& payload) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(payload.dataptr);
    NN_RESULT_DO(
        nn::bluetooth::LeClientWriteDescriptor(
            connectionHandle,
            param.service,
            param.isPrimary,
            param.characteristic,
            param.descriptor,
            reinterpret_cast<const uint8_t*>(payload.dataptr),
            static_cast<uint16_t>(payload.length),
            authType
        )
    );
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::xcd::detail
