﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>

namespace nn {
namespace wlan {

/*
 * ScanResultReaderBase
 */

ScanResultReaderBase::ScanResultReaderBase(const void* pBuffer, Bit8* pNext) NN_NOEXCEPT
: m_pHead(pBuffer), m_pNext(pNext)
{
    if (m_pHead != NULL && m_pNext == NULL)
    {
        m_pNext = const_cast<Bit8*>(static_cast<const Bit8*>(pBuffer) + sizeof(ScanResultHeader));
    }
}

ScanResultReaderBase::ScanResultReaderBase(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT
: m_pHead(pBuffer), m_pNext(NULL)
{
    ScanResultHeader* pHeader = reinterpret_cast<ScanResultHeader*>(const_cast<void*>(m_pHead));

    if( m_pHead != NULL && m_pNext == NULL )
    {
        m_pNext = const_cast<Bit8*>(static_cast<const Bit8*>(pBuffer) + sizeof(ScanResultHeader));
    }

    if( pHeader != NULL )
    {
        pHeader->bufferLength = bufferSize;
        pHeader->bssCount = 0;
        pHeader->resultLength = sizeof(ScanResultHeader);
    }
}


uint32_t ScanResultReaderBase::GetCount() const NN_NOEXCEPT
{
    const ScanResultHeader* pHeader = static_cast<const ScanResultHeader*>(m_pHead);

    if( pHeader != NULL )
    {
        return pHeader->bssCount;
    }
    else
    {
        return 0;
    }
}

size_t ScanResultReaderBase::GetTotalResultLength() const NN_NOEXCEPT
{
    const ScanResultHeader* pHeader = static_cast<const ScanResultHeader*>(m_pHead);
    if( pHeader != NULL )
    {
        return pHeader->resultLength;
    }
    else
    {
        return 0;
    }
}

size_t ScanResultReaderBase::GetBufferLength() const NN_NOEXCEPT
{
    const ScanResultHeader* pHeader = static_cast<const ScanResultHeader*>(m_pHead);
    if (pHeader)
    {
        return pHeader->bufferLength;
    }
    else
    {
        return 0;
    }
}

const Bit8* ScanResultReaderBase::ProcessBssPointer() NN_NOEXCEPT
{
    if( m_pHead != NULL )
    {
        if( m_pNext >= static_cast<const Bit8*>(m_pHead) + GetTotalResultLength() )
        {
            return NULL; // 終端ならNULLを返す
        }

        const BssInfo* pHeader = reinterpret_cast<const BssInfo*>(m_pNext);
        if( pHeader != NULL )
        {
            m_pNext += pHeader->bssSize; // 次要素のポインタへ進める
            return m_pNext;
        }
        else
        {
            return NULL;
        }
    }
    else
    {
        return NULL;
    }
}

uint32_t ScanResultReaderBase::GetDescriptions(const BssDescription* ppBssDescriptions[], uint32_t count) const NN_NOEXCEPT
{
    uint32_t maxCount = std::min<uint32_t>(count, GetCount());
    uint32_t i;
    const BssInfo* pHeader;
    for(i = 0, pHeader = reinterpret_cast<const BssInfo*>(static_cast<const Bit8*>(m_pHead) + sizeof(ScanResultHeader));
        pHeader != NULL && i < maxCount;
        ++i, pHeader = reinterpret_cast<const BssInfo*>(reinterpret_cast<const Bit8*>(pHeader) + pHeader->bssSize))
    {
        ppBssDescriptions[i] = reinterpret_cast<const BssDescription*>(pHeader);
    }
    return i;
}

bool ScanResultReaderBase::AddDescription(const BssDescription* pBssDescription) NN_NOEXCEPT
{
    ScanResultHeader* pHeader = reinterpret_cast<ScanResultHeader*>(const_cast<void*>(m_pHead));

    if( IsOverCapacity(pBssDescription) == true )
    {
        return false;
    }

    size_t length = GetBssDescriptionSize(pBssDescription);
    std::memcpy(reinterpret_cast<Bit8*>(pHeader) + pHeader->resultLength, pBssDescription, length);
    pHeader->bssCount += 1;
    pHeader->resultLength += length;
    return true;
}

bool ScanResultReaderBase::IsOverCapacity(const BssDescription* pBssDescription) NN_NOEXCEPT
{
    const ScanResultHeader* pHeader = static_cast<const ScanResultHeader*>(m_pHead);
    return (pHeader->bufferLength < pHeader->resultLength + GetBssDescriptionSize(pBssDescription)) ? true : false;
}

size_t ScanResultReaderBase::GetBssDescriptionSize(const BssDescription* pBssDescription) NN_NOEXCEPT
{
    return reinterpret_cast<const BssInfo*>(pBssDescription)->bssSize;
}


/*
 * BeaconScanResultReader
 */

const BeaconDescriptionReader BeaconScanResultReader::GetFirstDescription() NN_NOEXCEPT
{
    if( m_pHead != NULL )
    {
        m_pNext = const_cast<Bit8*>(static_cast<const Bit8*>(m_pHead) + sizeof(ScanResultHeader));
        return GetNextDescription();
    }
    else
    {
        return BeaconDescriptionReader(NULL);
    }
}

const BeaconDescriptionReader BeaconScanResultReader::GetNextDescription() NN_NOEXCEPT
{
    const Bit8* pCurrent = m_pNext;

    if (ProcessBssPointer() == NULL)
    {
        pCurrent = NULL;
    }

    return BeaconDescriptionReader(reinterpret_cast<const BssDescription*>(pCurrent));
}



} // end of namespace wlan
} // end of namespace nn

