﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ServerManager.h"

#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/vi/sf/vi_ServiceName.h>
#include <nn/vi/vi_LogicalResolution.h>
#include <hardware/hwcomposer_defs.h>
#include "visrv_Config.h"
#include "visrv_Log.h"
#include "service/visrv_ApplicationRootServiceHolder.h"
#include "service/visrv_SystemRootServiceHolder.h"
#include "service/visrv_ManagerRootServiceHolder.h"
#include "client/visrv_LayerPool.h"
#include "client/visrv_PresentationTracerPool.h"
#include "native/visrv_GetSurfaceComposerClient.h"
#include "client/visrv_SystemEventSplitter.h"
#include "master/detail/visrv_ExternalDisplay.h"

#include "indirect/visrv_IndirectCopyImageMapDefer.h"
#include "indirect/visrv_IndirectFlipManager.h"
#include "fbshare/visrv_SharingServerObject.h"
#include "local/visrv_LocalServerObject.h"

#include "settings/visrv_Settings.h"
#include "settings/visrv_PlatformConfig.h"
#include "settings/visrv_InterfaceType.h"

namespace nn{ namespace visrv{

    ServerManager g_ServerManager;

    //--------------------------------------------------

    ServerManager::DisplayServerManager::DisplayServerManager() NN_NOEXCEPT
        : m_pDeferExecutionHandler(nullptr)
        , m_pDeferExecutionUserPtr(nullptr)
    {
    }

    nn::Result ServerManager::DisplayServerManager::ProcessMyInvokeRequest(nn::os::MultiWaitHolderType* pSession) NN_NOEXCEPT
    {
        NN_RESULT_TRY(ProcessInvokeRequestWithDefer(pSession))
            NN_RESULT_CATCH(nn::sf::ResultProcessDeferred)
            {
                NN_SDK_ASSERT_NOT_NULL(m_pDeferExecutionHandler);
                m_pDeferExecutionHandler(this, pSession, m_pDeferExecutionUserPtr);
                m_pDeferExecutionHandler = nullptr;
                m_pDeferExecutionUserPtr = nullptr;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT();
            }
        NN_RESULT_END_TRY;

        NN_SDK_ASSERT(m_pDeferExecutionHandler == nullptr);
        NN_RESULT_SUCCESS;
    }

    void ServerManager::DisplayServerManager::SetDeferExecutionHandler(void (*f)(DisplayServerManager*, nn::os::MultiWaitHolderType*, void*), void* p) NN_NOEXCEPT
    {
        m_pDeferExecutionHandler = f;
        m_pDeferExecutionUserPtr = p;
    }

    //--------------------------------------------------

    namespace detail
    {
        void ProcessVsyncEvent(void* pArg) NN_NOEXCEPT
        {
            client::SystemEventSplitter::OnSourceEventSignaled(pArg);
            indirect::g_IndirectFlipManager.ProcessVsync();
        }
    }

    ServerManager::ServerManager() NN_NOEXCEPT
        : m_pDisplayServerManager(nullptr)
        , m_SystemEventWaitList()
    {
        std::memset(&m_DisplayServerManagerStorage, 0, sizeof(m_DisplayServerManagerStorage));
    }

    void ServerManager::Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_pDisplayServerManager);
        m_pDisplayServerManager  = new (&m_DisplayServerManagerStorage) DisplayServerManager;

        // Register service ports
        {
            // vi:u
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_pDisplayServerManager->RegisterObjectForPort(
                    service::ApplicationRootServiceHolder::Get(),
                    static_cast<int32_t>(SessionCountMax),
                    nn::vi::sf::ApplicationDisplayServiceName
                )
            );

            // vi:s
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_pDisplayServerManager->RegisterObjectForPort(
                    service::SystemRootServiceHolder::Get(),
                    static_cast<int32_t>(SessionCountMax),
                    nn::vi::sf::SystemDisplayServiceName
                )
            );

            // vi:m
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_pDisplayServerManager->RegisterObjectForPort(
                    service::ManagerRootServiceHolder::Get(),
                    static_cast<int32_t>(SessionCountMax),
                    nn::vi::sf::ManagerDisplayServiceName
                )
            );
        }

        // Initialize system events
        {
            m_SystemEventWaitList.Initialize();
            android::SurfaceComposerClient& client = native::GetSurfaceComposerClient();

            // VSYNC
            {
                nn::os::NativeHandle handle;
                bool isManaged;

                NN_ABORT_UNLESS_EQUAL(client.getVsyncEventHandle(handle, isManaged), android::OK);
                client::g_VsyncEventSplitter.Initialize(handle, isManaged);

                RegisterSystemEventWait(
                    0,
                    client::g_VsyncEventSplitter.Get()->GetSourceMultiWaitHolder(),
                    client::g_VsyncEventSplitter.Get()->GetSourceEvent(),
                    detail::ProcessVsyncEvent,
                    client::g_VsyncEventSplitter.Get()
                );
            }

            // HOTPLUG
            {
                nn::os::NativeHandle handle;
                bool isManaged;

                NN_ABORT_UNLESS_EQUAL(client.getHotplugEventHandle(handle, isManaged), android::OK);
                client::g_HotplugEventSplitter.Initialize(handle, isManaged);

                RegisterSystemEventWait(
                    1,
                    client::g_HotplugEventSplitter.Get()->GetSourceMultiWaitHolder(),
                    client::g_HotplugEventSplitter.Get()->GetSourceEvent(),
                    client::SystemEventSplitter::OnSourceEventSignaled,
                    client::g_HotplugEventSplitter.Get()
                );
            }
            // ERROR REPORT
            {
                nn::os::NativeHandle handle;
                bool isManaged;

                NN_ABORT_UNLESS_EQUAL(client.getErrorEventHandle(handle, isManaged), android::OK);
                client::g_ErrorReportEventSplitter.Initialize(handle, isManaged);

                RegisterSystemEventWait(
                    2,
                    client::g_ErrorReportEventSplitter.Get()->GetSourceMultiWaitHolder(),
                    client::g_ErrorReportEventSplitter.Get()->GetSourceEvent(),
                    client::SystemEventSplitter::OnSourceEventSignaled,
                    client::g_ErrorReportEventSplitter.Get()
                );
            }
            // RESOLUTION CHANGED
            {
                nn::os::SystemEvent* pEvent = master::detail::ExternalDisplay::GetModeChangedEvent();
                client::g_ExternalModeChangedSplitter.Initialize(pEvent->GetReadableHandle(), true);

                RegisterSystemEventWait(
                    3,
                    client::g_ExternalModeChangedSplitter.Get()->GetSourceMultiWaitHolder(),
                    client::g_ExternalModeChangedSplitter.Get()->GetSourceEvent(),
                    client::SystemEventSplitter::OnSourceEventSignaled,
                    client::g_ExternalModeChangedSplitter.Get()
                );
            }
        }

        // WAR: Both DCs are enabled at boot
        // Disable DC first so there isn't a race between nvnflinger setting up initial resolution
        // and viewport setup.  (Both viewport and current resolution are required for android APIs.)

        // These indices line up with how the DCs are setup on NX
        NN_STATIC_ASSERT(android::ISurfaceComposer::eDisplayIdMain == 0);
        NN_STATIC_ASSERT(android::ISurfaceComposer::eDisplayIdHdmi == 1);

        auto pConfig = settings::GetPlatformConfig();

        for( int i = 0; i < sizeof(pConfig->dcList) / sizeof(pConfig->dcList[0]); ++i )
        {
            if( pConfig->dcList[i].interface != settings::InterfaceType::Null )
            {
                auto pDisplay = native::GetSurfaceComposerClient().getBuiltInDisplay(i);
                NN_ABORT_UNLESS_NOT_NULL(pDisplay);
                NN_ABORT_UNLESS(native::GetSurfaceComposerClient().setDisplayPowerMode(pDisplay, HWC_POWER_MODE_OFF) == android::OK);
                native::GetSurfaceComposerClient().setDisplayProjection(pDisplay,
                                                                        0,
                                                                        android::Rect(0, 0, nn::vi::LogicalResolutionWidth, nn::vi::LogicalResolutionHeight),
                                                                        android::Rect(0, 0));
            }
        }

        m_pDisplayServerManager->Start();
    }

    void ServerManager::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pDisplayServerManager);

        client::g_ExternalModeChangedSplitter.Finalize();
        client::g_HotplugEventSplitter.Finalize();
        client::g_VsyncEventSplitter.Finalize();
        client::g_ErrorReportEventSplitter.Finalize();
        //m_SystemEventWaitList.Finalize();

        m_pDisplayServerManager->~DisplayServerManager();
        m_pDisplayServerManager = nullptr;
    }

    ServerManager::DisplayServerManager* ServerManager::GetDisplayServerManager() NN_NOEXCEPT
    {
        return m_pDisplayServerManager;
    }

    void ServerManager::Run() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pDisplayServerManager);
        auto pManager = m_pDisplayServerManager;

        for(;;)
        {
            auto pHolder = pManager->Wait();
            if(!pHolder)
            {
                NN_VISRV_LOG("Vi server stopping\n");
                break;
            }

            auto tag = pHolder->userData;

            if(tag == DisplayServerManager::InvokeTag)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessMyInvokeRequest(pHolder));
            }
            else if(tag == DisplayServerManager::AcceptTag)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessAcceptRequest(pHolder));
            }
            // System Event Split
            else if(tag >= MultiWaitIndex_SystemEventHandlerMin && tag <= MultiWaitIndex_SystemEventHandlerMax)
            {
                auto& e = m_SystemEventWaitList.Get(static_cast<int>(tag - MultiWaitIndex_SystemEventHandlerMin));
                NN_SDK_ASSERT_NOT_NULL(e.pEvent);
                if(nn::os::TryWaitSystemEvent(e.pEvent))
                {
                    if(e.onSignaledCallback)
                    {
                        e.onSignaledCallback(e.onSignaledParameter);
                    }
                }
                // link holder again
                pManager->AddUserWaitHolder(e.pHolder);
            }
            // MapDeferCopy
            else if(tag >= MultiWaitIndex_IndirectMapCopyImageMin && tag <= MultiWaitIndex_IndirectMapCopyImageMax)
            {
                indirect::g_IndirectMapCopyImageContextTable.ResumeExecution(pHolder);
            }
            // FlipManager
            else if(tag >= MultiWaitIndex_IndirectFlipMin && tag <= MultiWaitIndex_IndirectFlipMax)
            {
                indirect::g_IndirectFlipManager.ProcessSignal(pHolder);
            }
            // PresentationTracer
            else if(tag >= MultiWaitIndex_PresentationTracerMin && tag <= MultiWaitIndex_PresentationTracerMax)
            {
                client::g_PresentationTracerPool.ProcessMultiWaitSignal(pHolder, true);
            }
            // SharedLowLevelLayer
            else if(tag >= MultiWaitIndex_SharedLowLevelLayerMin && tag <= MultiWaitIndex_SharedLowLevelLayerMax)
            {
                fbshare::g_SharedLowLevelLayerManager.HandleMultiWait(pHolder);
            }
            // SharedClientLayer
            else if(tag >= MultiWaitIndex_SharedClientLayerMin && tag <= MultiWaitIndex_SharedClientLayerMax)
            {
                fbshare::g_SharedClientLayerManager.HandleMultiWait(pHolder);
            }
            // LocalRequestQueue
            else if(tag == MultiWaitIndex_LocalRequestQueue)
            {
                local::g_LocalRequestQueue.Process(pHolder, true);
            }
            else
            {
                NN_VISRV_LOG_ERR("ServerManager: Unknown MultiWaitHolder userData %lld(0x%llX)\n", tag, tag);
            }
        }
    }

    void ServerManager::RegisterSystemEventWait(
        int slotIndex,
        nn::os::MultiWaitHolderType* pHolder,
        nn::os::SystemEventType* pEvent,
        void (*onSignaledCallback)(void*),
        void* onSignaledParameter
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_REQUIRES_NOT_NULL(pEvent);
        NN_SDK_REQUIRES_RANGE(slotIndex, 0, detail::SystemEventWaitEntryCountMax);

        detail::SystemEventWaitEntry e = {};
        e.pHolder = pHolder;
        e.pEvent = pEvent;
        e.onSignaledCallback = onSignaledCallback;
        e.onSignaledParameter = onSignaledParameter;
        m_SystemEventWaitList.Set(slotIndex, e);
        pHolder->userData = static_cast<uintptr_t>(MultiWaitIndex_SystemEventHandlerMin + slotIndex);
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitIndex_SystemEventHandlerMin, MultiWaitIndex_SystemEventHandlerMax);
        m_pDisplayServerManager->AddUserWaitHolder(pHolder);
    }

    void ServerManager::UnregisterSystemEventWait(int index) NN_NOEXCEPT
    {
        auto& e = m_SystemEventWaitList.Get(index);
        nn::os::UnlinkMultiWaitHolder(e.pHolder);
        m_SystemEventWaitList.Remove(index);
    }

}}
