﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ProcessHeapBlockManager.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nn/os.h>
#include "visrv_Log.h"

namespace nn{ namespace visrv{

    ProcessHeapBlockManager g_ProcessHeapBlockManager;

    ProcessHeapBlockManager::ProcessHeapBlockManager() NN_NOEXCEPT
    {
        m_CurrentHeapSize = 0;
        std::memset(m_EntryList, 0, sizeof(m_EntryList));
    }

    void ProcessHeapBlockManager::Initialize() NN_NOEXCEPT
    {
        m_CurrentHeapSize = 0;
        std::memset(m_EntryList, 0, sizeof(m_EntryList));
        nn::os::InitializeMutex(&m_Mutex, false, 0);
    }

    void ProcessHeapBlockManager::Finalize() NN_NOEXCEPT
    {
        m_CurrentHeapSize = 0;
        std::memset(m_EntryList, 0, sizeof(m_EntryList));
        nn::os::SetMemoryHeapSize(0);
        nn::os::FinalizeMutex(&m_Mutex);
    }

    nn::Result ProcessHeapBlockManager::AllocateMemoryBlock(ResourceId* pOutBlockId, size_t size) NN_NOEXCEPT
    {
        NN_VISRV_LOG_MEM("Allocating process heap block (%llu bytes)\n", size);
        NN_RESULT_THROW_UNLESS(size > 0, nn::vi::ResultInvalidRange());

        size_t requestedSize = ((size + nn::os::MemoryBlockUnitSize - 1) / nn::os::MemoryHeapUnitSize) * nn::os::MemoryHeapUnitSize;

        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        // find emptry slot
        Entry* pEntry = nullptr;
        for(int i = 0; i < BlockCountMax; i++)
        {
            if(m_EntryList[i].blockId == 0)
            {
                pEntry = &m_EntryList[i];
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(pEntry != nullptr, nn::vi::ResultResourceLimit());

        uintptr_t address = 0;
        if(nn::os::AllocateMemoryBlock(&address, requestedSize).IsFailure())
        {
            // calculate new heap size
            size_t newSize = m_CurrentHeapSize + requestedSize;
            NN_VISRV_LOG_MEM("Expanding process heap block (%llu bytes)\n", newSize);

            NN_RESULT_TRY(nn::os::SetMemoryHeapSize(newSize))
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(nn::vi::ResultOperationFailed());
                }
            NN_RESULT_END_TRY;

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, requestedSize));

            m_CurrentHeapSize = newSize;
        }

        pEntry->blockId = AcquireResourceId();
        pEntry->pMemory = reinterpret_cast<void*>(address);
        pEntry->size    = requestedSize;
        *pOutBlockId = pEntry->blockId;
        NN_VISRV_LOG_MEM("Allocated process heap block #%lld addr=%llX\n", pEntry->blockId, address);
        NN_RESULT_SUCCESS;
    }

    nn::Result ProcessHeapBlockManager::FreeMemoryBlock(ResourceId blockId) NN_NOEXCEPT
    {
        NN_VISRV_LOG_MEM("Deallocating process heap block #%lld\n", blockId);
        NN_RESULT_THROW_UNLESS(blockId != 0, nn::vi::ResultInvalidRange());

        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        Entry* pEntry = nullptr;
        for(int i = 0; i < BlockCountMax; i++)
        {
            if(m_EntryList[i].blockId == blockId)
            {
                pEntry = &m_EntryList[i];
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(pEntry != nullptr, nn::vi::ResultNotFound());

        nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(pEntry->pMemory), pEntry->size);
        ReleaseResourceId(pEntry->blockId);


        pEntry->blockId = 0;
        pEntry->pMemory = nullptr;
        pEntry->size    = 0;

        // shrink heap size
        AdjustMemoryHeapSizeImpl();
        NN_RESULT_SUCCESS;
    }

    void ProcessHeapBlockManager::AdjustMemoryHeapSizeImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::os::IsMutexLockedByCurrentThread(&m_Mutex));

        for(int i = 0; i < BlockCountMax; i++)
        {
            if(m_EntryList[i].blockId != 0)
            {
                return;
            }
        }

        NN_VISRV_LOG_MEM("Shrinking process heap block to zero\n");
        // set zero if no block is allocated
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(0));
        m_CurrentHeapSize = 0;
    }

    nn::Result ProcessHeapBlockManager::GetMemory(void** pOutMemory, size_t* pOutSize, ResourceId blockId) NN_NOEXCEPT
    {
        NN_VISRV_LOG_MEM("Get memory of block #%lld\n", blockId);
        NN_RESULT_THROW_UNLESS(blockId != 0, nn::vi::ResultInvalidRange());

        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        Entry* pEntry = nullptr;
        for(int i = 0; i < BlockCountMax; i++)
        {
            if(m_EntryList[i].blockId == blockId)
            {
                pEntry = &m_EntryList[i];
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(pEntry != nullptr, nn::vi::ResultNotFound());

        *pOutMemory = pEntry->pMemory;
        *pOutSize   = pEntry->size;
        NN_VISRV_LOG_MEM("  ->addr=%llX, size=%llX\n", *pOutMemory, *pOutSize);
        NN_RESULT_SUCCESS;
    }

}}
