﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_VicImageBuffer.h"

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>
#include "detail/visrv_VicImageFormatInfo.h"

namespace nn{ namespace visrv{ namespace vic{

    namespace {
        void PrepareSurface(NvRmSurface* pOutSurface, const VicImageBuffer::InfoType& info) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(info.width, 0);
            NN_SDK_REQUIRES_GREATER(info.height, 0);

            NvColorFormat surfaceFormat = detail::VicImageFormatInfo::GetNvColorFormat(info.format);
            NvRmSurfaceLayout surfaceLayout = detail::VicImageFormatInfo::GetNvRmSurfaceLayout(info.format);

            const NvU32 surfaceAttributes[] = {
                NvRmSurfaceAttribute_Layout, surfaceLayout,
                NvRmSurfaceAttribute_None,
            };

            NvRmSurfaceSetup(
                pOutSurface,
                static_cast<NvU32>(info.width),
                static_cast<NvU32>(info.height),
                surfaceFormat,
                surfaceAttributes
            );
        }

        size_t CalculateRequiredMemorySize(NvRmSurface* pPreparedSurface) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pPreparedSurface);
            size_t size = static_cast<size_t>(NvRmSurfaceComputeSize(pPreparedSurface));
            size = nn::util::align_up(size, nn::os::MemoryPageSize);
            return size;
        }

        size_t CalculateRequiredMemoryAlignment(VicModule* pModule, NvRmSurface* pPreparedSurface) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pModule);
            NN_SDK_REQUIRES(pModule->IsInitialized());
            NN_SDK_REQUIRES_NOT_NULL(pPreparedSurface);
            return static_cast<size_t>(NvRmSurfaceComputeAlignment(pModule->GetDevice(), pPreparedSurface));
        }
    }

    size_t VicImageBuffer::GetRequiredMemorySize(const InfoType& info) NN_NOEXCEPT
    {
        NvRmSurface surface;
        PrepareSurface(&surface, info);
        return CalculateRequiredMemorySize(&surface);
    }

    size_t VicImageBuffer::GetRequiredMemoryAlignment(VicModule* pModule, const InfoType& info) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NvRmSurface surface;
        PrepareSurface(&surface, info);
        return CalculateRequiredMemoryAlignment(pModule, &surface);
    }

    VicImageBuffer::VicImageBuffer() NN_NOEXCEPT
        : m_pMemoryPool(nullptr)
        , m_Offset(0)
        , m_Size(0)
    {
        memset(&m_Info, 0, sizeof(m_Info));
        memset(&m_Surface, 0, sizeof(m_Surface));
    }

    nn::Result VicImageBuffer::Initialize(VicModule* pModule, const InfoType& info, VicMemoryPool* pMemoryPool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES(pModule->IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pMemoryPool);
        NN_SDK_REQUIRES(pMemoryPool->IsInitialized());

        NN_RESULT_THROW_UNLESS(offset >= 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(offset + size <= pMemoryPool->GetSize(), nn::vi::ResultInvalidRange());

        bool isSuccess = false;


        PrepareSurface(&m_Surface, info);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                memset(&m_Surface, 0, sizeof(m_Surface));
            }
        };
        size_t requiredSize      = CalculateRequiredMemorySize(&m_Surface);
        size_t requiredAlignment = CalculateRequiredMemoryAlignment(pModule, &m_Surface);
        NN_RESULT_THROW_UNLESS((offset % requiredAlignment) == 0, nn::vi::ResultOperationFailed());
        NN_RESULT_THROW_UNLESS(size >= requiredSize, nn::vi::ResultOperationFailed());

        m_Surface.hMem = pMemoryPool->GetHandle();
        m_Surface.Offset = static_cast<NvU32>(offset);

        m_Info = info;
        m_pMemoryPool = pMemoryPool;
        m_Offset = offset;
        m_Size = requiredSize;

        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void VicImageBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        memset(&m_Info, 0, sizeof(m_Info));
        memset(&m_Surface, 0, sizeof(m_Surface));
        m_pMemoryPool = nullptr;
        m_Offset = 0;
        m_Size = 0;
    }

    bool VicImageBuffer::IsInitialized() const NN_NOEXCEPT
    {
        return m_pMemoryPool != nullptr;
    }

    int VicImageBuffer::GetWidth() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_Info.width;
    }

    int VicImageBuffer::GetHeight() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_Info.height;
    }

    VicImageFormat VicImageBuffer::GetImageFormat() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_Info.format;
    }

    size_t VicImageBuffer::GetStride() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return static_cast<size_t>(m_Surface.Pitch);
    }

    const NvRmSurface* VicImageBuffer::GetSurface() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_Surface;
    }

    NvRmSurface* VicImageBuffer::GetSurface() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_Surface;
    }

    void VicImageBuffer::FlushCache() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        nn::os::FlushDataCache(reinterpret_cast<const void*>(m_pMemoryPool->GetAddress() + m_Offset), m_Size);
    }

}}}
