﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/vi/vi_Result.h>

#include "visrv_ContainerOption.h"

namespace nn{ namespace visrv{ namespace util{

    // T TNullValue::Get()
    template<typename T, int TSize, typename TNullValue, int TThreadSaftyOption>
    class TNullableContainer;

    template<typename T, int TSize, typename TNullValue>
    class TNullableContainer<T, TSize, TNullValue, ContainerThreadSafetyOption_SingleThread>
    {
    public:
        static const int Size = TSize;
        typedef T ValueType;

    public:
        TNullableContainer() NN_NOEXCEPT
        {
        };

        static ValueType GetInvalidValue() NN_NOEXCEPT
        {
            return TNullValue::Get();
        }

        void Initialize() NN_NOEXCEPT
        {
            for(int i = 0; i < Size; i++)
            {
                m_List[i] = GetInvalidValue();
            }
        }

        void Finalize() NN_NOEXCEPT
        {
        }

        void Clear() NN_NOEXCEPT
        {
            for(int i = 0; i < Size; i++)
            {
                m_List[i] = GetInvalidValue();
            }
        }

        // @pre value != InvalidValue
        nn::Result Register(ValueType value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(value, GetInvalidValue());

            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] == GetInvalidValue())
                {
                    m_List[i] = value;
                    NN_RESULT_SUCCESS;
                }
            }
            NN_RESULT_THROW(nn::vi::ResultResourceLimit());
        }

        // @pre value != InvalidValue
        nn::Result Unregister(ValueType value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(value, GetInvalidValue());

            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] == value)
                {
                    m_List[i] = GetInvalidValue();
                    NN_RESULT_SUCCESS;
                }
            }
            NN_RESULT_THROW(nn::vi::ResultNotFound());
        }

        int GetCount() const NN_NOEXCEPT
        {
            int count = 0;
            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] != GetInvalidValue())
                {
                    count++;
                }
            }
            return count;
        }

        // @pre value != InvalidValue
        bool Contains(ValueType value) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(value, GetInvalidValue());

            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] == value)
                {
                    return true;
                }
            }
            return false;
        }

        ValueType GetHead() const NN_NOEXCEPT
        {
            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] != GetInvalidValue())
                {
                    return m_List[i];
                }
            }
            return GetInvalidValue();
        }

        // @tparam bool (*F)(ValueType v) 削除する場合 true を返す
        // @return 削除した要素数
        // 条件にマッチするものをすべて削除します
        template<typename F>
        int RemoveIf(F cond) NN_NOEXCEPT
        {
            int count = 0;
            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] != GetInvalidValue())
                {
                    if(cond(m_List[i]))
                    {
                        m_List[i] = GetInvalidValue();
                        count++;
                    }
                }
            }
            return count;
        }

        // @tparam void (*F)(ValueType v) 有効な要素に対して呼び出す関数
        // すべての有効なオブジェクトに対して func を呼び出します。
        template<typename F>
        void Foreach(F func) const NN_NOEXCEPT
        {
            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] != GetInvalidValue())
                {
                    func(m_List[i]);
                }
            }
        }

    protected:
        // @tparam bool (*F)(ValueType v) 条件にマッチする場合 true を返す
        template<typename F>
        nn::Result FindImpl(ValueType* pOutValue, F cond) NN_NOEXCEPT
        {
            for(int i = 0; i < Size; i++)
            {
                if(m_List[i] != GetInvalidValue())
                {
                    if(cond(m_List[i]))
                    {
                        *pOutValue = m_List[i];
                        NN_RESULT_SUCCESS;
                    }
                }
            }
            NN_RESULT_THROW(nn::vi::ResultNotFound());
        }

    private:
        ValueType m_List[Size];
    };


    template<typename T, int TSize, typename TNullValue>
    class TNullableContainer<T, TSize, TNullValue, ContainerThreadSafetyOption_MultiThread>
    {
    public:
        static const int Size = TSize;
        typedef T ValueType;

    private:
        typedef TNullableContainer<T, TSize, TNullValue, ContainerThreadSafetyOption_SingleThread> CoreType;

    public:
        TNullableContainer() NN_NOEXCEPT
            : m_Core()
            , m_Mutex()
        {
        };

        void Initialize() NN_NOEXCEPT
        {
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            m_Core.Initialize();
        }

        void Finalize() NN_NOEXCEPT
        {
            m_Core.Finalize();
            nn::os::FinalizeMutex(&m_Mutex);
        }

        void Clear() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            return m_Core.Clear();
        }

        nn::Result Register(ValueType value) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            return m_Core.Register(value);
        }

        nn::Result Unregister(ValueType value) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            return m_Core.Unregister(value);
        }

        ValueType GetHead() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            return m_Core.GetHead();
        }

    protected:
        // bool (*F)(ValueType v)
        template<typename F>
        nn::Result FindImpl(ValueType* pOutValue, F cond) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            return m_Core.FindImpl(pOutValue, cond);
        }

    private:
        CoreType m_Core;
        nn::os::MutexType m_Mutex;
    };

}}}
