﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SystemDisplayServiceImpl.h"

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/vi/vi_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "../visrv_Config.h"
#include "../visrv_Log.h"

namespace nn{ namespace visrv{ namespace service{

    SystemDisplayServiceImpl::SystemDisplayServiceImpl(client::ClientObjectSmartHolder* pClientHolder) NN_NOEXCEPT
        : m_ClientHolder(pClientHolder)
    {
        NN_VISRV_LOG_IPC("s:ctor\n");
        NN_VISRV_LOG_LIFETIME("SystemDisplayServiceImpl ctor\n");
    }

    SystemDisplayServiceImpl::~SystemDisplayServiceImpl() NN_NOEXCEPT
    {
        NN_VISRV_LOG_IPC("s:dtor\n");
        NN_VISRV_LOG_LIFETIME("SystemDisplayServiceImpl dtor\n");
    }

    nn::Result SystemDisplayServiceImpl::GetZOrderCountMin(nn::sf::Out<std::int64_t> outValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetZOrderCountMin);
        return m_ClientHolder->GetZOrderCountMin(outValue.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::GetZOrderCountMax(nn::sf::Out<std::int64_t> outValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetZOrderCountMax);
        return m_ClientHolder->GetZOrderCountMax(outValue.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetLayerPosition(nn::vi::LayerId layerId, float x, float y) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetLayerPosition);
        return m_ClientHolder->SetLayerPosition(layerId, x, y);
    }

    nn::Result SystemDisplayServiceImpl::SetLayerSize(nn::vi::LayerId layerId, std::int64_t width, std::int64_t height) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetLayerSize);
        return m_ClientHolder->SetLayerSize(layerId, width, height);
    }

    nn::Result SystemDisplayServiceImpl::GetLayerZ(nn::sf::Out<std::int64_t> outZ, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetLayerZ);
        return m_ClientHolder->GetLayerZ(outZ.GetPointer(), layerId);
    }

    nn::Result SystemDisplayServiceImpl::SetLayerZ(nn::vi::LayerId layerId, std::int64_t z) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetLayerZ);
        return m_ClientHolder->SetLayerZ(layerId, z);
    }

    nn::Result SystemDisplayServiceImpl::SetLayerVisibility(nn::vi::LayerId layerId, bool isVisible) NN_NOEXCEPT
    {
        // Note:  Do not remove this function for IPC compatibility
        NN_VISRV_TRACE_IPC_S(SetLayerVisibility);
        NN_UNUSED(layerId);
        NN_UNUSED(isVisible);
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::SetLayerAlpha(nn::vi::LayerId layerId, float alpha) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetLayerAlpha);
        return m_ClientHolder->SetLayerAlpha(layerId, alpha);
    }

    nn::Result SystemDisplayServiceImpl::ListDisplayModes(nn::sf::Out<std::int64_t> outCount, const nn::sf::OutArray<nn::vi::DisplayModeInfo>& outModes, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(ListDisplayModes);
        return m_ClientHolder->ListDisplayModes(outCount.GetPointer(), outModes.GetData(), static_cast<int>(outModes.GetLength()), displayId);
    }

    nn::Result SystemDisplayServiceImpl::ListDisplayRgbRanges(nn::sf::Out<std::int64_t> outCount, const nn::sf::OutArray<nn::vi::RgbRangeType>& outRanges, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(ListDisplayRgbRanges);
        return m_ClientHolder->ListDisplayRgbRanges(outCount.GetPointer(), outRanges.GetData(), static_cast<int>(outRanges.GetLength()), displayId);
    }

    nn::Result SystemDisplayServiceImpl::ListDisplayContentTypes(nn::sf::Out<std::int64_t> outCount, const nn::sf::OutArray<nn::vi::ContentTypeType>& outTypes, nn::vi::DisplayId dispayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(ListDisplayContentTypes);
        return m_ClientHolder->ListDisplayContentTypes(outCount.GetPointer(), outTypes.GetData(), static_cast<int>(outTypes.GetLength()), dispayId);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayMode(nn::sf::Out<nn::vi::DisplayModeInfo> outMode, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayMode);
        return m_ClientHolder->GetDisplayMode(outMode.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayMode(nn::vi::DisplayId displayId, const nn::vi::DisplayModeInfo& mode) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayMode);
        return m_ClientHolder->SetDisplayMode(displayId, mode);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayUnderscan(nn::sf::Out<std::int64_t> outUnderscan, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayUnderscan);
        return m_ClientHolder->GetDisplayUnderscan(outUnderscan.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayUnderscan(nn::vi::DisplayId displayId, std::int64_t underscan) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayUnderscan);
        return m_ClientHolder->SetDisplayUnderscan(displayId, underscan);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayContentType(nn::sf::Out<nn::vi::ContentTypeType> outType, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayContentType);
        return m_ClientHolder->GetDisplayContentType(outType.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayContentType(nn::vi::DisplayId displayId, nn::vi::ContentTypeType type) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayContentType);
        return m_ClientHolder->SetDisplayContentType(displayId, type);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayRgbRange(nn::sf::Out<nn::vi::RgbRangeType> outRange, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayRgbRange);
        return m_ClientHolder->GetDisplayRgbRange(outRange.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayRgbRange(nn::vi::DisplayId displayId, nn::vi::RgbRangeType range) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayRgbRange);
        return m_ClientHolder->SetDisplayRgbRange(displayId, range);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayCmuMode(nn::sf::Out<nn::vi::CmuModeType> outMode, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayCmuMode);
        return m_ClientHolder->GetDisplayCmuMode(outMode.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayCmuMode(nn::vi::DisplayId displayId, nn::vi::CmuModeType mode) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayCmuMode);
        return m_ClientHolder->SetDisplayCmuMode(displayId, mode);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayContrastRatio(nn::sf::Out<float> outRatio, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayContrastRatio);
        return m_ClientHolder->GetDisplayContrastRatio(outRatio.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayContrastRatio(nn::vi::DisplayId displayId, float ratio) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayContrastRatio);
        return m_ClientHolder->SetDisplayContrastRatio(displayId, ratio);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayGamma(nn::sf::Out<float> outGamma, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayGamma);
        return m_ClientHolder->GetDisplayGamma(outGamma.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayGamma(nn::vi::DisplayId displayId, float gamma) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayGamma);
        return m_ClientHolder->SetDisplayGamma(displayId, gamma);
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayCmuLuma(nn::sf::Out<float> outValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayCmuLuma);
        return m_ClientHolder->GetDisplayCmuLuma(outValue.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayCmuLuma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayCmuLuma);
        return m_ClientHolder->SetDisplayCmuLuma(displayId, value);
    }

    nn::Result SystemDisplayServiceImpl::CreateStrayLayer(
        nn::sf::Out<nn::vi::LayerId> outLayerId,
        nn::sf::Out<std::int64_t> outNativeWindowDataSize,
        const nn::sf::OutBuffer& outNativeWindowData,
        nn::vi::DisplayId displayId,
        nn::vi::LayerSettingsType settings
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(CreateStrayLayer);
        size_t dataSize = 0;
        NN_RESULT_DO(
            m_ClientHolder->CreateStrayLayer(
                outLayerId.GetPointer(),
                &dataSize,
                outNativeWindowData.GetPointerUnsafe(),
                outNativeWindowData.GetSize(),
                displayId,
                settings
            )
        );
        outNativeWindowDataSize.Set(dataSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::GetDisplayLogicalResolution(nn::sf::Out<std::int32_t> outWidth, nn::sf::Out<std::int32_t> outHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetDisplayLogicalResolution);
        return m_ClientHolder->GetDisplayLogicalResolution(outWidth.GetPointer(), outHeight.GetPointer(), displayId);
    }

    nn::Result SystemDisplayServiceImpl::OpenIndirectLayer(
        nn::sf::Out<std::int64_t> outNativeWindowDataSize,
        const nn::sf::OutBuffer& outNativeWindowData,
        nn::vi::IndirectProducerHandleType producerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(OpenIndirectLayer);
        size_t dataSize = 0;
        NN_RESULT_DO(
            m_ClientHolder->BindIndirectProducerEndPoint(
                &dataSize,
                outNativeWindowData.GetPointerUnsafe(),
                outNativeWindowData.GetSize(),
                producerHandle,
                aruid
            )
        );
        outNativeWindowDataSize.Set(dataSize);
        NN_RESULT_SUCCESS;
    }


    nn::Result SystemDisplayServiceImpl::CloseIndirectLayer(nn::vi::IndirectProducerHandleType producerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(CloseIndirectLayer);
        NN_RESULT_DO(m_ClientHolder->UnbindIndirectProducerEndPoint(producerHandle));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::FlipIndirectLayer(nn::vi::IndirectProducerHandleType producerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPCV_S(FlipIndirectLayer);
        NN_RESULT_DO(m_ClientHolder->FlipIndirectProducerBuffer(producerHandle));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::SetDisplayMagnification(nn::vi::DisplayId displayId, int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(SetDisplayMagnification);
        NN_RESULT_DO(m_ClientHolder->SetDisplayMagnification(displayId, x, y, width, height));
        NN_RESULT_SUCCESS;
    }
    nn::Result SystemDisplayServiceImpl::GetSharedBufferMemoryHandleId(
        nn::sf::Out<nn::vi::native::NativeMemoryHandleId> outMemoryHandleId,
        nn::sf::Out<std::uint64_t> outMemorySize,
        nn::sf::Out<nn::vi::fbshare::SharedMemoryPoolLayout> outLayout,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetSharedBufferMemoryHandle);
        nn::vi::native::NativeMemoryHandleId memId = {};
        size_t size = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_RESULT_DO(m_ClientHolder->GetSharedBufferMemoryHandleId(&memId, &size, &layout, hBuffer, aruid));
        outMemoryHandleId.Set(memId);
        outMemorySize.Set(static_cast<uint64_t>(size));
        outLayout.Set(layout);
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::OpenSharedLayer(nn::vi::fbshare::SharedLayerHandle hLayer, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(OpenSharedLayer);
        NN_RESULT_DO(m_ClientHolder->BindSharedClientLayer(hLayer, aruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::CloseSharedLayer(nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(CloseSharedLayer);
        NN_RESULT_DO(m_ClientHolder->UnbindSharedClientLayer(hLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::ConnectSharedLayer(nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(ConnectSharedLayer);
        NN_RESULT_DO(m_ClientHolder->ConnectSharedClientLayer(hLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::DisconnectSharedLayer(nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(DisconnectSharedLayer);
        NN_RESULT_DO(m_ClientHolder->DisconnectSharedClientLayer(hLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::AcquireSharedFrameBuffer(nn::sf::Out<std::int64_t> outIndex, nn::sf::Out<nn::vi::native::NativeSync> outSync, nn::sf::Out<nn::vi::fbshare::SharedLayerTextureIndexList> outIndexList, nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPCV_S(AcquireSharedFrameBuffer);
        int index = -1;
        nn::vi::native::NativeSync sync = {};
        auto list = nn::vi::fbshare::SharedLayerTextureIndexList::GetInvalidValue();
        NN_RESULT_DO(m_ClientHolder->AcquireSharedClientLayerFrameBuffer(&index, &sync, &list, hLayer));
        outIndex.Set(static_cast<int64_t>(index));
        outSync.Set(sync);
        outIndexList.Set(list);
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::PresentSharedFrameBuffer(nn::vi::fbshare::SharedLayerHandle hLayer, std::int64_t index, const nn::vi::native::NativeSync& sync, const nn::vi::CropRegion& crop, nn::vi::ImageTransformType transform, std::int32_t presentInterval) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPCV_S(PresentSharedFrameBuffer);
        NN_RESULT_DO(m_ClientHolder->PresentSharedClientLayerFrameBuffer(hLayer, static_cast<int>(index), sync, crop, transform, presentInterval));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::CancelSharedFrameBuffer(nn::vi::fbshare::SharedLayerHandle hLayer, std::int64_t index) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPCV_S(CancelSharedFrameBuffer);
        NN_RESULT_DO(m_ClientHolder->CancelSharedClientLayerFrameBuffer(hLayer, static_cast<int>(index)));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::GetSharedFrameBufferAcquirableEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(GetSharedFrameBufferAcquirableEvent);
        nn::sf::NativeHandle handle;
        NN_RESULT_DO(m_ClientHolder->GetSharedClientLayerAcquirableEvent(handle, hLayer));
        outHandle.Set(std::move(handle));
        NN_RESULT_SUCCESS;
    }

    nn::Result SystemDisplayServiceImpl::FillSharedFrameBufferColor(nn::vi::fbshare::SharedLayerHandle hLayer, std::int64_t index, std::int32_t r, std::int32_t g, std::int32_t b, std::int32_t a) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_S(FillSharedFrameBufferColor);
        NN_RESULT_DO(m_ClientHolder->FillSharedClientLayerFrameBufferColor(
            hLayer,
            static_cast<int>(index),
            static_cast<int>(r),
            static_cast<int>(g),
            static_cast<int>(b),
            static_cast<int>(a)
        ));
        NN_RESULT_SUCCESS;
    }

}}}

