﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ApplicationDisplayServiceImpl.h"

#include <nn/nn_Common.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>

#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_Types.h>
#include "../visrv_Log.h"
#include "../visrv_Config.h"
#include "visrv_RelayServiceFactory.h"
#include "visrv_SystemDisplayServiceFactory.h"
#include "visrv_ManagerDisplayServiceFactory.h"
#include "visrv_IndirectDisplayTransactionServiceFactory.h"
#include "visrv_CheckPermission.h"

#include "../master/detail/visrv_Common.h"
#include "../master/detail/visrv_Globals.h"
#include "../master/detail/visrv_DisplayManager.h"
#include "../master/detail/visrv_MemoryMap.h"
#include "../master/detail/visrv_PlatformDisplayInfo.h"
#include "../master/detail/visrv_PlatformDisplayDefines.h"

#define NN_VI_CHECK_ALIVE() \
    NN_RESULT_THROW_UNLESS(m_ClientHolder->IsAlive(), nn::vi::ResultDenied());

#define NN_VI_CHECK_SERVICE_PERMISSION(funcname, serviceLevel)                  \
    if(!CheckServicePermission(m_ClientHolder->GetConstants().GetPermission(), serviceLevel)) \
    {                                                                           \
        NN_VISRV_LOG_ERR(funcname " failed: permission denied(%d,%d).\n",       \
            static_cast<int>(m_ClientHolder->GetConstants().GetPermission()),   \
            static_cast<int>(serviceLevel)                                      \
        );                                                                      \
        NN_RESULT_THROW(nn::vi::ResultDenied());                                \
    }


namespace nn{ namespace visrv{ namespace service{

    ApplicationDisplayServiceImpl::ApplicationDisplayServiceImpl(const client::ClientConstants& clientConstants) NN_NOEXCEPT
        : m_ClientHolder(clientConstants)
    {
        NN_VISRV_LOG_IPC("u:ctor\n");
        NN_VISRV_LOG_LIFETIME("ApplicationDisplayServiceImpl ctor\n");
    }

    ApplicationDisplayServiceImpl::~ApplicationDisplayServiceImpl() NN_NOEXCEPT
    {
        NN_VISRV_LOG_IPC("u:dtor\n");
        NN_VISRV_LOG_LIFETIME("ApplicationDisplayServiceImpl dtor\n");
    }

    nn::Result ApplicationDisplayServiceImpl::GetRelayService(
        nn::sf::Out<nn::sf::SharedPointer<nns::hosbinder::IHOSBinderDriver>> outService
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetRelayService);
        NN_VI_CHECK_ALIVE();
        NN_VI_CHECK_SERVICE_PERMISSION("GetRelayService", ServiceLevel_Relay);

        auto p = RelayServiceFactory::Create(&m_ClientHolder);
        if(p == nullptr)
        {
            NN_VISRV_LOG_ERR("GetRelayService failed: create service object failed.\n");
            NN_RESULT_THROW(nn::vi::ResultOperationFailed());
        }

        outService.Set(p);
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetSystemDisplayService(
        nn::sf::Out<nn::sf::SharedPointer<nn::visrv::sf::ISystemDisplayService>> outService
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetSystemDisplayService);
        NN_VI_CHECK_ALIVE();
        NN_VI_CHECK_SERVICE_PERMISSION("GetSystemDisplayService", ServiceLevel_SystemDisplay);

        auto p = SystemDisplayServiceFactory::Create(&m_ClientHolder);
        if(p == nullptr)
        {
            NN_VISRV_LOG_ERR("GetSystemDisplayService failed: create service object failed.\n");
            NN_RESULT_THROW(nn::vi::ResultOperationFailed());
        }

        outService.Set(p);
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetManagerDisplayService(
        nn::sf::Out<nn::sf::SharedPointer<nn::visrv::sf::IManagerDisplayService>> outService
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetManagerDisplayService);
        NN_VI_CHECK_ALIVE();
        NN_VI_CHECK_SERVICE_PERMISSION("GetManagerDisplayService", ServiceLevel_ManagerDisplay);

        auto p = ManagerDisplayServiceFactory::Create(&m_ClientHolder);
        if(p == nullptr)
        {
            NN_VISRV_LOG_ERR("GetManagerDisplayService failed: create service object failed.\n");
            NN_RESULT_THROW(nn::vi::ResultOperationFailed());
        }

        outService.Set(p);
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetIndirectDisplayTransactionService(
        nn::sf::Out<nn::sf::SharedPointer<nns::hosbinder::IHOSBinderDriver>> outService
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetIndirectDisplayTransactionService);
        NN_VI_CHECK_ALIVE();
        NN_VI_CHECK_SERVICE_PERMISSION("GetIndirectDisplayTransactionService", ServiceLevel_IndirectDisplay);

        auto p = IndirectDisplayTransactionServiceFactory::Create(&m_ClientHolder);
        if(p == nullptr)
        {
            NN_VISRV_LOG_ERR("GetIndirectDisplayTransactionService failed: create service object failed.\n");
            NN_RESULT_THROW(nn::vi::ResultOperationFailed());
        }

        outService.Set(p);
        NN_RESULT_SUCCESS;
    }


    nn::Result ApplicationDisplayServiceImpl::ListDisplays(
        nn::sf::Out<std::int64_t> outCount,
        const nn::sf::OutArray<nn::vi::DisplayInfo>& outInfo
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(ListDisplays);
        return m_ClientHolder->ListDisplays(outCount.GetPointer(), outInfo.GetData(), static_cast<int>(outInfo.GetLength()));
    }

    nn::Result ApplicationDisplayServiceImpl::OpenDisplay(
        nn::sf::Out<nn::vi::DisplayId> outDisplayId,
        const nn::vi::DisplayName& name
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(OpenDisplay);
        return m_ClientHolder->OpenDisplay(outDisplayId.GetPointer(), name.value);
    }

    nn::Result ApplicationDisplayServiceImpl::OpenDefaultDisplay(
        nn::sf::Out<nn::vi::DisplayId> outDisplayId
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(OpenDefaultDisplay);
        return m_ClientHolder->OpenDisplay(outDisplayId.GetPointer(), master::detail::DefaultDisplayName);
    }

    nn::Result ApplicationDisplayServiceImpl::CloseDisplay(
        nn::vi::DisplayId displayId
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(CloseDisplay);
        return m_ClientHolder->CloseDisplay(displayId);
    }

    nn::Result ApplicationDisplayServiceImpl::SetDisplayEnabled(
        nn::vi::DisplayId displayId,
        bool isEnabled
        ) NN_NOEXCEPT
    {
        // Note:  Do not remove this function for IPC compatibility
        NN_VISRV_TRACE_IPC_U(SetDisplayEnabled);
        NN_UNUSED(displayId);
        NN_UNUSED(isEnabled);
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetDisplayResolution(
        nn::sf::Out<std::int64_t> outWidth,
        nn::sf::Out<std::int64_t> outHeight,
        nn::vi::DisplayId displayId
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetDisplayResolution);
        NN_RESULT_THROW_UNLESS(outWidth.GetPointer() != nullptr &&
                               outHeight.GetPointer() != nullptr,
                               nn::vi::ResultOperationFailed()
        );

        // vi:s path now uses GetDisplayMode
        // When this API was active, vi:u only had access to Default display.
        // Always return 720p in this case to match libnn_vi behavior.
        *outWidth = 1280;
        *outHeight = 720;

        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::OpenLayer(
        nn::sf::Out<std::int64_t> outNativeWindowDataSize,
        const nn::sf::OutBuffer& outNativeWindowData,
        nn::vi::LayerId layerId,
        nn::vi::DisplayName displayName,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(OpenLayer);
        size_t dataSize = 0;
        NN_RESULT_DO(
            m_ClientHolder->BindManagedLayer(
                &dataSize,
                outNativeWindowData.GetPointerUnsafe(),
                outNativeWindowData.GetSize(),
                layerId,
                displayName.value,
                aruid
            )
        );
        outNativeWindowDataSize.Set(static_cast<int64_t>(dataSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::CloseLayer(
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(CloseLayer);
        return m_ClientHolder->UnbindManagedLayer(layerId);
    }

    nn::Result ApplicationDisplayServiceImpl::CreateStrayLayer(
        nn::sf::Out<nn::vi::LayerId> outLayerId,
        nn::sf::Out<std::int64_t> outNativeWindowDataSize,
        const nn::sf::OutBuffer& outNativeWindowData,
        nn::vi::DisplayId displayId,
        nn::vi::LayerSettingsType settings
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(CreateStrayLayer);
        nn::vi::LayerSettings defaults;
        nn::vi::SetLayerSettingsDefaults(&defaults);

        size_t dataSize = 0;
        NN_RESULT_DO(
            m_ClientHolder->CreateStrayLayer(
                outLayerId.GetPointer(),
                &dataSize,
                outNativeWindowData.GetPointerUnsafe(),
                outNativeWindowData.GetSize(),
                displayId,
                (settings & ~LayerSettingsProtectedMask) | (defaults._storage[0] & LayerSettingsProtectedMask)
            )
        );
        outNativeWindowDataSize.Set(static_cast<int64_t>(dataSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::DestroyStrayLayer(
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(DestroyStrayLayer);
        return m_ClientHolder->DestroyStrayLayer(layerId);
    }

    nn::Result ApplicationDisplayServiceImpl::SetLayerScalingMode(nn::vi::LayerId layerId, nn::vi::ScalingModeType mode) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(SetLayerScalingMode);
        // This path is deprecated since this must be set on the client's producer end of the buffer queue.
        // Just attempt a conversion to ensure the value is valid in case the clients somehow depended on the
        // results returned.
        std::int64_t outMode;
        NN_RESULT_DO(m_ClientHolder->ConvertScalingMode(&outMode, mode));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::ConvertScalingMode(nn::sf::Out<std::int64_t> outMode, nn::vi::ScalingModeType viMode) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(ConvertScalingMode);
        NN_RESULT_THROW_UNLESS(outMode.GetPointer() != nullptr, nn::vi::ResultOperationFailed());
        return m_ClientHolder->ConvertScalingMode(outMode.GetPointer(), viMode);
    }

    nn::Result ApplicationDisplayServiceImpl::GetIndirectLayerImageMap(
        nn::sf::Out<std::int64_t> outSize,
        nn::sf::Out<std::int64_t> outStride,
        const nn::sf::OutBuffer& outBuffer,
        std::int64_t width,
        std::int64_t height,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetIndirectLayerImageMap);
        size_t size = 0;
        size_t stride = 0;

        NN_RESULT_DO(
            m_ClientHolder->GetIndirectLayerImageMap(
                &size,
                &stride,
                outBuffer.GetPointerUnsafe(),
                outBuffer.GetSize(),
                static_cast<int>(width),
                static_cast<int>(height),
                0, /* sourceX */
                0, /* sourceY */
                1, /* sourceWidth */
                1, /* sourceHeight */
                consumerHandle,
                aruid
            )
        );

        outSize.Set(static_cast<int64_t>(size));
        outStride.Set(static_cast<int64_t>(stride));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetIndirectLayerImageCropMap(
        nn::sf::Out<std::int64_t> outSize,
        nn::sf::Out<std::int64_t> outStride,
        const nn::sf::OutBuffer& outBuffer,
        std::int64_t width,
        std::int64_t height,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetIndirectLayerImageCropMap);
        size_t size = 0;
        size_t stride = 0;

        NN_RESULT_DO(
            m_ClientHolder->GetIndirectLayerImageMap(
                &size,
                &stride,
                outBuffer.GetPointerUnsafe(),
                outBuffer.GetSize(),
                static_cast<int>(width),
                static_cast<int>(height),
                sourceRectX,
                sourceRectY,
                sourceRectWidth,
                sourceRectHeight,
                consumerHandle,
                aruid
            )
        );

        outSize.Set(static_cast<int64_t>(size));
        outStride.Set(static_cast<int64_t>(stride));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetIndirectLayerImageRequiredMemoryInfo(
        nn::sf::Out<std::int64_t> outSize,
        nn::sf::Out<std::int64_t> outAlignment,
        std::int64_t width,
        std::int64_t height
    ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetIndirectLayerImageRequiredMemoryInfo);
        size_t size = 0;
        size_t alignment = 0;

        NN_RESULT_DO(
            m_ClientHolder->GetIndirectLayerImageRequiredMemoryInfo(
                &size,
                &alignment,
                static_cast<int>(width),
                static_cast<int>(height)
            )
        );

        outSize.Set(static_cast<int64_t>(size));
        outAlignment.Set(static_cast<int64_t>(alignment));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::OpenIndirectImageTransfer(
        nn::sf::Out<nn::sf::NativeHandle> outIsImageReadyEventHandle,
        nn::sf::NativeHandle&& bufferHandle,
        std::uint64_t bufferSize,
        std::int64_t width,
        std::int64_t height,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(OpenIndirectImage);

        nn::os::NativeHandle eventHandle = nn::os::InvalidNativeHandle;
        bool isEventHandleManaged = false;
        NN_UTIL_SCOPE_EXIT{
            if(isEventHandleManaged)
            {
                nn::os::CloseNativeHandle(eventHandle);
                isEventHandleManaged = false;
            }
        };

        nn::os::NativeHandle bufferOsHandle = bufferHandle.GetOsHandle();
        bool isBufferHandleManaged = bufferHandle.IsManaged();
        bufferHandle.Detach();
        NN_UTIL_SCOPE_EXIT{
            if(isBufferHandleManaged)
            {
                nn::os::CloseNativeHandle(bufferOsHandle);
                isBufferHandleManaged = false;
            }
        };

        NN_RESULT_DO(m_ClientHolder->BindIndirectConsumerEndPointTransfer(
            &eventHandle,
            &isEventHandleManaged,
            bufferOsHandle,
            &isBufferHandleManaged,
            static_cast<size_t>(bufferSize),
            static_cast<int>(width),
            static_cast<int>(height),
            consumerHandle,
            aruid
        ));

        outIsImageReadyEventHandle.Set(nn::sf::NativeHandle(eventHandle, isEventHandleManaged));
        isEventHandleManaged = false;
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::CloseIndirectImageTransfer(nn::vi::IndirectConsumerHandleType consumerHandle) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(CloseIndirectImage);
        NN_RESULT_DO(m_ClientHolder->UnbindIndirectConsumerEndPoint(consumerHandle));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::UpdateIndirectImageTransfer(
        nn::sf::Out<std::int64_t> outSize,
        nn::sf::Out<std::int64_t> outStride,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight
        ) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(UpdateIndirectImage);
        size_t size = 0;
        size_t stride = 0;
        NN_RESULT_DO(m_ClientHolder->KickIndirectCopyImageTransfer(&size, &stride, consumerHandle, sourceRectX, sourceRectY, sourceRectWidth, sourceRectHeight));
        outSize.Set(static_cast<int64_t>(size));
        outStride.Set(static_cast<int64_t>(stride));
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationDisplayServiceImpl::GetDisplayVsyncEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetDisplayVsyncEvent);
        nn::os::NativeHandle handle;
        bool isManaged;

        nn::Result result = m_ClientHolder->GetDisplayVsyncEvent(&handle, &isManaged, displayId);

        if( result.IsSuccess() )
        {
            *outHandle = nn::sf::NativeHandle(handle, isManaged);
        }

        return result;
    }

    nn::Result ApplicationDisplayServiceImpl::GetDisplayVsyncEventForDebug(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_TRACE_IPC_U(GetDisplayVsyncEventForDebug);
        nn::os::NativeHandle handle;
        bool isManaged;

        nn::Result result = m_ClientHolder->GetDisplayVsyncEventForDebug(&handle, &isManaged, displayId);

        if( result.IsSuccess() )
        {
            *outHandle = nn::sf::NativeHandle(handle, isManaged);
        }

        return result;
    }

    client::ClientObject* ApplicationDisplayServiceImpl::GetClient() NN_NOEXCEPT
    {
        return m_ClientHolder.Get();
    }

}}}
