﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SyncpointWaiterQueue.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>

namespace nn{ namespace visrv{ namespace native{ namespace detail{

    SyncpointWaiterQueue::SyncpointWaiterQueue() NN_NOEXCEPT
        : m_HeadIndex(0)
        , m_TailIndex(0)
        , m_Mutex()
        , m_EnqueueSemaphore()
        , m_DequeueSemaphore()
    {
    }

    void SyncpointWaiterQueue::Initialize() NN_NOEXCEPT
    {
        m_HeadIndex = 0;
        m_TailIndex = 0;
        nn::os::InitializeMutex(&m_Mutex, false, 0);
        nn::os::InitializeSemaphore(&m_EnqueueSemaphore, Capacity, Capacity);
        nn::os::InitializeSemaphore(&m_DequeueSemaphore,        0, Capacity);
    }

    void SyncpointWaiterQueue::Finalize() NN_NOEXCEPT
    {
        nn::os::FinalizeSemaphore(&m_DequeueSemaphore);
        nn::os::FinalizeSemaphore(&m_EnqueueSemaphore);
        nn::os::FinalizeMutex(&m_Mutex);
    }

    void SyncpointWaiterQueue::Enqueue(SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        nn::os::AcquireSemaphore(&m_EnqueueSemaphore);
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_ABORT_UNLESS_RESULT_SUCCESS(EnqueueImpl(pEntry));

        nn::os::ReleaseSemaphore(&m_DequeueSemaphore);
    }

    void SyncpointWaiterQueue::Dequeue(SyncpointEntry** pOutEntry) NN_NOEXCEPT
    {
        nn::os::AcquireSemaphore(&m_DequeueSemaphore);
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_ABORT_UNLESS_RESULT_SUCCESS(DequeueImpl(pOutEntry));

        nn::os::ReleaseSemaphore(&m_EnqueueSemaphore);
    }

    bool SyncpointWaiterQueue::Cancel(SyncpointEntry* pEntry) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        for(int i = 0; i < Size; i++)
        {
            if(m_EntryList[i] == pEntry)
            {
                m_EntryList[i] = nullptr;
                return true;
            }
        }
        return false;
    }

    nn::Result SyncpointWaiterQueue::EnqueueImpl(SyncpointEntry* entry) NN_NOEXCEPT
    {
        int nextTail = (m_TailIndex + 1) % Size;
        NN_RESULT_THROW_UNLESS(nextTail != m_HeadIndex, nn::vi::ResultNotReady());

        m_EntryList[m_TailIndex] = entry;
        m_TailIndex = nextTail;
        NN_RESULT_SUCCESS;
    }

    nn::Result SyncpointWaiterQueue::DequeueImpl(SyncpointEntry** pOutEntry) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_HeadIndex != m_TailIndex, nn::vi::ResultNotReady());

        *pOutEntry = m_EntryList[m_HeadIndex];
        m_EntryList[m_HeadIndex] = nullptr;
        m_HeadIndex = (m_HeadIndex + 1) % Size;
        NN_RESULT_SUCCESS;
    }

    void SyncpointWaiterQueue::InitializeDequeueMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        nn::os::InitializeMultiWaitHolder(pHolder, &m_DequeueSemaphore);
    }

    void SyncpointWaiterQueue::FinalizeDequeueMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        nn::os::FinalizeMultiWaitHolder(pHolder);
    }

}}}}

