﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/edid.h>
#include <nn/vi/vi_Result.h>
#include <hos/nvdisp_devctls.h>
#include "visrv_ExternalDisplay.h"
#include "visrv_Nvdc.h"
#include "../../settings/visrv_InterfaceType.h"

namespace nn{ namespace visrv{ namespace master{ namespace detail{

    nn::os::SystemEvent ExternalDisplay::g_ModeChangedEvent(nn::os::EventClearMode_AutoClear, true);

    ExternalDisplay::ExternalDisplay(const IModeFilter* pFilter) NN_NOEXCEPT
        : AndroidDisplay(pFilter, android::ISurfaceComposer::eDisplayIdHdmi, nn::vi::LayerStack_Default)
    {
    }

    bool ExternalDisplay::IsHotplugEventSupported() const NN_NOEXCEPT
    {
        return true;
    }

    bool ExternalDisplay::IsVsyncEventSupported() const NN_NOEXCEPT
    {
        return true;
    }

    bool ExternalDisplay::IsModeChangedEventSupported() const NN_NOEXCEPT
    {
        return true;
    }

    int ExternalDisplay::ListRgbRanges(nn::vi::RgbRangeType* pOutRanges, int rgbCountMax) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutRanges);
        NN_SDK_ASSERT_GREATER(rgbCountMax, 0);

        // note that EDID read failures will return false and write limited RGB as a supported option
        return WriteCeFormatRgbRange(pOutRanges, rgbCountMax, IsRgbQuantizationSelectable());
    }

    nn::Result ExternalDisplay::SetRgbRange(nn::vi::RgbRange range) NN_NOEXCEPT
    {
        if( range == nn::vi::RgbRange_Auto )
        {
            // Note: This is assuming CE video formats...
            //       Full is default for IT formats.
            if( IsRgbQuantizationSelectable() )
            {
                range = nn::vi::RgbRange_Full;
            }
            else
            {
                // EDID read failures will also fall back to this branch
                range = nn::vi::RgbRange_Limited;
            }
        }

        int convertedRange = ConvertRgbRangeToAndroid(range);

        if( android::SurfaceComposerClient::setRGBRange(m_Display, convertedRange) == 0 )
        {
            return nn::ResultSuccess();
        }

        return nn::vi::ResultOperationFailed();
    }

    bool ExternalDisplay::IsRgbQuantizationSelectable() const NN_NOEXCEPT
    {
        android::DisplayEDID raw;

        if( android::SurfaceComposerClient::getEDID(m_Display, raw) != 0 )
        {
            return false;
        }

        if( raw.getData() == nullptr || raw.getSize() == 0 )
        {
            return false;
        }

        nn::edid::Edid edid;

        if( nn::edid::OpenEdid(&edid, raw.getData(), raw.getSize()) != nn::edid::Error_None )
        {
            return false;
        }

        nn::edid::DisplayInfo info;
        nn::edid::GetDisplayInfo(&info, &edid);

        return info.isRgbQuantizationSelectable;
    }

    // Note:  This is a safeguard for Icosa and other platforms.  In general,
    //        LCD is always enabled first and thus will activate the transitional
    //        logic in ClientObject::SetDefaultDisplay.
    nn::Result ExternalDisplay::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        nn::vi::PowerStateType current;
        NN_RESULT_DO(GetPowerState(&current));

        if( current == nn::vi::PowerState_Off && state == nn::vi::PowerState_On )
        {
            NN_RESULT_DO(AndroidDisplay::SetPowerState(nn::vi::PowerState_Blank));
        }

        NN_RESULT_DO(AndroidDisplay::SetPowerState(state));
        NN_RESULT_SUCCESS;
    }

    nn::Result ExternalDisplay::SetMode(const nn::vi::DisplayModeInfo& mode) NN_NOEXCEPT
    {
        NN_RESULT_DO(AndroidDisplay::SetMode(mode));
        g_ModeChangedEvent.Signal();
        NN_RESULT_SUCCESS;
    }

    nn::os::SystemEvent* ExternalDisplay::GetModeChangedEvent() NN_NOEXCEPT
    {
        return &g_ModeChangedEvent;
    }

}}}}
