﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/vi/sf/vi_ServiceTypes.h>

#include "../visrv_Config.h"

#include <nvrm_channel.h>
#include <gui/BufferQueue.h>

#include "visrv_IndirectLockCounter.h"
#include "visrv_IndirectConsumerBindState.h"
#include "../vic/visrv_VicMemoryPool.h"
#include "../vic/visrv_VicImageBuffer.h"

namespace nn{ namespace visrv{ namespace client{
    class ClientObject;
}}}

namespace nn{ namespace visrv{ namespace indirect{

    enum IndirectLayerState
    {
        // 未初期化の状態。
        // Initialize して使用可能になる。
        IndirectLayerState_Uninitialized,
        // 初期化済の状態。
        IndirectLayerState_Initialized,
        // 初期化済だがエラーが発生した状態。
        // Finalize して Initialize することで再使用可能になる（ハズ）。
        IndirectLayerState_Broken,
    };

    enum IndirectProducerEndPointState
    {
        // 生成側のハンドルが割り当てられていない状態。
        IndirectProducerEndPointState_Invalid,
        // 生成側のハンドルが割り当てられているが、クライアントが使用していない状態。
        IndirectProducerEndPointState_Unbound,
        // 生成側のハンドルが割り当てられており、クライアントが使用している状態。
        IndirectProducerEndPointState_Bound,
        // 生成側のハンドルが割り当てられており、クライアントが接続している状態。
        IndirectProducerEndPointState_Connected,
    };

    enum IndirectConsumerEndPointState
    {
        // 消費側のハンドルが割り当てられていない状態。
        IndirectConsumerEndPointState_Invalid,
        // 消費側のハンドルが割り当てられているが、クライアントが使用していない状態。
        IndirectConsumerEndPointState_Unbound,
        // 消費側のハンドルが割り当てられており、クライアントが使用している状態。
        IndirectConsumerEndPointState_Bound,
        //// 消費側のハンドルが割り当てられており、クライアントが接続している状態。
        //IndirectConsumerEndPointState_Connected,
    };

    class IndirectLayerBuffer
    {
    public:
        IndirectLayerBuffer() NN_NOEXCEPT;

        void Clear() NN_NOEXCEPT;

        int GetIndex() const NN_NOEXCEPT;
        android::sp<android::GraphicBuffer> GetGraphicBuffer() const NN_NOEXCEPT;
        android::sp<android::Fence> GetAcquireFence() const NN_NOEXCEPT;
        android::sp<android::Fence> GetReleaseFence() const NN_NOEXCEPT;
        uint64_t GetFrameNumber() const NN_NOEXCEPT;

        void SetIndex(int index) NN_NOEXCEPT;
        void SetGraphicBuffer(const android::sp<android::GraphicBuffer>& pGraphicBuffer) NN_NOEXCEPT;
        void SetAcquireFence(const android::sp<android::Fence>& pAcquireFence) NN_NOEXCEPT;
        void SetFrameNumber(uint64_t value) NN_NOEXCEPT;
        void UpdateVicReleaseFence(const NvRmFence& fence) NN_NOEXCEPT;
        void ResetVicReleaseFence() NN_NOEXCEPT;

    private:
        int                                 m_Index;
        android::sp<android::GraphicBuffer> m_pGraphicBuffer;
        android::sp<android::Fence>         m_pAcquireFence;
        uint64_t                            m_FrameNumber;
        NvRmFence                           m_VicReleaseFence;
    };

    class IndirectLayerBufferStage
    {
    public:
        IndirectLayerBufferStage() NN_NOEXCEPT;

        void Clear() NN_NOEXCEPT;
        bool HasPendingBuffer() const NN_NOEXCEPT;
        bool HasReadyBuffer() const NN_NOEXCEPT;

        // AcquireFence がシグナル済のバッファを返します。
        const IndirectLayerBuffer* GetReadyBuffer() const NN_NOEXCEPT;
        // AcquireFence 待ちのバッファを返します。
        const IndirectLayerBuffer* GetPendingBuffer() const NN_NOEXCEPT;

        // ReadyBuffer の VIC リリースフェンスを更新します。
        nn::Result UpdateVicReleaseFence(const NvRmFence& fence) NN_NOEXCEPT;
        // 最後に更新された VIC リリースフェンスを取得します。
        NvRmFence GetLastVicReleaseFence() NN_NOEXCEPT;

        // 新しくバッファを挿入します。
        // シグナル状態に関わらず PendingBuffer になります。
        // pBuffer が nullptr の場合、同じ index の以前のバッファの値が維持されます。
        // @pre !IsFull()
        nn::Result PushBuffer(
            int index,
            const android::sp<android::GraphicBuffer>& pBuffer,
            const android::sp<android::Fence>& pAcquireFence,
            uint64_t frameNumber
        ) NN_NOEXCEPT;

        // PendingBuffer の AcquireFence がシグナルされているかをチェックします。
        // シグナルされている場合、以下の操作を行います。
        //   - PendingBuffer の AcquireFence を nullptr に設定します。
        //   - PendingBuffer を ReadyBuffer に移動します。
        //   - 元の ReadyBuffer を *pOutLastBuffer に代入します。最初の 1 枚目の場合、空のバッファが代入されます。
        //   - nn::ResultSuccess を返します。
        // @retval nn::vi::ResultNotReady PendingBuffer の AcqureFence がシグナル状態でない。
        // @retval nn::vi::ResultBroken   意図しないエラーが発生した。
        nn::Result FlipBuffer(IndirectLayerBuffer* pOutLastBuffer) NN_NOEXCEPT;

    private:
        IndirectLayerBuffer m_BufferList[IndirectLayerBufferCountMax];
        int m_ReadyIndex;
        int m_PendingIndex;
        NvRmFence m_LastVicReleaseFence;
    };

    namespace detail{
        struct ProducerClientState
        {
        public:
            void Clear() NN_NOEXCEPT
            {
                this->connectedApi = 0;
                this->flipEntryIndex = -1;
            }
        public:
            int connectedApi;
            int flipEntryIndex;
        };
    }

    class IndirectLayer
    {
    public:
        IndirectLayer() NN_NOEXCEPT;
        ~IndirectLayer() NN_NOEXCEPT;

        // 初期化済かを返します。
        // 壊れている場合も true を返します。
        bool IsInitialized() const NN_NOEXCEPT;
        // 壊れているかを返します。
        bool IsBroken() const NN_NOEXCEPT;

        IndirectProducerEndPointState GetProducerState() const NN_NOEXCEPT;
        IndirectConsumerEndPointState GetConsumerState() const NN_NOEXCEPT;

        client::ClientObject* GetOwnerClient() NN_NOEXCEPT;
        client::ClientObject* GetProducerClient() NN_NOEXCEPT;
        client::ClientObject* GetConsumerClient() NN_NOEXCEPT;

        nn::vi::IndirectLayerHandleType GetHandle() const NN_NOEXCEPT;
        nn::vi::IndirectProducerHandleType GetProducerEndPointHandle() const NN_NOEXCEPT;
        nn::vi::IndirectConsumerHandleType GetConsumerEndPointHandle() const NN_NOEXCEPT;

        android::sp<android::IGraphicBufferProducer> GetGraphicBufferProducer() NN_NOEXCEPT;
        android::sp<android::IGraphicBufferConsumer> GetGraphicBufferConsumer() NN_NOEXCEPT;

        // 旧インタフェース用
        IndirectLayerBufferStage* GetBufferStage() NN_NOEXCEPT{ return &m_BufferStage; }
        IndirectLockCounter*      GetBufferLockCounter() NN_NOEXCEPT{ return &m_ReadyBufferFenceLock; }
        nn::applet::AppletResourceUserId GetConsumerAruid() NN_NOEXCEPT{ return m_ConsumerAruid; }

    // 状態遷移

        // レイヤを初期化します。
        nn::Result Initialize(
            nn::vi::IndirectLayerHandleType* pOutHandle,
            client::ClientObject* pOwnerClient
        ) NN_NOEXCEPT;

        // レイヤを破棄します。
        // @pre ProducerState == Invalid
        // @pre ConsumerState == Invalid
        void Finalize() NN_NOEXCEPT;


        // Producer の ARUID を指定します。
        // ProducerEndPoint のハンドルを返します。
        // @pre ProducerState == Invalid
        // @post ProducerState == Disconnected
        // @retval nn::ResultBroken
        nn::Result RegisterProducerAruid(nn::vi::IndirectProducerHandleType* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // Consumer の ARUID を指定します。
        // ConsumerEndPoint のハンドルを返します。
        // @pre ConsumerState == Invalid
        // @post ConsumerState == Disconnected
        // @retval nn::ResultBroken
        nn::Result RegisterConsumerAruid(nn::vi::IndirectConsumerHandleType* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // Producer の ARUID を解除します。
        // @pre ProducerState == Unbound
        // @post ProducerState == Invalid
        void UnregisterProducerAruid() NN_NOEXCEPT;

        // Consumer の ARUID を解除します。
        // @pre ConsumerState == Unbound
        // @post ConsumerState == Invalid
        void UnregisterConsumerAruid() NN_NOEXCEPT;

        // Producer をバインドします。
        // ClientObject の値は変更しません。
        // @pre ProducerState == Unbound
        // @pre pObject != nullptr
        // @post ProducerState == Bound
        nn::Result BindProducer(
            client::ClientObject* pObject,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;


        // Consumer を MapDefer モードでバインドします。
        // ClientObject の値は変更しません。
        // @pre ConsumerState == Unbound
        // @pre pObject != nullptr
        // @post ConsumerState == Bound
        nn::Result BindConsumerMapDefer(
            client::ClientObject* pClientObject,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

        // Consumer をバインドします。
        // ClientObject の値は変更しません。
        // @pre ConsumerState == Unbound
        // @pre pObject != nullptr
        // @post ConsumerState == Bound
        nn::Result BindConsumerTransfer(
            nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
            bool* pOutIsBufferReadyEventHandleManaged,
            client::ClientObject* pClientObject,
            nn::applet::AppletResourceUserId aruid,
            nn::os::NativeHandle bufferTransferMemoryHandle,
            bool* pIsBufferTransferMemoryHandleManaged,
            size_t bufferTransferMemorySize,
            int width,
            int height
        ) NN_NOEXCEPT;

        // Producer をアンバインドします。
        // ClientObject の値は変更しません。
        // @pre ProducerState == Bound
        // @post ProducerState == Unbound
        void UnbindProducer() NN_NOEXCEPT;

        // Consumer をアンバインドします。
        // Bind したモードに応じて適切にアンバインドします。
        // ClientObject の値は変更しません。
        // @pre ConsumerState == Bound
        // @post ConsumerState == Unbound
        void UnbindConsumer() NN_NOEXCEPT;

    public:
        // Producer を接続します。
        // IGraphicBufferProducer::connect() に相当します。
        // @pre ProducerState == Bound
        // @post ProducerState == Connected
        // @retval nn::vi::ResultBroken
        nn::Result ConnectProducer(android::IGraphicBufferProducer::QueueBufferOutput* pOutValue, int api, bool controlledByApp) NN_NOEXCEPT;

        //// Consumer を接続します。
        //// @pre ConsumerState == Bound
        //// @post ConsumerState == Connected
        //// @retval nn::vi::ResultBroken
        //nn::Result ConnectConsumer() NN_NOEXCEPT;

        // Producer を切断します。
        // ClientObject の値は変更しません。
        // @pre ProducerState == Connected
        // @post ProducerState == Bound
        void DisconnectProducer() NN_NOEXCEPT;

        //// Consumer を切断します。
        //// ClientObject の値は変更しません。
        //// @pre CusumerState == Connected
        //// @post ConsumerState == Bound
        //void DisconnectConsumer() NN_NOEXCEPT;

        // レイヤを壊れた状態としてマークします。
        // 終了処理以外に対して nn::vi::ResultBroken が返されるようになります。
        void SetBroken() NN_NOEXCEPT;

    // 操作

        // Producer のフリップタイミングのオフセット（VSYNC からの時間）を取得します。
        nn::TimeSpan GetProducerFlipOffset() const NN_NOEXCEPT;

        // Producer のフリップタイミングのオフセット（VSYNC からの時間）を設定します。
        // @pre offset > 0
        void SetProducerFlipOffset(nn::TimeSpan offset) NN_NOEXCEPT;

        // Producer のフレームバッファをフリップします
        nn::Result FlipStagedBuffer() NN_NOEXCEPT;

        // ReadyBuffer をコピーします。
        nn::Result KickCopyImageTransfer(size_t* pOutSize, size_t* pOutStride, float sourceRectX, float sourceRectY, float sourceRectWidth, float sourceRectHeight) NN_NOEXCEPT;

        // Consumer のバインドモードを取得します
        IndirectConsumerBoundMode GetConsumerBoundMode() const NN_NOEXCEPT;

        // Consumer のバインド ID を取得します
        IndirectConsumerBoundId GetConsumerBoundId() const NN_NOEXCEPT;

        // Consumer のバインドロックを取得します
        // @pre GetConsumerBoundMode() != IndirectConsumerBoundMode_None
        IndirectLockCounter* GetConsumerBoundLockCounter() NN_NOEXCEPT;


    public:
        nn::util::IntrusiveListNode m_OwnerClientListNode;
        nn::util::IntrusiveListNode m_ProducerClientListNode;
        nn::util::IntrusiveListNode m_ConsumerClientListNode;

    private:
        friend class IndirectCopyVicTask;

        IndirectLayerState            m_State;
        IndirectProducerEndPointState m_ProducerEndPointState;
        IndirectConsumerEndPointState m_ConsumerEndPointState;

        nn::vi::IndirectLayerHandleType    m_LayerHandle;
        nn::vi::IndirectProducerHandleType m_ProducerEndPointHandle;
        nn::vi::IndirectConsumerHandleType m_ConsumerEndPointHandle;

        nn::applet::AppletResourceUserId m_ProducerAruid;
        nn::applet::AppletResourceUserId m_ConsumerAruid;

        client::ClientObject* m_pOwnerClient;
        client::ClientObject* m_pProducerClient;
        client::ClientObject* m_pConsumerClient;


        android::sp<android::IGraphicBufferProducer> m_pProducer;
        android::sp<android::IGraphicBufferConsumer> m_pConsumer;
        IndirectLayerBufferStage m_BufferStage;
        // BufferStage の ReadyBuffer のロック。
        // ロック中はバッファが変化しないようにする。
        IndirectLockCounter m_ReadyBufferFenceLock;

        // Producer が自動で Flip するタイミングの Vsync からのオフセット
        // この値は ProducerEndPoint が作られていれば設定可能（bind, connect不要）
        nn::TimeSpan m_ProducerFlipOffset;

        detail::ProducerClientState m_ProducerClientState;
        IndirectConsumerBindState   m_ConsumerBindState;
    };

    typedef nn::util::IntrusiveList<
        IndirectLayer, nn::util::IntrusiveListMemberNodeTraits<IndirectLayer, &IndirectLayer::m_OwnerClientListNode>
        > IndirectLayerOwnerList;

    typedef nn::util::IntrusiveList<
        IndirectLayer, nn::util::IntrusiveListMemberNodeTraits<IndirectLayer, &IndirectLayer::m_ProducerClientListNode>
        > IndirectLayerProducerList;

    typedef nn::util::IntrusiveList<
        IndirectLayer, nn::util::IntrusiveListMemberNodeTraits<IndirectLayer, &IndirectLayer::m_ConsumerClientListNode>
        > IndirectLayerConsumerList;

}}}
