﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectLayer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>

#include "../visrv_Log.h"
#include "../visrv_ResourceIdManagement.h"

#include <nvsync.h>
#include <../services/nvnflinger/nvnflinger_nullalloc.h>
#include <gui/IProducerListener.h>
#include <binder/Parcel.h>
#include "../native/visrv_BinderFunctions.h"
#include "../native/visrv_SyncpointWaiter.h"
#include "../vic/visrv_VicModule.h"
#include "../vic/visrv_VicTaskWorker.h"
#include "visrv_IndirectFlipManager.h"
#include "../client/visrv_ClientObject.h"

#define NN_VISRV_INDIRECT_CHECK_BROKEN() \
    NN_RESULT_THROW_UNLESS(m_State != IndirectLayerState_Broken, nn::vi::ResultBroken());


namespace nn{ namespace visrv{ namespace indirect{

    IndirectLayerBuffer::IndirectLayerBuffer() NN_NOEXCEPT
        : m_Index(-1)
        , m_FrameNumber(0)
    {
        m_VicReleaseFence.SyncPointID = NVRM_INVALID_SYNCPOINT_ID;
        m_VicReleaseFence.Value = 0;
    }

    void IndirectLayerBuffer::Clear() NN_NOEXCEPT
    {
        m_Index = -1;
        m_pGraphicBuffer.clear();
        m_pAcquireFence.clear();
        m_FrameNumber = 0;
        ResetVicReleaseFence();
    }

    int IndirectLayerBuffer::GetIndex() const NN_NOEXCEPT
    {
        return m_Index;
    }

    android::sp<android::GraphicBuffer> IndirectLayerBuffer::GetGraphicBuffer() const NN_NOEXCEPT
    {
        return m_pGraphicBuffer;
    }

    android::sp<android::Fence> IndirectLayerBuffer::GetAcquireFence() const NN_NOEXCEPT
    {
        return m_pAcquireFence;
    }

    android::sp<android::Fence> IndirectLayerBuffer::GetReleaseFence() const NN_NOEXCEPT
    {
        if(m_VicReleaseFence.SyncPointID == NVRM_INVALID_SYNCPOINT_ID)
        {
            return android::Fence::NO_FENCE;
        }

        int fd = nvsync_from_fence("", &m_VicReleaseFence, 1);
        return android::sp<android::Fence>(new android::Fence(fd));
    }

    uint64_t IndirectLayerBuffer::GetFrameNumber() const NN_NOEXCEPT
    {
        return m_FrameNumber;
    }

    void IndirectLayerBuffer::SetIndex(int index) NN_NOEXCEPT
    {
        m_Index = index;
    }

    void IndirectLayerBuffer::SetGraphicBuffer(const android::sp<android::GraphicBuffer>& pGraphicBuffer) NN_NOEXCEPT
    {
        m_pGraphicBuffer = pGraphicBuffer;
    }

    void IndirectLayerBuffer::SetAcquireFence(const android::sp<android::Fence>& pAcquireFence) NN_NOEXCEPT
    {
        m_pAcquireFence = pAcquireFence;
    }

    void IndirectLayerBuffer::SetFrameNumber(uint64_t value) NN_NOEXCEPT
    {
        m_FrameNumber = value;
    }

    void IndirectLayerBuffer::UpdateVicReleaseFence(const NvRmFence& fence) NN_NOEXCEPT
    {
        if(m_VicReleaseFence.SyncPointID == NVRM_INVALID_SYNCPOINT_ID)
        {
            m_VicReleaseFence = fence;
            return;
        }

        NN_SDK_ASSERT_EQUAL(m_VicReleaseFence.SyncPointID, fence.SyncPointID);
        m_VicReleaseFence.Value = std::max(m_VicReleaseFence.Value, fence.Value);
    }

    void IndirectLayerBuffer::ResetVicReleaseFence() NN_NOEXCEPT
    {
        m_VicReleaseFence.SyncPointID = NVRM_INVALID_SYNCPOINT_ID;
        m_VicReleaseFence.Value = 0;
    }

    //-------------------------------------------------------------------

    IndirectLayerBufferStage::IndirectLayerBufferStage() NN_NOEXCEPT
        : m_ReadyIndex(-1)
        , m_PendingIndex(-1)
    {
        m_LastVicReleaseFence.SyncPointID = NVRM_INVALID_SYNCPOINT_ID;
        m_LastVicReleaseFence.Value = 0;
    }

    void IndirectLayerBufferStage::Clear() NN_NOEXCEPT
    {
        for(int i = 0; i < IndirectLayerBufferCountMax; i++)
        {
            m_BufferList[i].Clear();
        }
        m_ReadyIndex   = -1;
        m_PendingIndex = -1;
        m_LastVicReleaseFence.SyncPointID = NVRM_INVALID_SYNCPOINT_ID;
        m_LastVicReleaseFence.Value = 0;
    }

    bool IndirectLayerBufferStage::HasPendingBuffer() const NN_NOEXCEPT
    {
        return m_PendingIndex >= 0;
    }

    bool IndirectLayerBufferStage::HasReadyBuffer() const NN_NOEXCEPT
    {
        return m_ReadyIndex >= 0;
    }

    const IndirectLayerBuffer* IndirectLayerBufferStage::GetReadyBuffer() const NN_NOEXCEPT
    {
        if(m_ReadyIndex < 0 || m_ReadyIndex >= IndirectLayerBufferCountMax)
        {
            return nullptr;
        }
        return &m_BufferList[m_ReadyIndex];
    }

    const IndirectLayerBuffer* IndirectLayerBufferStage::GetPendingBuffer() const NN_NOEXCEPT
    {
        if(m_PendingIndex < 0 || m_PendingIndex >= IndirectLayerBufferCountMax)
        {
            return nullptr;
        }
        return &m_BufferList[m_PendingIndex];
    }

    nn::Result IndirectLayerBufferStage::UpdateVicReleaseFence(const NvRmFence& fence) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(HasReadyBuffer(), nn::vi::ResultNotReady());

        m_BufferList[m_ReadyIndex].UpdateVicReleaseFence(fence);

        // Stage のフェンスも更新
        if(fence.SyncPointID != NVRM_INVALID_SYNCPOINT_ID)
        {
            if(m_LastVicReleaseFence.SyncPointID != NVRM_INVALID_SYNCPOINT_ID)
            {
                NN_SDK_ASSERT_EQUAL(fence.SyncPointID, m_LastVicReleaseFence.SyncPointID);
            }
            m_LastVicReleaseFence = fence;
        }
        NN_RESULT_SUCCESS;
    }

    NvRmFence IndirectLayerBufferStage::GetLastVicReleaseFence() NN_NOEXCEPT
    {
        return m_LastVicReleaseFence;
    }

    nn::Result IndirectLayerBufferStage::PushBuffer(
        int index,
        const android::sp<android::GraphicBuffer>& pBuffer,
        const android::sp<android::Fence>& pAcquireFence,
        uint64_t frameNumber
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!HasPendingBuffer());
        NN_RESULT_THROW_UNLESS(index >= 0 && index < IndirectLayerBufferCountMax, nn::vi::ResultBroken());

        m_BufferList[index].SetIndex(index);
        if(pBuffer != nullptr)
        {
            m_BufferList[index].SetGraphicBuffer(pBuffer);
        }
        m_BufferList[index].SetAcquireFence(pAcquireFence);
        m_BufferList[index].SetFrameNumber(frameNumber);
        m_BufferList[index].ResetVicReleaseFence();

        m_PendingIndex = index;
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayerBufferStage::FlipBuffer(IndirectLayerBuffer* pOutLastBuffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            m_PendingIndex >= 0 && m_PendingIndex < IndirectLayerBufferCountMax,
            nn::vi::ResultNotReady()
        );

        {
            auto& pending = m_BufferList[m_PendingIndex];
            auto pFence = pending.GetAcquireFence();
            if(pFence != nullptr)
            {
                auto waitResult = pFence->wait(0);
                if(waitResult == android::NO_ERROR)
                {
                    // ok
                    NN_VISRV_LOG_INDIRECT_FLIP("pending buffer aquire fence complete\n");
                    pending.SetAcquireFence(nullptr);
                }
                else if(waitResult == -EAGAIN) // -EAGAIN = -11
                {
                    NN_VISRV_LOG_INDIRECT_FLIP("pending buffer aquire fence timeout\n");
                    NN_RESULT_THROW(nn::vi::ResultNotReady());
                }
                else
                {
                    NN_VISRV_LOG_INDIRECT_ERR("flip: pending buffer fence unknown error(%d)\n", waitResult);
                    NN_RESULT_THROW(nn::vi::ResultBroken());
                }
            }
            else
            {
                // ok
                NN_VISRV_LOG_INDIRECT_FLIP("pending buffer doesn't have aquire fence\n");
            }
        }

        if(m_ReadyIndex >= 0 && m_ReadyIndex < IndirectLayerBufferCountMax)
        {
            *pOutLastBuffer = m_BufferList[m_ReadyIndex];
        }
        else
        {
            *pOutLastBuffer = IndirectLayerBuffer();
        }

        m_ReadyIndex = m_PendingIndex;
        m_PendingIndex = -1;
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------

    IndirectLayer::IndirectLayer() NN_NOEXCEPT
        : m_State(IndirectLayerState_Uninitialized)
        , m_ProducerEndPointState(IndirectProducerEndPointState_Invalid)
        , m_ConsumerEndPointState(IndirectConsumerEndPointState_Invalid)
        , m_LayerHandle(0)
        , m_ProducerEndPointHandle(0)
        , m_ConsumerEndPointHandle(0)
        , m_ProducerAruid(nn::applet::AppletResourceUserId::GetInvalidId())
        , m_ConsumerAruid(nn::applet::AppletResourceUserId::GetInvalidId())
        , m_pProducerClient(nullptr)
        , m_pConsumerClient(nullptr)
        , m_ProducerFlipOffset(0)
        , m_ConsumerBindState()
    {
        m_ProducerClientState.Clear();
    }

    IndirectLayer::~IndirectLayer() NN_NOEXCEPT
    {
    }

    IndirectProducerEndPointState IndirectLayer::GetProducerState() const NN_NOEXCEPT
    {
        return m_ProducerEndPointState;
    }

    IndirectConsumerEndPointState IndirectLayer::GetConsumerState() const NN_NOEXCEPT
    {
        return m_ConsumerEndPointState;
    }

    client::ClientObject* IndirectLayer::GetOwnerClient() NN_NOEXCEPT
    {
        return m_pOwnerClient;
    }

    client::ClientObject* IndirectLayer::GetProducerClient() NN_NOEXCEPT
    {
        return m_pProducerClient;
    }

    client::ClientObject* IndirectLayer::GetConsumerClient() NN_NOEXCEPT
    {
        return m_pConsumerClient;
    }

    nn::vi::IndirectLayerHandleType IndirectLayer::GetHandle() const NN_NOEXCEPT
    {
        return m_LayerHandle;
    }

    nn::vi::IndirectProducerHandleType IndirectLayer::GetProducerEndPointHandle() const NN_NOEXCEPT
    {
        return m_ProducerEndPointHandle;
    }

    nn::vi::IndirectConsumerHandleType IndirectLayer::GetConsumerEndPointHandle() const NN_NOEXCEPT
    {
        return m_ConsumerEndPointHandle;
    }

    android::sp<android::IGraphicBufferProducer> IndirectLayer::GetGraphicBufferProducer() NN_NOEXCEPT
    {
        return m_pProducer;
    }

    android::sp<android::IGraphicBufferConsumer> IndirectLayer::GetGraphicBufferConsumer() NN_NOEXCEPT
    {
        return m_pConsumer;
    }

    nn::Result IndirectLayer::Initialize(
        nn::vi::IndirectLayerHandleType* pOutHandle,
        client::ClientObject* pOwnerClient
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, IndirectLayerState_Uninitialized);
        NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
        NN_SDK_REQUIRES_NOT_NULL(pOwnerClient);

        bool isSuccess = false;

        android::sp<android::IGraphicBufferAlloc> pBufferAllocator(new NULLGraphicBufferAlloc());
        NN_RESULT_THROW_UNLESS(pBufferAllocator != nullptr, nn::vi::ResultOperationFailed());
        android::BufferQueue::createBufferQueue(&m_pProducer, &m_pConsumer, pBufferAllocator);
        NN_RESULT_THROW_UNLESS(m_pProducer != nullptr && m_pConsumer != nullptr, nn::vi::ResultOperationFailed());
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_pProducer.clear();
                m_pConsumer.clear();
            }
        };

        // connect to consumer
        {
            android::sp<android::BufferQueue::ProxyConsumerListener> pConsumerListener(new android::BufferQueue::ProxyConsumerListener(nullptr));
            NN_RESULT_THROW_UNLESS(pConsumerListener != nullptr, nn::vi::ResultOperationFailed());
            NN_RESULT_THROW_UNLESS(m_pConsumer->consumerConnect(pConsumerListener, true) == android::NO_ERROR, nn::vi::ResultOperationFailed());
            m_pConsumer->setConsumerUsageBits(android::GraphicBuffer::USAGE_HW_COMPOSER);
        }

        // intialize ready buffer lock
        m_ReadyBufferFenceLock.Initialize(nn::os::GetThreadId(nn::os::GetCurrentThread()));

        m_ProducerEndPointState = IndirectProducerEndPointState_Invalid;
        m_ConsumerEndPointState = IndirectConsumerEndPointState_Invalid;
        m_ProducerEndPointHandle = 0;
        m_ConsumerEndPointHandle = 0;
        m_ProducerAruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_ConsumerAruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_pProducerClient = nullptr;
        m_pConsumerClient = nullptr;
        m_BufferStage.Clear();
        m_ProducerClientState.Clear();
        m_ProducerFlipOffset = 0;

        m_State        = IndirectLayerState_Initialized;
        m_pOwnerClient = pOwnerClient;
        m_LayerHandle  = AcquireResourceId();

        *pOutHandle = m_LayerHandle;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void IndirectLayer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, IndirectLayerState_Uninitialized);
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Invalid);
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Invalid);

        m_ReadyBufferFenceLock.Finalize();

        m_pConsumer.clear();
        m_pProducer.clear();

        ReleaseResourceId(m_LayerHandle);
        m_LayerHandle  = 0;
        m_pOwnerClient = nullptr;
        m_State = IndirectLayerState_Uninitialized;
    }

    bool IndirectLayer::IsInitialized() const NN_NOEXCEPT
    {
        return m_State == IndirectLayerState_Initialized || m_State == IndirectLayerState_Broken;
    }

    bool IndirectLayer::IsBroken() const NN_NOEXCEPT
    {
        return m_State == IndirectLayerState_Broken;
    }

    void IndirectLayer::SetBroken() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        if(NN_STATIC_CONDITION(BreakOnIndirectLayerBroken))
        {
            NN_VISRV_LOG_INDIRECT_ERR("indirect layer is broken!\n");
            for(;;)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                NN_VISRV_LOG_INDIRECT_ERR("indirect layer is broken!\n");
            }
        }
        m_State = IndirectLayerState_Broken;
    }

    nn::Result IndirectLayer::RegisterProducerAruid(nn::vi::IndirectProducerHandleType* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Invalid);

        NN_SDK_ASSERT_EQUAL(m_ProducerAruid, nn::applet::AppletResourceUserId::GetInvalidId());
        NN_SDK_ASSERT_EQUAL(m_ProducerEndPointHandle, 0);

        NN_VISRV_INDIRECT_CHECK_BROKEN();

        m_ProducerEndPointState = IndirectProducerEndPointState_Unbound;
        m_ProducerAruid = aruid;
        m_ProducerEndPointHandle = AcquireResourceId();

        m_ProducerFlipOffset = 0;

        *pOutHandle = m_ProducerEndPointHandle;
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayer::RegisterConsumerAruid(nn::vi::IndirectConsumerHandleType* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Invalid);

        NN_SDK_ASSERT_EQUAL(m_ConsumerAruid, nn::applet::AppletResourceUserId::GetInvalidId());
        NN_SDK_ASSERT_EQUAL(m_ConsumerEndPointHandle, 0);

        NN_VISRV_INDIRECT_CHECK_BROKEN();

        m_ConsumerEndPointState = IndirectConsumerEndPointState_Unbound;
        m_ConsumerAruid = aruid;
        m_ConsumerEndPointHandle = AcquireResourceId();

        *pOutHandle = m_ConsumerEndPointHandle;
        NN_RESULT_SUCCESS;
    }

    void IndirectLayer::UnregisterProducerAruid() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Unbound);

        NN_SDK_ASSERT_NOT_EQUAL(m_ProducerEndPointHandle, 0);

        ReleaseResourceId(m_ProducerEndPointHandle);
        m_ProducerEndPointState = IndirectProducerEndPointState_Invalid;
        m_ProducerAruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_ProducerEndPointHandle = 0;
    }

    void IndirectLayer::UnregisterConsumerAruid() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Unbound);

        NN_SDK_ASSERT_NOT_EQUAL(m_ConsumerEndPointHandle, 0);

        ReleaseResourceId(m_ConsumerEndPointHandle);
        m_ConsumerEndPointState = IndirectConsumerEndPointState_Invalid;
        m_ConsumerAruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_ConsumerEndPointHandle = 0;
    }

    nn::Result IndirectLayer::BindProducer(
        client::ClientObject* pObject,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Unbound);
        NN_SDK_REQUIRES_NOT_NULL(pObject);

        NN_VISRV_INDIRECT_CHECK_BROKEN();
        NN_RESULT_THROW_UNLESS(
            pObject->GetConstants().GetPermission() == client::ClientPermission_Manager ||
            aruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultDenied()
        );
        NN_RESULT_THROW_UNLESS(aruid == m_ProducerAruid, nn::vi::ResultDenied());

        m_ProducerEndPointState = IndirectProducerEndPointState_Bound;
        m_pProducerClient = pObject;

        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayer::BindConsumerMapDefer(
        client::ClientObject* pObject,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Unbound);
        NN_SDK_REQUIRES_EQUAL(m_ConsumerBindState.GetBoundMode(), IndirectConsumerBoundMode_None);
        NN_SDK_REQUIRES_NOT_NULL(pObject);

        NN_VISRV_INDIRECT_CHECK_BROKEN();
        NN_RESULT_THROW_UNLESS(aruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(aruid == m_ConsumerAruid, nn::vi::ResultDenied());
        bool isSuccess = false;

        m_ConsumerBindState.InitializeMapDefer();

        m_ConsumerEndPointState = IndirectConsumerEndPointState_Bound;
        m_pConsumerClient = pObject;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayer::BindConsumerTransfer(
        nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
        bool* pOutIsBufferReadyEventHandleManaged,
        client::ClientObject* pObject,
        nn::applet::AppletResourceUserId aruid,
        nn::os::NativeHandle bufferTransferMemoryHandle,
        bool* pIsBufferTransferMemoryHandleManaged,
        size_t bufferTransferMemorySize,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Unbound);
        NN_SDK_REQUIRES_NOT_NULL(pObject);

        NN_VISRV_INDIRECT_CHECK_BROKEN();
        NN_RESULT_THROW_UNLESS(aruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(aruid == m_ConsumerAruid, nn::vi::ResultDenied());
        bool isSuccess = false;

        NN_RESULT_DO(m_ConsumerBindState.InitializeTransfer(
            pOutIsBufferReadyEventHandle,
            pOutIsBufferReadyEventHandleManaged,
            bufferTransferMemoryHandle,
            pIsBufferTransferMemoryHandleManaged,
            bufferTransferMemorySize,
            width,
            height
        ));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_ConsumerBindState.Finalize();
            }
        };

        m_ConsumerEndPointState = IndirectConsumerEndPointState_Bound;
        m_pConsumerClient = pObject;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void IndirectLayer::UnbindProducer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Bound);

        m_ProducerEndPointState = IndirectProducerEndPointState_Unbound;
        m_pProducerClient = nullptr;
    }

    void IndirectLayer::UnbindConsumer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Bound);
        NN_SDK_ASSERT_NOT_EQUAL(m_ConsumerBindState.GetBoundMode(), IndirectConsumerBoundMode_None);

        m_ConsumerBindState.GetLockCounter()->WaitUnlocked();
        m_ConsumerBindState.Finalize();

        m_ConsumerEndPointState = IndirectConsumerEndPointState_Unbound;
        m_pConsumerClient = nullptr;
    }

    nn::Result IndirectLayer::ConnectProducer(
        android::IGraphicBufferProducer::QueueBufferOutput* pOutValue,
        int api,
        bool controlledByApp
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Bound);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_VISRV_INDIRECT_CHECK_BROKEN();

        // BufferQueue に Connect
        {
            android::Parcel request;
            android::Parcel reply;

            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            request.writeInt32(0);
            request.writeInt32(api);
            request.writeInt32(controlledByApp ? 1 : 0);
            if(request.errorCheck() != android::NO_ERROR)
            {
                SetBroken();
                NN_RESULT_THROW(nn::vi::ResultBroken());
            }

            auto err = m_pProducer->asBinder()->transact(native::IGraphicBufferProducerFunctionCode_Connect, request, &reply, 0);
            if(err != android::NO_ERROR)
            {
                NN_VISRV_LOG_INDIRECT_ERR("producer connect failed %d\n", err);
                SetBroken();
                NN_RESULT_THROW(nn::vi::ResultBroken());
            }

            m_ProducerClientState.connectedApi = api;
        }

        // FlipManager に登録
        {
            int index = g_IndirectFlipManager.Register(this);
            NN_SDK_ASSERT_RANGE(index, 0, IndirectLayerCountMax);
            m_ProducerClientState.flipEntryIndex = index;
        }

        m_ProducerEndPointState = IndirectProducerEndPointState_Connected;
        NN_RESULT_SUCCESS;
    }

    //nn::Result IndirectLayer::ConnectConsumer() NN_NOEXCEPT
    //{
    //    NN_SDK_REQUIRES(IsInitialized());
    //    NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Bound);

    //    NN_VISRV_INDIRECT_CHECK_BROKEN();

    //    m_ConsumerEndPointState = IndirectConsumerEndPointState_Connected;
    //    NN_RESULT_SUCCESS;
    //}

    void IndirectLayer::DisconnectProducer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Connected);

        // BufferStage の ReleaseFence が確定するのを待つ
        m_ReadyBufferFenceLock.WaitUnlocked();

        // VIC が終わるのを待つ
        auto fence = m_BufferStage.GetLastVicReleaseFence();
        if(fence.SyncPointID != NVRM_INVALID_SYNCPOINT_ID)
        {
            native::SyncpointEntry e(fence);
            native::g_SyncpointWaiter.Enqueue(&e);
            e.Wait();
        }

        // FlipManager から登録解除
        g_IndirectFlipManager.Unregister(m_ProducerClientState.flipEntryIndex);

        // Producer のバッファを解放
        m_pProducer->disconnect(m_ProducerClientState.connectedApi);
        // pConsumer に Acquire しているバッファを戻す
        if(auto p = m_BufferStage.GetReadyBuffer())
        {
            auto pGraphicBuffer = p->GetGraphicBuffer();
            if(pGraphicBuffer != nullptr)
            {
                m_pConsumer->releaseBuffer(p->GetIndex(), p->GetFrameNumber(), 0, 0, p->GetReleaseFence());
            }
        }
        if(auto p = m_BufferStage.GetPendingBuffer())
        {
            auto pGraphicBuffer = p->GetGraphicBuffer();
            if(pGraphicBuffer != nullptr)
            {
                m_pConsumer->releaseBuffer(p->GetIndex(), p->GetFrameNumber(), 0, 0, p->GetReleaseFence());
            }
        }

        // キャッシュしている内容を解放
        m_BufferStage.Clear();

        m_ProducerClientState.flipEntryIndex = -1;
        m_ProducerClientState.connectedApi = 0;
        m_ProducerEndPointState = IndirectProducerEndPointState_Bound;
    }

    //void IndirectLayer::DisconnectConsumer() NN_NOEXCEPT
    //{
    //    NN_SDK_REQUIRES(IsInitialized());
    //    NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Connected);

    //    m_ConsumerEndPointState = IndirectConsumerEndPointState_Bound;
    //}

    nn::Result IndirectLayer::FlipStagedBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Connected);
        NN_VISRV_INDIRECT_CHECK_BROKEN();

        // 空いていたら次のバッファを取ってくる
        if(!m_BufferStage.HasPendingBuffer())
        {
            android::IGraphicBufferConsumer::BufferItem item;
            auto err = m_pConsumer->acquireBuffer(&item, 0);
            if(err == android::NO_ERROR)
            {
                NN_VISRV_LOG_INDIRECT_FLIP("staging pending buffer(%d)\n", item.mBuf);
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_BufferStage.PushBuffer(item.mBuf, item.mGraphicBuffer, item.mFence, item.mFrameNumber));
            }
            else if(err == android::IGraphicBufferConsumer::NO_BUFFER_AVAILABLE)
            {
                // ok
                NN_VISRV_LOG_INDIRECT_FLIP("no buffer is acquired from queue\n");
            }
            else
            {
                NN_VISRV_LOG_INDIRECT_ERR("flip: acquire buffer error(%d)\n", err);
                SetBroken();
                NN_RESULT_THROW(nn::vi::ResultBroken());
            }
        }

        // Acquire 済のバッファが準備できているか確認
        if(m_BufferStage.HasPendingBuffer())
        {
            // TORIAEZU:
            // ReadyBuffer の ReleaseFence が確定するのを待つ
            // 本当は PendingBuffer の AcquireFence がシグナルされてからの方が効率がいい
            m_ReadyBufferFenceLock.WaitUnlocked();

            indirect::IndirectLayerBuffer releasingBuffer;
            NN_RESULT_TRY(m_BufferStage.FlipBuffer(&releasingBuffer))
                NN_RESULT_CATCH(nn::vi::ResultBroken)
                {
                    SetBroken();
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;

            // バッファがリリースされたらキューに戻す
            if(releasingBuffer.GetGraphicBuffer() != nullptr)
            {
                NN_VISRV_LOG_INDIRECT_FLIP("releasing buffer to queue(%d)\n", releasingBuffer.GetIndex());
                m_pConsumer->releaseBuffer(releasingBuffer.GetIndex(), releasingBuffer.GetFrameNumber(), 0, 0, releasingBuffer.GetReleaseFence());
                releasingBuffer.Clear();
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectLayer::KickCopyImageTransfer(size_t* pOutSize, size_t* pOutStride, float sourceRectX, float sourceRectY, float sourceRectWidth, float sourceRectHeight) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutStride);
        NN_SDK_REQUIRES_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Connected);
        NN_SDK_REQUIRES_EQUAL(m_ConsumerEndPointState, IndirectConsumerEndPointState_Bound);
        NN_VISRV_INDIRECT_CHECK_BROKEN();

        // 前回のコピーが完了していなければ失敗
        NN_RESULT_THROW_UNLESS(
            !m_ConsumerBindState.GetLockCounter()->IsLocked(),
            nn::vi::ResultAlreadyOpened()
        );

        auto pExState = m_ConsumerBindState.GetExStateTransfer();
        NN_RESULT_THROW_UNLESS(pExState != nullptr, nn::vi::ResultDenied());


        // VIC の ReleaseFence が確定するまでコピー元をロック
        NN_RESULT_THROW_UNLESS(m_BufferStage.HasReadyBuffer(), nn::vi::ResultNotReady());
        auto pReadyBuffer = m_BufferStage.GetReadyBuffer();
        m_ReadyBufferFenceLock.AcquireLock();

        // コピーが完了するまでコピー先をロック
        auto pDestLock = m_ConsumerBindState.GetLockCounter();
        pDestLock->AcquireLock();

        // VIC のパラメータを設定
        IndirectConsumerBindExStateTransfer::Parameter param = {};
        param.sourceRectX = sourceRectX;
        param.sourceRectY = sourceRectY;
        param.sourceRectWidth  = sourceRectWidth;
        param.sourceRectHeight = sourceRectHeight;
        param.pSourceBufferStage       = &m_BufferStage;
        param.pSourceGraphicBuffer     = pReadyBuffer->GetGraphicBuffer();
        param.pSourceBufferLockCounter = &m_ReadyBufferFenceLock;
        param.pDestinationBoundLockCounter = pDestLock;
        pExState->SetParameter(param);

        *pOutSize   = pExState->GetImageSize();
        *pOutStride = pExState->GetImageStride();

        pExState->ClearIsBufferReadySystemEvent();
        vic::g_VicTaskWorker.SubmitTask(pExState);
        NN_RESULT_SUCCESS;
    }

    nn::TimeSpan IndirectLayer::GetProducerFlipOffset() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Invalid);
        return m_ProducerFlipOffset;
    }

    void IndirectLayer::SetProducerFlipOffset(nn::TimeSpan offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_EQUAL(m_ProducerEndPointState, IndirectProducerEndPointState_Invalid);
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, nn::TimeSpan());
        m_ProducerFlipOffset = offset;
    }

    IndirectConsumerBoundMode IndirectLayer::GetConsumerBoundMode() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_ConsumerBindState.GetBoundMode();
    }

    IndirectConsumerBoundId IndirectLayer::GetConsumerBoundId() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_ConsumerBindState.GetBoundId();
    }

    IndirectLockCounter* IndirectLayer::GetConsumerBoundLockCounter() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_ConsumerBindState.GetLockCounter();
    }

}}}
