﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectFlipManager.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../visrv_Config.h"
#include "../visrv_ServerManager.h"
#include "visrv_IndirectLayer.h"

namespace nn{ namespace visrv{ namespace indirect{

    IndirectFlipManager g_IndirectFlipManager;

    //-------------------------------------------------------------------

    IndirectFlipEntry::IndirectFlipEntry() NN_NOEXCEPT
        : m_pLayer(nullptr)
        , m_TimerEvent()
        , m_TimerHolder()
        , m_IsStarted(false)
    {
    }

    void IndirectFlipEntry::Initialize(int index, IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, IndirectLayerCountMax);
        NN_SDK_REQUIRES_NOT_NULL(pLayer);

        m_pLayer = pLayer;
        m_IsStarted = false;

        nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_TimerHolder, &m_TimerEvent);
        m_TimerHolder.userData = MultiWaitIndex_IndirectFlipMin + index;
        //NN_SDK_ASSERT_RANGE(m_TimerHolder.userData, MultiWaitIndex_IndirectFlipMin, MultiWaitIndex_IndirectFlipMax);
    }

    void IndirectFlipEntry::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsStarted);
        nn::os::FinalizeMultiWaitHolder(&m_TimerHolder);
        nn::os::FinalizeTimerEvent(&m_TimerEvent);
        m_pLayer = nullptr;
    }

    bool IndirectFlipEntry::IsInitialized() const NN_NOEXCEPT
    {
        return m_pLayer != nullptr;
    }

    bool IndirectFlipEntry::IsStarted() const NN_NOEXCEPT
    {
        return m_IsStarted;
    }

    nn::os::MultiWaitHolderType* IndirectFlipEntry::Start() NN_NOEXCEPT
    {
        if(m_IsStarted)
        {
            return nullptr;
        }

        auto time = m_pLayer->GetProducerFlipOffset();

        NN_SDK_ASSERT(!nn::os::TryWaitTimerEvent(&m_TimerEvent));
        nn::os::StartOneShotTimerEvent(&m_TimerEvent, time);
        m_IsStarted = true;
        return &m_TimerHolder;
    }

    void IndirectFlipEntry::Stop() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsStarted);

        nn::os::UnlinkMultiWaitHolder(&m_TimerHolder);
        nn::os::StopTimerEvent(&m_TimerEvent);
        nn::os::ClearTimerEvent(&m_TimerEvent);
        m_IsStarted = false;
    }

    void IndirectFlipEntry::Flip() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsStarted);

        m_pLayer->FlipStagedBuffer();

        nn::os::ClearTimerEvent(&m_TimerEvent);
        m_IsStarted = false;
    }

    //-------------------------------------------------------------------

    IndirectFlipManager::IndirectFlipManager() NN_NOEXCEPT
        : m_EntryList()
    {
    }

    void IndirectFlipManager::Initialize() NN_NOEXCEPT
    {
    }

    void IndirectFlipManager::Finalize() NN_NOEXCEPT
    {
    }

    int IndirectFlipManager::Register(IndirectLayer* pLayer) NN_NOEXCEPT
    {
        for(int i = 0; i < IndirectLayerCountMax; i++)
        {
            if(m_EntryList[i].IsInitialized())
            {
                continue;
            }
            m_EntryList[i].Initialize(i, pLayer);
            return i;
        }
        NN_ABORT("IndirectFlipManager::Register empty slot not found");
    }

    void IndirectFlipManager::Unregister(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, IndirectLayerCountMax);
        NN_SDK_REQUIRES(m_EntryList[index].IsInitialized());

        auto& e = m_EntryList[index];
        if(e.IsStarted())
        {
            e.Stop();
        }
        e.Finalize();
    }

    void IndirectFlipManager::ProcessVsync() NN_NOEXCEPT
    {
        for(int i = 0; i < IndirectLayerCountMax; i++)
        {
            if(!m_EntryList[i].IsInitialized())
            {
                continue;
            }
            auto pHolder = m_EntryList[i].Start();
            if(pHolder)
            {
                g_ServerManager.GetDisplayServerManager()->AddUserWaitHolder(pHolder);
            }
        }
    }

    void IndirectFlipManager::ProcessSignal(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        int index = static_cast<int>(pHolder->userData - MultiWaitIndex_IndirectFlipMin);
        NN_SDK_ASSERT_RANGE(index, 0, IndirectLayerCountMax);

        auto& e = m_EntryList[index];
        NN_SDK_ASSERT(e.IsInitialized());
        NN_SDK_ASSERT(e.IsStarted());
        e.Flip();
    }



}}}
